/*******************************************************************************
* manager.cpp: Vls manager
*-------------------------------------------------------------------------------
* (c)1999-2001 VideoLAN
* $Id: manager.cpp,v 1.20 2002/09/30 15:17:53 jpsaman Exp $
*
* Authors: Benoit Steiner <benny@via.ecp.fr>
*          Arnaud de Bossoreille de Ribou <bozo@via.ecp.fr>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*
*-------------------------------------------------------------------------------
*
*******************************************************************************/


//------------------------------------------------------------------------------
// Preamble
//------------------------------------------------------------------------------
#include "../core/defs.h"

#include "config.h"

#include "../core/core.h"

#include "../mpeg/mpeg.h"
#include "../mpeg/ts.h"
#include "../mpeg/rtp.h"

#include "program.h"
#include "request.h"
#include "admin.h"
#include "buffer.h"
#include "output.h"
#include "channel.h"
#include "broadcast.h"
#include "input.h"
#include "repository.h"
#include "directory.h"
#include "tsstreamer.h"

#include "manager.h"
#ifndef  _WIN32
	#include "daemon.h"
#endif
#include "vls.h"

#include "repository.cpp"


/*******************************************************************************
* C_Manager
********************************************************************************
*
*******************************************************************************/

//------------------------------------------------------------------------------
// Constructor
//------------------------------------------------------------------------------
C_Manager::C_Manager(handle hLog, C_EventHub* pHub) : m_cEventQueued(0)
{
  ASSERT(hLog);

  m_hLog = hLog;
  m_pEventHub = pHub;

  m_bStop = false;
}


//------------------------------------------------------------------------------
// Destructor
//------------------------------------------------------------------------------
C_Manager::~C_Manager()
{
}


//------------------------------------------------------------------------------
// Initialization
//------------------------------------------------------------------------------
int C_Manager::Init()
{
  int iRc = NO_ERR;

  try
  {
    InitChannels();
    InitInputs();
    InitPgrmTable();
    Create();
  }
  catch(E_Exception e)
  {
    Log(m_hLog, LOG_ERROR, "Unable to init manager: " + e.Dump());
    iRc = GEN_ERR;
  }

  return iRc;
}


//------------------------------------------------------------------------------
// Execution
//------------------------------------------------------------------------------
int C_Manager::Run()
{
  int iRc = NO_ERR;

  // Nothing to do yet

  return iRc;
}


//------------------------------------------------------------------------------
// Stop the execution
//------------------------------------------------------------------------------
int C_Manager::Stop()
{
  Log(m_hLog, LOG_NOTE, "Stopping the manager");

  int iRc = NO_ERR;

  try
  {
    StopPrograms();
    C_Thread::Stop();
    Log(m_hLog, LOG_NOTE, "Manager stopped");
  }
  catch(E_Exception e)
  {
    Log(m_hLog, LOG_ERROR, "Unable to stop the manager:\n" + e.Dump());
    iRc = GEN_ERR;
  } 

  return iRc;
}


//------------------------------------------------------------------------------
// Destruction
//------------------------------------------------------------------------------
int C_Manager::Destroy()
{
  int iRc = NO_ERR;

  Log(m_hLog, LOG_NOTE, "Destroying the manager");

  try
  {
    DestroyPgrmTable();
    DestroyInputs();
    DestroyChannels();
  }
  catch(E_Exception e)
  {
    Log(m_hLog, LOG_ERROR, "Unable to destroy manager: " + e.Dump());
    iRc = e.GetCode();
  }

  if(!iRc)
    Log(m_hLog, LOG_NOTE, "Manager destroyed");

  return iRc;
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_Manager::InitPgrmTable()
{
  // Lock the input repository and the program list
  m_cInputList.Lock();
  m_cProgramList.Lock();

  // Get the input lists of programs and merge them with the manager one
  C_RepositoryBrowser<C_String, C_Input> cIterator =
                                                  m_cInputList.CreateBrowser();
  
  while(cIterator.HasNextItem())
  {
    C_Input* pInput = cIterator.GetNextItem();
    C_List<C_Program> cInputPgrms = pInput->GetAvailablePgrms();
    unsigned int iPgrmNumber = cInputPgrms.Size();
    for(unsigned int j = 0; j < iPgrmNumber; j++)
    {
      m_cProgramList.Add(cInputPgrms[j], pInput);
    }
  }
  
  // Unlock the input repository and the program list
  m_cInputList.UnLock();
  m_cProgramList.UnLock();
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_Manager::DestroyPgrmTable()
{
  // Nothing to do yet
}



//------------------------------------------------------------------------------
// Inputs initialization
//------------------------------------------------------------------------------
void C_Manager::InitInputs()
{
  // Get the names of all the sources
  C_Application* pApp = C_Application::GetApp();
  ASSERT(pApp);
  C_Vector<C_Setting> vInputs = pApp->GetSettings("Inputs");

  // Lock the input repository
  m_cInputList.Lock();

  // Create the corresponding inputs
  for(unsigned int i = 0; i < vInputs.Size(); i++)
  {
    // Current input
    C_Input* pInput = NULL;

    // Get input name and type
    C_Setting cCurrentInput = vInputs[i];
    C_String strInputName = cCurrentInput.GetName();
    C_String strInputType = cCurrentInput.GetValue();

    // Create the input for the given source
    C_InputModule* pModule = (C_InputModule*)
                             C_Application::GetModuleManager()
                                        ->GetModule("input", strInputType);
    if(pModule)
    {
      pInput = pModule->NewInput(strInputName);
      ASSERT(pInput);
      pInput->SetEventHandler(this);
      Log(m_hLog, LOG_NOTE, "Starting input '"+strInputName+"'");
      try
      {
        // Initialize the input
        pInput->Init();

        // And register it
        m_cInputList.Add(strInputName, pInput);

        Log(m_hLog, LOG_NOTE, "Input '" + strInputName +
            "' sucessfully initialised");
      }
      catch(E_Exception e)
      {
        Log(m_hLog, LOG_ERROR, "Unable to start input '" + strInputName +
            "': " + e.Dump());
        delete pInput;
      }
    }
    else
    {
      Log(m_hLog, LOG_ERROR, "Input type \"" + strInputType + "\" invalid");
    }
  }

  // Unlock the input repository
  m_cInputList.UnLock();
}


//------------------------------------------------------------------------------
// Inputs destruction
//------------------------------------------------------------------------------
void C_Manager::DestroyInputs()
{
  // Lock the input repository
  m_cInputList.Lock();

  C_RepositoryBrowser<C_String, C_Input> cIterator =
                                                m_cInputList.CreateBrowser();
  
  while(cIterator.HasNextItem())
  {
    C_Input* pInput = cIterator.GetNextItem();
    
    try
    {
      pInput->Destroy();
      Log(m_hLog, LOG_NOTE, "Input "+pInput->GetName()+" correctly stopped");
    }
    catch(E_Exception e)
    {
      // Just log the pb and go on with inputs destruction
      C_String strInputName = pInput->GetName();
      Log(m_hLog, LOG_ERROR,
          "Unable to stop input "+strInputName+": "+e.Dump());
    }
  }

  // Unlock the input repository
  m_cInputList.UnLock();
}


//------------------------------------------------------------------------------
// Channels initialization
//------------------------------------------------------------------------------
void C_Manager::InitChannels()
{
  // Lock the channel repository
  m_cChannelList.Lock();

  // Get the names of all the channels
  C_Application* pApp = C_Application::GetApp();
  ASSERT(pApp);
  C_Vector<C_Setting> vChannels = pApp->GetSettings("Channels");
  
  // Create the corresponding channels
  for(unsigned int i = 0; i < vChannels.Size(); i++)
  {
    // Current channel
    C_Channel* pChannel = NULL;

    // Get channel name and type
    C_Setting cCurrentChannel = vChannels[i];
    C_String strChannelName = cCurrentChannel.GetName();
    C_String strChannelType = cCurrentChannel.GetValue();

    C_ChannelModule* pModule = (C_ChannelModule*)
                               C_Application::GetModuleManager()
                                        ->GetModule("channel", strChannelType);
    if(pModule)
    {
      pChannel = pModule->NewChannel(strChannelName);
      ASSERT(pChannel);
      m_cChannelList.Add(strChannelName, pChannel);
      Log(m_hLog, LOG_NOTE, "Channel '"+strChannelName+"' created");
    }
    else
    {
      Log(m_hLog, LOG_ERROR, "Channel type \"" + strChannelType + "\" invalid");
    }
  }

  // Unlock the channel repository
  m_cChannelList.UnLock();
}


//------------------------------------------------------------------------------
// Channels destruction
//------------------------------------------------------------------------------
void C_Manager::DestroyChannels()
{
  // Just make sure that the channel is free
  // to do
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_Manager::StopPrograms()
{
  // Stop all th programs so that the inputs could be destroyed

  m_cBroadcasts.Lock();

  C_Vector<C_Request> cRequests;

  C_RepositoryBrowser<C_String, C_Broadcast> cIterator1 =
                m_cBroadcasts.CreateBrowser();

  while(cIterator1.HasNextItem())
  {
    C_Broadcast* pBroadcast = cIterator1.GetNextItem();
    C_Request* pRequest = new C_Request("stop");
    pRequest->SetArg("channel", pBroadcast->GetChannel()->GetName());
    cRequests.Add(pRequest);
  }

  m_cBroadcasts.UnLock();

  unsigned int iCount = cRequests.Size();
  for(unsigned int i = 0; i < iCount; i++)
  {
    C_Request& cRequest = cRequests[i];
    HandleRequest(cRequest);
    LogDbg(m_hLog, "Remove the broadcast");
    m_cBroadcasts.Lock();
    int iRc = m_cBroadcasts.Remove(cRequest.GetArg("program") + ":" +
                                   cRequest.GetArg("input"));
    m_cBroadcasts.UnLock();
    LogDbg(m_hLog, C_String("Broadcast removed with status ") + iRc);
  }
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
C_Answer C_Manager::HandleRequest(const C_Request& cRequest)
{
  m_cCmdLock.Lock();
  LogDbg(m_hLog, "Manager locked");

  C_Answer cAnswer("Manager");
  
  C_String strCmd = cRequest.GetCmd();

  if(strCmd == "load")
  {
    // Load the input: Not yet implemented 
    ASSERT(false);
  }
  else if(strCmd == "unload")
  {
    // Unload the input: Not yet implemented 
    ASSERT(false);
  }
  else if(strCmd == "browse")
  {
    cAnswer = Browse(cRequest);
  }
  else if(strCmd == "show")
  {
    cAnswer = Show(cRequest);
  }
  else if(strCmd == "start")
  {
    cAnswer = Start(cRequest);
  }
  else if(strCmd == "resume")
  {
    cAnswer = Resume(cRequest);
  }
  else if(strCmd == "suspend")
  {
    cAnswer = Suspend(cRequest);
  }
  else if(strCmd == "stop")
  {
    cAnswer = Stop(cRequest);
  }
  else if(strCmd == "config")
  {
    cAnswer = Config(cRequest);
  }
  else if(strCmd == "program")
  {
    cAnswer = Program(cRequest);
  }
  else if(strCmd == "input")
  {
    cAnswer = Input(cRequest);
  }
  else if(strCmd == "channel")
  {
    cAnswer = Channel(cRequest);
  }
  else
  {
    cAnswer.SetStatus(GEN_ERR);
    cAnswer.AddMessage("Unknown command: "+strCmd);
  }

  m_cCmdLock.UnLock();
  LogDbg(m_hLog, "Manager unlocked");

  return cAnswer;
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_Manager::HandleEvent(const C_Event& cEvent)
{
  m_cCmdLock.Lock();
  LogDbg(m_hLog, "Manager locked");

  m_cEventFifoLock.Lock();
  m_cEventFifo.Add(new C_Event(cEvent));
  m_cEventFifoLock.UnLock();

  m_cEventQueued.Post();

  m_pEventHub->ForwardEvent(cEvent);

  m_cCmdLock.UnLock();
  LogDbg(m_hLog, "Manager unlocked");
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_Manager::PrivHandleEvent(const C_Event& cEvent)
{
  switch(cEvent.GetCode())
  {
  case EOF_EVENT :
  case CONNEXION_LOST_EVENT :
    C_String strInput = cEvent.GetBroadcast()->GetInput()->GetName();
    C_String strPgrm = cEvent.GetBroadcast()->GetProgram()->GetName();

    ASSERT(strInput != "");
    ASSERT(strPgrm != "");

    C_Request cRequest("stop");
    cRequest.SetArg("input", strInput);
    cRequest.SetArg("program", strPgrm);

    C_Answer cAnswer = HandleRequest(cRequest);

    break;
  }
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_Manager::InitWork()
{
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_Manager::DoWork()
{
  while(!m_bStop)
  {
    m_cEventQueued.Wait();

    m_cEventFifoLock.Lock();

    if((m_cEventFifo.Size() >= 1) && (!m_bStop))
    {
      C_Event* pEvent = m_cEventFifo.Remove(0);
      ASSERT(pEvent);

      m_cEventFifoLock.UnLock();

      PrivHandleEvent(*pEvent);

      delete pEvent;
    }
    else
    {
      ASSERT(m_bStop);

      m_cEventFifoLock.UnLock();
    }
  }
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_Manager::StopWork()
{
   m_bStop = true;
   m_cEventQueued.Post();
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_Manager::CleanWork()
{
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
C_Answer C_Manager::Browse(const C_Request& cRequest)
{
  C_Answer cAnswer("Manager");
  cAnswer.SetStatus(NO_ERR);
  cAnswer.AddMessage("Program Table");

  m_cProgramList.Lock();

  C_PgrmDirectoryBrowser cBrowser = m_cProgramList.CreateBrowser();
  
  while(cBrowser.HasNext())
  {
    C_PgrmInfo* pPgrmInfo = cBrowser.GetNext();
    ASSERT(pPgrmInfo);

    // Build a subanswer with the pgrm description and the list of providers
    const C_Program* pPgrm = pPgrmInfo->GetProgram();
    C_String strPgrmName = pPgrm->GetName();
    ASSERT(strPgrmName != "");
    C_Answer cPgrmDescr(strPgrmName);
    cPgrmDescr.SetStatus(NO_ERR);
    cPgrmDescr.AddMessage(pPgrm->GetDescription());

    const C_Vector<C_Input>& cProviders = pPgrmInfo->GetProviders();
    C_String strProviders = "Provided by";
    for(unsigned int i = 0; i < cProviders.Size(); i++)
    {
      strProviders += " " + cProviders[i].GetName();
    }
    cPgrmDescr.AddMessage(strProviders);

    // Add it to the answer
    cAnswer.Add(cPgrmDescr);
  }

  m_cProgramList.UnLock();

  return cAnswer;
}

//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
C_Answer C_Manager::Show(const C_Request& cRequest)
{
  C_Answer cAnswer("Manager");
  C_String strCommand = cRequest.GetArg("command");
  ASSERT(strCommand != "");

  if (strCommand.ToLower() == "broadcast")
  {
    cAnswer.AddMessage("Showing running broadcasts.");
    // Lock Broadcast repository
    m_cBroadcasts.Lock();

    C_RepositoryBrowser<C_String, C_Broadcast> cIterator1 =
                  m_cBroadcasts.CreateBrowser();

    while(cIterator1.HasNextItem())
    {
      C_Broadcast* pBroadcast = cIterator1.GetNextItem();
      C_Answer cBroadcast(pBroadcast->GetChannel()->GetName());
      cBroadcast.SetStatus(NO_ERR);
      cBroadcast.AddMessage("Channel '"+pBroadcast->GetChannel()->GetName()+"'");
      cBroadcast.AddMessage("Program '"+pBroadcast->GetProgram()->GetName()+"'");
      cBroadcast.AddMessage("Input '"+pBroadcast->GetInput()->GetName()+"'");
      cBroadcast.AddMessage("Input '"+pBroadcast->GetChannel()->GetName()+"'");
      C_String strOptions = "Options used are '";
      if (pBroadcast->GetOption("rtp")=="1")
         strOptions += "--rtp ";
      if (pBroadcast->GetOption("end")=="1")
         strOptions += "--end ";
      if (pBroadcast->GetOption("loop")=="1")
         strOptions += "--loop ";
      cBroadcast.AddMessage(strOptions+"'");
      cAnswer.Add(cBroadcast);
    }

    // UnLock Broadcast repository
    m_cBroadcasts.UnLock();
    cAnswer.SetStatus(NO_ERR);
  }
  else if (strCommand.ToLower() == "channel")
  {
    cAnswer.AddMessage("Showing available channels.");

    // Lock the channel repository
    m_cChannelList.Lock();

    // Get the names of all the channels
    C_Application* pApp = C_Application::GetApp();
    ASSERT(pApp);
    C_Vector<C_Setting> vChannels = pApp->GetSettings("Channels");

    // Create the corresponding channels
    for(unsigned int i = 0; i < vChannels.Size(); i++)
    {
      // Get channel name and type
      C_Setting cCurrentChannel = vChannels[i];
      C_String strChannelName = cCurrentChannel.GetName();
      C_String strChannelType = cCurrentChannel.GetValue();

      C_String strKey = strChannelName + C_String(".type");
      C_String strType = pApp->GetSetting(strKey, "");

      // Create answer
      C_Answer cChannelDescr(strChannelName);
      cChannelDescr.SetStatus(NO_ERR);
      cChannelDescr.AddMessage("Channel '"+strChannelName+"' of type '"+strChannelType+"'");
      cChannelDescr.AddMessage("Streamtype '"+strType+"'");
      cAnswer.Add(cChannelDescr);
    }
    cAnswer.SetStatus(NO_ERR);

    // Unlock the channel repository
    m_cChannelList.UnLock();
  }
  else if (strCommand.ToLower() == "program")
  {
    cAnswer.AddMessage("Showing available programs.");
    cAnswer.SetStatus(NO_ERR);

    m_cProgramList.Lock();
    C_PgrmDirectoryBrowser cBrowser = m_cProgramList.CreateBrowser();
    while(cBrowser.HasNext())
    {
      C_PgrmInfo* pPgrmInfo = cBrowser.GetNext();
      ASSERT(pPgrmInfo);

      // Build a subanswer with the pgrm description and the list of providers
      const C_Program* pPgrm = pPgrmInfo->GetProgram();
      C_String strPgrmName = pPgrm->GetName();
      ASSERT(strPgrmName != "");
      C_Answer cPgrmDescr(strPgrmName);
      cPgrmDescr.SetStatus(NO_ERR);
      cPgrmDescr.AddMessage(pPgrm->GetDescription());

      const C_Vector<C_Input>& cProviders = pPgrmInfo->GetProviders();
      C_String strProviders = "Provided by";
      for(unsigned int i = 0; i < cProviders.Size(); i++)
      {
        strProviders += " " + cProviders[i].GetName();
      }
      cPgrmDescr.AddMessage(strProviders);

      // Add it to the answer
      cAnswer.Add(cPgrmDescr);
    }
    m_cProgramList.UnLock();
  }
  else if (strCommand.ToLower() == "input")
  {
    cAnswer.AddMessage("Showing available inputs.");
    cAnswer.SetStatus(NO_ERR);

    // Lock the input repository
    m_cInputList.Lock();
    C_RepositoryBrowser<C_String, C_Input> cIterator =
                                                  m_cInputList.CreateBrowser();
    while(cIterator.HasNextItem())
    {
      C_Input* pInput = cIterator.GetNextItem();
      C_Answer cInputDescr(pInput->GetName());
      cInputDescr.SetStatus(NO_ERR);
      cInputDescr.AddMessage("Input '"+pInput->GetName()+"'");
      cAnswer.Add(cInputDescr);
    }
    // Unlock the input repository
    m_cInputList.UnLock();
  }
  else
  {
    cAnswer.SetStatus(GEN_ERR);
    cAnswer.AddMessage("What to show? Use: broadcast, channel, program or input.");
  }
  return cAnswer;
}

//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
C_Answer C_Manager::Start(const C_Request& cRequest)
{
  C_Answer cAnswer("Manager");
  cAnswer.SetStatus(NO_ERR);

  // Find the channel
  C_String strChannel = cRequest.GetArg("channel");
  ASSERT(strChannel != "");
  C_Channel* pChannel = m_cChannelList.Get(strChannel);
  if(!pChannel)
  {
    cAnswer.SetStatus(GEN_ERR);
    cAnswer.AddMessage("Unknown channel");
  }
  else if(!pChannel->IsFree())
  {
    cAnswer.SetStatus(GEN_ERR);
    cAnswer.AddMessage("Channel is busy");
    pChannel = NULL;
  }

  // Find the pgrm
  C_String strPgrm = cRequest.GetArg("program");
  ASSERT(strPgrm != "");
  const C_Program* pPgrm = m_cProgramList.GetPgrm(strPgrm);
  if(!pPgrm)
  {
    cAnswer.SetStatus(GEN_ERR);
    cAnswer.AddMessage("Unknown pgrm");
  }  
   
  // Find the input
  C_String strInput = cRequest.GetArg("input");
  C_Input* pInput = m_cInputList.Get(strInput);
  if(!pInput)
  {
    cAnswer.SetStatus(GEN_ERR);
    cAnswer.AddMessage("input doesn't exist");
  }

  // kludge; Dkludg ? --Bozo
  if(pPgrm && pInput && pChannel)
  {
    C_Broadcast* pBroadcast = new C_Broadcast(pPgrm, pInput, pChannel);
    pBroadcast->SetOptions(cRequest.GetArgs());

    C_Answer cInputAnswer = pInput->StartStreaming(pBroadcast);

    if(!cInputAnswer.GetStatus())
    {
      m_cBroadcasts.Lock();
      m_cBroadcasts.Add(strChannel, pBroadcast);
      m_cBroadcasts.UnLock();
    }
    else
    {
      cAnswer.SetStatus(cInputAnswer.GetStatus());
    }

    cAnswer.Add(cInputAnswer);
  }
  else if(pInput)
    m_cInputList.Release(strInput);

  return cAnswer;
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
C_Answer C_Manager::Resume(const C_Request& cRequest)
{
  C_Answer cAnswer("Manager");
  cAnswer.SetStatus(NO_ERR);

  C_String strChannel = cRequest.GetArg("channel");
  ASSERT(strChannel != "");

  m_cBroadcasts.Lock();
  C_Broadcast* pBroadcast = m_cBroadcasts.Find(strChannel);
  if (pBroadcast)
  {
    C_Input* pInput = pBroadcast->GetInput();

    C_Answer cInputAnswer = pInput->ResumeStreaming(pBroadcast);
    if(cInputAnswer.GetStatus())
    {
      cAnswer.SetStatus(cInputAnswer.GetStatus());
    }
    cAnswer.Add(cInputAnswer);

    // Release repository items (Channel, Program, Input)
    m_cChannelList.Release(strChannel);
    m_cProgramList.ReleasePgrm(pBroadcast->GetProgram()->GetName());
    m_cInputList.Release(pInput->GetName());
  }
  else
  {
    cAnswer.SetStatus(GEN_ERR);
    cAnswer.AddMessage("Nothing broadcasted to channel " + strChannel);
  }
  m_cBroadcasts.UnLock();

  return cAnswer;
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
C_Answer C_Manager::Suspend(const C_Request& cRequest)
{
   C_Answer cAnswer("Manager");
   cAnswer.SetStatus(NO_ERR);

   C_String strChannel = cRequest.GetArg("channel");
   ASSERT(strChannel != "");

   m_cBroadcasts.Lock();
   C_Broadcast* pBroadcast = m_cBroadcasts.Find(strChannel);
   if (pBroadcast)
   {
     C_Input* pInput = pBroadcast->GetInput();

     C_Answer cInputAnswer = pInput->SuspendStreaming(pBroadcast);
     if(cInputAnswer.GetStatus())
     {
       cAnswer.SetStatus(cInputAnswer.GetStatus());
     }
     cAnswer.Add(cInputAnswer);

     // Release repository items (Channel, Program, Input)
     m_cChannelList.Release(strChannel);
     m_cProgramList.ReleasePgrm(pBroadcast->GetProgram()->GetName());
     m_cInputList.Release(pInput->GetName());
   }
   else
   {
     cAnswer.SetStatus(GEN_ERR);
     cAnswer.AddMessage("Nothing broadcasted to channel " + strChannel);
   }
   m_cBroadcasts.UnLock();

   return cAnswer;
}

//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
C_Answer C_Manager::Stop(const C_Request& cRequest)
{
  C_Answer cAnswer("Manager");
  cAnswer.SetStatus(NO_ERR);

  C_String strChannel = cRequest.GetArg("channel");
  ASSERT(strChannel != "");

  m_cBroadcasts.Lock();
  C_Broadcast* pBroadcast = m_cBroadcasts.Find(strChannel);
  if (pBroadcast)
  {
    C_Input* pInput = pBroadcast->GetInput();
    C_Answer cInputAnswer = pInput->StopStreaming(pBroadcast);

    if(cInputAnswer.GetStatus())
    {
      cAnswer.SetStatus(cInputAnswer.GetStatus());
    }
    cAnswer.Add(cInputAnswer);

    // Release repository items (Channel, Program, Input)
    m_cChannelList.Release(strChannel);
    m_cProgramList.ReleasePgrm(pBroadcast->GetProgram()->GetName());
    m_cInputList.Release(pInput->GetName());

    LogDbg(m_hLog, "Remove the broadcast");
    int iRc = m_cBroadcasts.Remove(strChannel);
    LogDbg(m_hLog, C_String("Broadcast removed with status ") + iRc);
  }
  else
  {
    cAnswer.SetStatus(GEN_ERR);
    cAnswer.AddMessage("Nothing broadcasted to channel " + strChannel);
  }
  m_cBroadcasts.UnLock();

  return cAnswer;
}

//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
C_Answer C_Manager::Program(const C_Request& cRequest)
{
  C_Answer cAnswer("Manager");
  cAnswer.SetStatus(NO_ERR);

  C_String strProgram = cRequest.GetArg("name");
  ASSERT(strProgram != "");

  C_String strFileName = cRequest.GetArg("filename");
  ASSERT(strFileName != "");

  C_String strType = cRequest.GetArg("type");
  ASSERT(strType != "");

  C_String strInput = cRequest.GetArg("input");
  C_Input* pInput = m_cInputList.Get(strInput);
  if(!pInput)
  {
    cAnswer.SetStatus(GEN_ERR);
    cAnswer.AddMessage("input doesn't exist");
    return cAnswer;
  }

  // optional arguments
  C_String strAdd = cRequest.GetArg("add");
  C_String strDelete = cRequest.GetArg("delete");

  if ((strAdd == "1") && (strDelete == "1"))
  {
    cAnswer.SetStatus(GEN_ERR);
    cAnswer.AddMessage("Channel does not allow use of --add and --delete at the same time.");
    return cAnswer;
  }

  if (strAdd == "1") // Add Program
  {
    // Check if program is already present
    const C_Program* pProgram = m_cProgramList.GetPgrm(strProgram);
    if(pProgram)
    {
      cAnswer.SetStatus(GEN_ERR);
      cAnswer.AddMessage("Ignoring command program already added.");
      m_cProgramList.ReleasePgrm(pProgram->GetName());
      return cAnswer;
    }

    // Lock the input repository and the program list
    m_cProgramList.Lock();
    m_cInputList.Lock();

    // First update C_Settings in pInput
    C_Answer cInputAnswer = pInput->UpdateProgram(strProgram,strFileName,strType);
    if(cInputAnswer.GetStatus())
    {
      cAnswer.SetStatus(cInputAnswer.GetStatus());
    }
    cAnswer.Add(cInputAnswer);

    // Then Update m_cProgramList
    C_List<C_Program> cInputPgrms = pInput->GetAvailablePgrms();
    unsigned int iPgrmNumber = cInputPgrms.Size();
    if (iPgrmNumber>0)
      m_cProgramList.Add(cInputPgrms[iPgrmNumber-1], pInput);
    else
    {
      cAnswer.SetStatus(GEN_ERR);
      cAnswer.AddMessage("ProgramList is empty.");
    }
    // debug
    Log(m_hLog, LOG_NOTE, "Program configuration changed.");
    for(unsigned int j = 0; j < iPgrmNumber; j++)
    {
       Log(m_hLog, LOG_NOTE, "Program: " + cInputPgrms[j].GetName());
    }
    // Unlock the input repository and the program list
    m_cInputList.UnLock();
    m_cProgramList.UnLock();
  }
  else
  if (strDelete == "1") // Delete Program
  {
    Log(m_hLog, LOG_NOTE, "Program being deleted.");
    // Check if program is known
    const C_Program* pProgram = m_cProgramList.GetPgrm(strProgram);
    if(!pProgram)
    {
      cAnswer.SetStatus(GEN_ERR);
      cAnswer.AddMessage("Cannot delete unknown program.");
      return cAnswer;
    }
    m_cProgramList.ReleasePgrm(pProgram->GetName());

    // Lock the input repository and the program list
    m_cProgramList.Lock();
    m_cInputList.Lock();

    // First update C_Settings in pInput
    C_Answer cInputAnswer = pInput->DeleteProgram(strProgram);
    if(cInputAnswer.GetStatus())
    {
      cAnswer.SetStatus(cInputAnswer.GetStatus());
      cAnswer.AddMessage("Error removing program from input.");
      m_cProgramList.UnLock();
      m_cInputList.UnLock();
      return cAnswer;
    }
    cAnswer.Add(cInputAnswer);

    // Then Delete from m_cProgramList
    if(!cInputAnswer.GetStatus())
    {
      int iRc = m_cProgramList.Remove(*pProgram, pInput);
      if(iRc == GEN_ERR)
      {
        cAnswer.SetStatus(GEN_ERR);
        cAnswer.AddMessage("Error removing program from configuration.");
        m_cProgramList.UnLock();
        m_cInputList.UnLock();
        return cAnswer;
      }
    }
    // Unlock the input repository and the program list
    m_cInputList.UnLock();
    m_cProgramList.UnLock();

    // debug
    Log(m_hLog, LOG_NOTE, "Program configuration changed.");
    C_List<C_Program> cDebugInputPgrms = pInput->GetAvailablePgrms();
    unsigned int iPgrmNumber = cDebugInputPgrms.Size();
    if (iPgrmNumber <= 0)
    {
      cAnswer.SetStatus(GEN_ERR);
      cAnswer.AddMessage("ProgramList is empty.");
    }
    for(unsigned int j = 0; j < iPgrmNumber; j++)
    {
      Log(m_hLog, LOG_NOTE, "Program: " + cDebugInputPgrms[j].GetName());
    }
  }
  else
  {
    cAnswer.SetStatus(GEN_ERR);
    cAnswer.AddMessage("Commaned expects --add or --delete.");
    Log(m_hLog, LOG_ERROR, "Command Syntax Error.");
  }
  return cAnswer;
}

//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
C_Answer C_Manager::Channel(const C_Request& cRequest)
{
  C_Answer cAnswer("Manager");
  cAnswer.SetStatus(NO_ERR);

  C_String strChannelName = cRequest.GetArg("name");
  ASSERT(strChannelName != "");

  C_String strChannelType = cRequest.GetArg("type");
  ASSERT(strChannelType != "");

  C_String strDstHost = cRequest.GetArg("dest_host");
  ASSERT(strDstHost != "");

  C_String strDstPort = cRequest.GetArg("dest_port");
  ASSERT(strDstPort != "");

  C_String strStreamType = cRequest.GetArg("stream_type");
  ASSERT(strStreamType != "");

  // Optional arguments
  C_String strDomain = cRequest.GetArg("domain");
  if (strDomain == "")
     strDomain = "inet4";

  C_String strTTL = cRequest.GetArg("ttl");
  C_String strInterface = cRequest.GetArg("interface");

  C_String strAdd = cRequest.GetArg("add");
  C_String strDelete = cRequest.GetArg("delete");

  if ((strAdd == "1") && (strDelete == "1"))
  {
    cAnswer.SetStatus(GEN_ERR);
    cAnswer.AddMessage("Channel does not allow use of --add and --delete at the same time.");
    return cAnswer;
  }

  if (strAdd == "1") // Add Channel
  {
    // check if channel is already known
    C_Channel* pChannel = m_cChannelList.Get(strChannelName);
    if(pChannel)
    {
      cAnswer.SetStatus(GEN_ERR);
      cAnswer.AddMessage("Channel already present");
      m_cChannelList.Release(strChannelName);
      return cAnswer;
    }
    // Lock the input repository and the program list
    m_cChannelList.Lock();

    // Get the names of all the channels
    C_Application* pApp = C_Application::GetApp();
    ASSERT(pApp);

    C_String strKey = C_String("Channels.") + strChannelName;
    pApp->SetSettings( strKey, strChannelType );
    strKey = strChannelName+C_String(".DstHost");
    pApp->SetSettings( strKey, strDstHost );
    strKey = strChannelName+C_String(".DstPort");
    pApp->SetSettings( strKey, strDstPort );
    strKey = strChannelName+C_String(".Type");
    pApp->SetSettings( strKey, strStreamType );
    strKey = strChannelName+C_String(".Domain");
    pApp->SetSettings( strKey, strDomain );
    strKey = strChannelName+C_String(".TTL");
    pApp->SetSettings( strKey, strTTL );
    strKey = strChannelName+C_String(".Interface");
    pApp->SetSettings( strKey, strInterface );

    C_ChannelModule* pModule = (C_ChannelModule*)
                               C_Application::GetModuleManager()
                                        ->GetModule("channel", strChannelType);
    if(pModule)
    {
      // Current channel
      C_Channel* pChannel = NULL;
      pChannel = pModule->NewChannel(strChannelName);
      ASSERT(pChannel);
      m_cChannelList.Add(strChannelName, pChannel);
      Log(m_hLog, LOG_NOTE, "Channel '"+strChannelName+"' created");
    }
    else
    {
      cAnswer.SetStatus(GEN_ERR);
      cAnswer.AddMessage("Channel type is invalid.");
      Log(m_hLog, LOG_ERROR, "Channel type \"" + strChannelType + "\" invalid");
    }

    // debug
    Log(m_hLog, LOG_NOTE, "Channel configuration changed.");
    C_Vector<C_Setting> vChannels = pApp->GetSettings("Channels");
    // Unlock the chanel list
    m_cChannelList.UnLock();
  }
  else
  if (strDelete == "1") // Delete Channel
  {
    // check if channel is already known
    C_Channel* pChannel = m_cChannelList.Get(strChannelName);
    if(!pChannel)
    {
      cAnswer.SetStatus(GEN_ERR);
      cAnswer.AddMessage("Channel is unknown");
      return cAnswer;
    }
    else if(!pChannel->IsFree())
    {
      cAnswer.SetStatus(GEN_ERR);
      cAnswer.AddMessage("Channel is busy");
      pChannel = NULL;
      m_cChannelList.Release(strChannelName);
      return cAnswer;
    }
    m_cChannelList.Release(strChannelName);

    // Lock the input repository and the program list
    m_cChannelList.Lock();

    C_Application* pApp = C_Application::GetApp();
    ASSERT(pApp);

    // So now, we can remove channel from all the lists
    int iRc = m_cChannelList.Remove(strChannelName);
    if (iRc)
    {
      Log(m_hLog, LOG_NOTE, "Channel '" + strChannelName +
                                                 "' can not be deleted !");
      cAnswer.SetStatus(GEN_ERR);
      cAnswer.AddMessage("Channel can not be deleted");
    }
    else
    {
      C_String strKey = C_String("channels.")+strChannelName;
      if (pApp->GetSetting(strKey, "") != "")
        pApp->DeleteSetting(strKey);
      strKey = strChannelName + C_String(".dsthost") ;
      if (pApp->GetSetting(strKey, "") != "")
        pApp->DeleteSetting(strKey);
      strKey = strChannelName + C_String(".dstport") ;
      if (pApp->GetSetting(strKey, "") != "")
        pApp->DeleteSetting(strKey);
      strKey = strChannelName + C_String(".type") ;
      if (pApp->GetSetting(strKey, "") != "")
        pApp->DeleteSetting(strKey);
      strKey = strChannelName + C_String(".domain") ;
      if (pApp->GetSetting(strKey, "") != "")
        pApp->DeleteSetting(strKey);
      strKey = strChannelName + C_String(".interface") ;
      if (pApp->GetSetting(strKey, "") != "")
        pApp->DeleteSetting(strKey);
      strKey = strChannelName + C_String(".ttl");
      if (pApp->GetSetting(strKey, "") != "")
        pApp->DeleteSetting(strKey );
      Log(m_hLog, LOG_NOTE, "Channel '" + strChannelName + "' removed !");
    }
    m_cChannelList.UnLock();
  }
  else
  {
    cAnswer.SetStatus(GEN_ERR);
    cAnswer.AddMessage("Commaned expects --add or --delete.");
    Log(m_hLog, LOG_ERROR, "Command Syntax Error.");
  }
  return cAnswer;
}

//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
C_Answer C_Manager::Config(const C_Request& cRequest)
{
  C_Answer cAnswer("Manager");
  cAnswer.SetStatus(NO_ERR);

  C_String strFileName = cRequest.GetArg("filename");
  ASSERT(strFileName != "");

  C_String strLoad = cRequest.GetArg("load");
  C_String strSave = cRequest.GetArg("save");

  if ((strLoad == "1") && (strSave == "1"))
  {
    cAnswer.SetStatus(GEN_ERR);
    cAnswer.AddMessage("Config does not allow use of --add and --delete at the same time.");
    return cAnswer;
  }
  if ( strLoad == "1" )
  {
      // stop all programs
      StopPrograms();

      // Destroy current configuration
      DestroyPgrmTable();
      DestroyInputs();
      DestroyChannels();

      // Init new configuration
      InitChannels();
      InitInputs();
      InitPgrmTable();

      cAnswer.AddMessage("Configuration reloaded from default vls.cfg.");

      // debug
      Log(m_hLog, LOG_NOTE, "Configuration loaded.");
  }
  else if ( strSave == "1" )
  {
      // Lock the input repository and the program list
      m_cProgramList.Lock();
      m_cInputList.Lock();
      m_cChannelList.Lock();

      // Not Implemented yet
      cAnswer.SetStatus(GEN_ERR);
      cAnswer.AddMessage("Configuration save - Not Implemented Yet");

      // debug
      Log(m_hLog, LOG_NOTE, "Configuration written - Not Implemented Yet.");

      // Unlock the input repository and the program list
      m_cInputList.UnLock();
      m_cProgramList.UnLock();
      m_cChannelList.UnLock();
  }
  else
  {
    cAnswer.SetStatus(GEN_ERR);
    cAnswer.AddMessage("Commaned expects --load or --save.");
    Log(m_hLog, LOG_ERROR, "Command Syntax Error.");
  }
  return cAnswer;
}

//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
C_Answer C_Manager::Input(const C_Request& cRequest)
{
  C_Answer cAnswer("Manager");
  cAnswer.SetStatus(NO_ERR);

  C_String strInputName = cRequest.GetArg("name");
  ASSERT( strInputName != "" );

  C_String strInputType = cRequest.GetArg("type");
  ASSERT( strInputType != "" );

  // for type=local
  C_String strConfigPath;
  if (strInputType == "local")
  {
    strConfigPath = cRequest.GetArg("dir");
    ASSERT( strConfigPath != "" );
    if (strConfigPath == "")
    {
      cAnswer.SetStatus(GEN_ERR);
      cAnswer.AddMessage( "Input type \"local\" expects \"config path\" to input.cfg file." );
    }
  }
  // for type=video
  C_String strDevice;
  C_String strFormat;
  if (strInputType == "video")
  {
    strDevice = cRequest.GetArg("device");
    ASSERT( strDevice != "" );
    if (strDevice == "")
    {
      cAnswer.SetStatus(GEN_ERR);
      cAnswer.AddMessage( "Input type \"video\" expects \"device\"." );
    }
    strFormat = cRequest.GetArg("format");
    ASSERT( strFormat != "" );
    if (strFormat == "")
    {
      cAnswer.SetStatus(GEN_ERR);
      cAnswer.AddMessage( "Input type \"video\" expects \"stream format type\" (Mpeg1-PS, Mpeg2-PS, Mpeg2-TS)." );
    }
  }
  // for type=dvb
  C_String strDeviceNumber;
  C_String strDeviceType;
  C_String strFrequency;
  C_String strPolarization;
  C_String strSymbolRate;
  C_String strUseDiSEqC;
  C_String strLNB_Lof1;
  C_String strLNB_Lof2;
  C_String strLNB_SLof;
  if (strInputType == "dvb")
  {
    strDeviceNumber = cRequest.GetArg("device_number");
    ASSERT( strDeviceNumber != "" );
    if (strDeviceNumber == "")
    {
      cAnswer.SetStatus(GEN_ERR);
      cAnswer.AddMessage( "Input type \"dvb\" expects \"device number\"." );
    }
    strDeviceType = cRequest.GetArg("device_type");
    ASSERT( strDeviceType != "" );
    if (strDeviceType == "")
    {
      cAnswer.SetStatus(GEN_ERR);
      cAnswer.AddMessage( "Input type \"dvb\" expects \"device type\" (nova)." );
    }
    strFrequency = cRequest.GetArg("frequency");
    ASSERT( strFrequency != "" );
    if (strFrequency == "")
    {
      cAnswer.SetStatus(GEN_ERR);
      cAnswer.AddMessage( "Input type \"dvb\" expects \"transponder frequency\"." );
    }
    strPolarization = cRequest.GetArg("polarization");
    ASSERT( strPolarization != "" );
    if (strPolarization == "")
    {
      cAnswer.SetStatus(GEN_ERR);
      cAnswer.AddMessage( "Input type \"dvb\" expects \"polarization\" (0 is vertical and 1 is horizontal)." );
    }
    strSymbolRate = cRequest.GetArg("symbol_rate");
    ASSERT( strSymbolRate != "" );
    if (strSymbolRate == "")
    {
      cAnswer.SetStatus(GEN_ERR);
      cAnswer.AddMessage( "Input type \"dvb\" expects \"symbol rate\"." );
    }
    strUseDiSEqC = cRequest.GetArg("use_disecq");
    ASSERT( strUseDiSEqC != "" );
    if (strUseDiSEqC == "")
    {
      cAnswer.SetStatus(GEN_ERR);
      cAnswer.AddMessage( "Input type \"dvb\" expects \"use DiSEqC\" (0 is false and 1 is true)." );
    }
    strLNB_Lof1 = cRequest.GetArg("lnb_lof1");
    ASSERT( strLNB_Lof1 != "" );
    if (strLNB_Lof1  == "")
    {
      cAnswer.SetStatus(GEN_ERR);
      cAnswer.AddMessage( "Input type \"dvb\" expects \"LNB_Lof1\"." );
    }
    strLNB_Lof2 = cRequest.GetArg("lnb_lof2");
    ASSERT( strLNB_Lof2 != "" );
    if (strLNB_Lof2  == "")
    {
      cAnswer.SetStatus(GEN_ERR);
      cAnswer.AddMessage( "Input type \"dvb\" expects \"LNB_Lof2\"." );
    }
    strLNB_SLof = cRequest.GetArg("lnb_slof");
    ASSERT( strLNB_SLof != "" );
    if (strLNB_SLof  == "")
    {
      cAnswer.SetStatus(GEN_ERR);
      cAnswer.AddMessage( "Input type \"dvb\" expects \"SLNB_Lof\"." );
    }
  }
  // Check for errors in command syntax
  if (cAnswer.GetStatus() == GEN_ERR)
  {
    Log(m_hLog, LOG_ERROR, "Input Command Syntax Error.");
    cAnswer.AddMessage( "Input Command Syntax Error." );
    return cAnswer;
  }

  // Optional arguments
  C_String strAdd = cRequest.GetArg("add");
  C_String strDelete = cRequest.GetArg("delete");
  if ((strAdd == "1") && (strDelete == "1"))
  {
    cAnswer.SetStatus(GEN_ERR);
    cAnswer.AddMessage("Input does not allow use of --add and --delete at the same time.");
    Log(m_hLog, LOG_ERROR, "Input Command Syntax Error.");
    return cAnswer;
  }

  // Get the names of all the sources
  C_Application* pApp = C_Application::GetApp();
  ASSERT(pApp);

  // Process input commands
  if (strAdd == "1")
  {
    Log(m_hLog, LOG_NOTE,"Adding input: \"" +strInputName+ "\"");

    // check if input is already known
    C_Input* pInput = m_cInputList.Get(strInputName);
    if(pInput)
    {
      cAnswer.SetStatus(GEN_ERR);
      cAnswer.AddMessage("Input already present");
      m_cInputList.Release(strInputName);
      return cAnswer;
    }

    // Lock the input repository
    m_cInputList.Lock();

    C_String strKey = C_String("Inputs.") + strInputName;
    pApp->SetSettings( strKey, strInputType );
    // for type=local
    if (strInputType == "local")
    {
      strKey = strInputName+C_String(".ConfigPath");
      pApp->SetSettings( strKey, strConfigPath );
    }
    else // for type=video
    if (strInputType == "video")
    {
      strKey = strInputName+C_String(".Device");
      pApp->SetSettings( strKey, strDevice );
      strKey = strInputName+C_String(".Type");
      pApp->SetSettings( strKey, strFormat );
    }
    else // for type=dvb
    if (strInputType == "dvb")
    {
      strKey = strInputName+C_String(".DeviceNumber");
      pApp->SetSettings( strKey, strDeviceNumber );
      strKey = strInputName+C_String(".DeviceType");
      pApp->SetSettings( strKey, strDeviceType );
      strKey = strInputName+C_String(".Frequency");
      pApp->SetSettings( strKey, strFrequency );
      strKey = strInputName+C_String(".Polarization");
      pApp->SetSettings( strKey, strPolarization );
      strKey = strInputName+C_String(".SymbolRate");
      pApp->SetSettings( strKey, strSymbolRate );
      strKey = strInputName+C_String(".DiSEqc");
      pApp->SetSettings( strKey, strUseDiSEqC );
      strKey = strInputName+C_String(".LNB_Lof1");
      pApp->SetSettings( strKey, strLNB_Lof1 );
      strKey = strInputName+C_String(".LNB_Lof2");
      pApp->SetSettings( strKey, strLNB_Lof2 );
      strKey = strInputName+C_String(".LNB_SLof");
      pApp->SetSettings( strKey, strLNB_SLof );
    }

    // Create the input for the given source
    C_InputModule* pModule = (C_InputModule*)
                             C_Application::GetModuleManager()
                                        ->GetModule("input", strInputType);
    if(pModule)
    {
      C_Input* pNewInput = NULL;

      pNewInput = pModule->NewInput(strInputName);
      ASSERT(pNewInput);
      pNewInput->SetEventHandler(this);
      cAnswer.AddMessage("Initializing input '"+strInputName+"'");
      try
      {
        // Initialize the input
        pNewInput->Init();

        // And register it
        m_cInputList.Add(strInputName, pNewInput);

        cAnswer.AddMessage("Input '" + strInputName + "' sucessfully initialised");
        Log(m_hLog, LOG_NOTE,"Adding input: \"" +strInputName+ "\" succeeded.");
      }
      catch(E_Exception e)
      {
        cAnswer.SetStatus(GEN_ERR);
        cAnswer.AddMessage("Unable to add input '" + strInputName + "': " + e.Dump());
        Log(m_hLog, LOG_ERROR,"Adding input: \"" +strInputName+ "\" failed.");
        delete pNewInput;
      }
    }
    else
    {
      cAnswer.SetStatus(GEN_ERR);
      cAnswer.AddMessage("Input type \"" + strInputType + "\" invalid");
      Log(m_hLog, LOG_ERROR,"Input type: \"" +strInputName+ "\" invalid");
    }
    // Unlock the input repository
    m_cInputList.UnLock();
  }
  else if (strDelete == "1")
  {
    Log(m_hLog, LOG_NOTE,"Deleting input: \"" +strInputName+ "\"");

    // check if input is already known
    C_Input* pInput = m_cInputList.Get(strInputName);
    if(!pInput)
    {
      cAnswer.SetStatus(GEN_ERR);
      cAnswer.AddMessage("Input unknown present");
      return cAnswer;
    }
    else if(!pInput->IsFree())
    {
      cAnswer.SetStatus(GEN_ERR);
      cAnswer.AddMessage("Input is busy");
      pInput = NULL;
      m_cInputList.Release(strInputName);
      return cAnswer;
    }
    m_cInputList.Release(strInputName);

    // Lock the input repository
    m_cInputList.Lock();

    // So now, we can remove input from all the lists
    int iRc = m_cInputList.Remove(strInputName);
    if (iRc)
    {
      Log(m_hLog, LOG_ERROR, "Input '" + strInputName + "' can not be deleted !");
      cAnswer.SetStatus(GEN_ERR);
      cAnswer.AddMessage("Input can not be deleted");
    }
    else
    {
      C_String strKey = C_String("inputs.")+strInputName;
      if (pApp->GetSetting(strKey, "") != "")
        pApp->DeleteSetting(strKey);
      // for type=local
      if (strInputType == "local")
      {
        strKey = strInputName+C_String(".ConfigPath");
        if (pApp->GetSetting(strKey, "") != "")
          pApp->DeleteSetting(strKey);
      }
      else // for type=video
      if (strInputType == "video")
      {
        strKey = strInputName+C_String(".Device");
        if (pApp->GetSetting(strKey, "") != "")
          pApp->DeleteSetting(strKey);
        strKey = strInputName+C_String(".Type");
        if (pApp->GetSetting(strKey, "") != "")
          pApp->DeleteSetting(strKey);
      }
      else // for type=dvb
      if (strInputType == "dvb")
      {
        strKey = strInputName+C_String(".DeviceNumber");
        if (pApp->GetSetting(strKey, "") != "")
          pApp->DeleteSetting(strKey);
        strKey = strInputName+C_String(".DeviceType");
        if (pApp->GetSetting(strKey, "") != "")
          pApp->DeleteSetting(strKey);
        strKey = strInputName+C_String(".Frequency");
        if (pApp->GetSetting(strKey, "") != "")
          pApp->DeleteSetting(strKey);
        strKey = strInputName+C_String(".Polarization");
        if (pApp->GetSetting(strKey, "") != "")
          pApp->DeleteSetting(strKey);
        strKey = strInputName+C_String(".SymbolRate");
        if (pApp->GetSetting(strKey, "") != "")
          pApp->DeleteSetting(strKey);
        strKey = strInputName+C_String(".DiSEqc");
        if (pApp->GetSetting(strKey, "") != "")
          pApp->DeleteSetting(strKey);
        strKey = strInputName+C_String(".LNB_Lof1");
        if (pApp->GetSetting(strKey, "") != "")
          pApp->DeleteSetting(strKey);
        strKey = strInputName+C_String(".LNB_Lof2");
        if (pApp->GetSetting(strKey, "") != "")
          pApp->DeleteSetting(strKey);
        strKey = strInputName+C_String(".LNB_SLof");
        if (pApp->GetSetting(strKey, "") != "")
          pApp->DeleteSetting(strKey);
      }
      Log(m_hLog, LOG_NOTE, "Input '" + strInputName + "' removed !");
    }
    m_cInputList.UnLock();
  }
  else
  {
    cAnswer.SetStatus(GEN_ERR);
    cAnswer.AddMessage("Commaned expects --add or --delete.");
    Log(m_hLog, LOG_ERROR, "Command Syntax Error.");
  }
  return cAnswer;
}


