#!/bin/bash

if [ -n "$debug" ] ; then
    set -x
fi

SRCDIR="$1"
BASE_OUTPUT="$2"

pushd "$SRCDIR" > /dev/null


function header
{
    local title="$1"
    local breadcrumb="$2"
    local output="$3"

    cat - > $output <<EOF
<!DOCTYPE HTML PUBLIC "-//IETF//DTD HTML//EN">
<html>
<head>
        <title>$title</title>

        <style>
        * { margin: 0; padding: 0; }
        body { font-family: sans-serif; font-size: 12px; }
        #head { padding: 20px; background: #00A500; }
        #head a { color: #000; }
        #body { padding: 20px; }
        #foot { padding: 10px; font-size: 9px; border-top: 1px #18A303 solid; margin-top: 25px; }
        p { line-height: 1.7em; margin-bottom: 1em; }
        pre { margin-bottom: 0.5em; }
        .multi-col { -moz-column-width: 20em; -webkit-column-width: 20em; -moz-column-gap: 1em; -webkit-column-gap: 1em; }
        h1 { margin-bottom: 0.5em; }
        h2,h3,h4 { margin: 1.3em 0 0.5em 0; }
        ul, ol { margin: 0.5em 1.5em; }
        </style>
</head>
<body>
        <div id="head">
        <h1>${title}</h1>
        <p>${breadcrumb}</p>
        </div>
        <div id="body" style="multi-col">
EOF
}

function footer
{
local output="$1"

  cat - >> $output <<EOF

</div>
<div id="foot">
  <small>
    <p>Generated by Libreoffice CI on $(hostname)</p>
    <p>Last updated:
EOF

date '+%F %T' >> $output
cat - >> $output <<EOF
    </p>
  </small>
</div>
</body>
</html>
EOF

}

function proc_text
{
    # Local links: [[...]]
    # Git links: [git:...]
    # Other remote links: [...]
    # Headings: == bleh ==
    # Paragraphs: \n\n
    sed -re ' s/\[\[([-_a-zA-Z0-9]+)\]\]/<a href="\1.html">\1<\/a>/g' - \
        | sed -re ' s/\[git:([^]]+)\]/<a href="http:\/\/cgit.freedesktop.org\/libreoffice\/core\/tree\/\1">\1<\/a>/g' \
        | sed -re ' s/\[([^]]+)\]/<a href="\1">\1<\/a>/g' \
        | sed -re ' s/====([^=]+)====/<h4>\1<\/h4>/g' \
        | sed -re ' s/===([^=]+)===/<h3>\1<\/h3>/g' \
        | sed -re ' s/==([^=]+)==/<h2>\1<\/h2>/g' \
        | sed -re ':a;N;$!ba;s/\n\n/<\/p><p>/g' \
        | awk 'BEGIN { print "<p>" } { print } END { print "</p>" }'
}

# generate entry page

echo "generating index page"
header "LibreOffice Modules" " " "$BASE_OUTPUT/index.html"

for module_name in *; do
    if [ -d $module_name ]; then
        if [ -f $module_name/readme.txt ] ; then
            cur_file="$module_name/readme.txt"
        elif [ -f $module_name/README ] ; then
            cur_file="$module_name/README"
        fi
        if [ -n "$cur_file" ]; then
            # write index.html entry
            text="<h2><a href=\"${module_name}.html\">${module_name}</a></h2>\n"
            text="${text}$(head -n1 $cur_file | proc_text )"
            echo -e $text >> "$BASE_OUTPUT/index.html"

            # write detailed module content
            header "$module_name" "<a href=\"index.html\">LibreOffice</a> &raquo; ${module_name}" "$BASE_OUTPUT/${module_name}.html"
            text="<p><b>View module in:</b>"
            text="${text} &nbsp; <a href=\"http://cgit.freedesktop.org/libreoffice/core/tree/${module_name}\">cgit</a>"
            if [ -d ./docs/${module_name} ] ; then
                text="${text} &nbsp; <a href=\"${module_name}/html/classes.html\">Doxygen</a>"
            fi
            text="${text} </p><p>&nbsp;</p>"
            echo -e $text >> "$BASE_OUTPUT/${module_name}.html"
            proc_text < $cur_file >> "$BASE_OUTPUT/${module_name}.html"
            footer "$BASE_OUTPUT/${module_name}.html"
        else
            empty_modules[${#empty_modules[*]}]=$module_name
        fi
    fi
done

if [ ${#empty_modules[*]} -gt 0 ]; then
    echo -e "<p>&nbsp;</p><p>READMEs were not available for these modules:</p><ul>\n" >> "$BASE_OUTPUT/index.html"
    for module_name in "${empty_modules[@]}"; do
        echo -e "<li><a href=\"http://cgit.freedesktop.org/libreoffice/core/tree/${module_name}\">${module_name}</a></li>\n" >> "$BASE_OUTPUT/index.html"
    done
    echo -e "</ul>\n" >> "$BASE_OUTPUT/index.html"
fi

footer "$BASE_OUTPUT/index.html"

popd > /dev/null

## done
