// libTorrent - BitTorrent library
// Copyright (C) 2005-2008, Jari Sundell
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// In addition, as a special exception, the copyright holders give
// permission to link the code of portions of this program with the
// OpenSSL library under certain conditions as described in each
// individual source file, and distribute linked combinations
// including the two.
//
// You must obey the GNU General Public License in all respects for
// all of the code used other than OpenSSL.  If you modify file(s)
// with this exception, you may extend this exception to your version
// of the file(s), but you are not obligated to do so.  If you do not
// wish to do so, delete this exception statement from your version.
// If you delete this exception statement from all source files in the
// program, then also delete it here.
//
// Contact:  Jari Sundell <jaris@ifi.uio.no>
//
//           Skomakerveien 33
//           3185 Skoppum, NORWAY

// A simple string with no constructors (i.e. POD) plus a derived
// class with constructors and conversion operators. In most cases,
// SimpleString is the class to use, except in unions where it needs
// to be SimpleStringBase.
//
// For efficient conversion from C string literals, depends on the
// compiler optimizing strlen("literal") to an integer literal.
// Then a comparison with either a C string literal or a SimpleString
// literal is a memcmp call plus (if equal) a comparison of the lengths.

#ifndef LIBTORRENT_SIMPLE_STRING_H
#define LIBTORRENT_SIMPLE_STRING_H

#include <cstring>
#include <memory>
#include <string>
#include <torrent/common.h>

namespace torrent {

// Simple string base class (POD).
struct LIBTORRENT_EXPORT SimpleStringBase {
  int               cmp(const SimpleStringBase& other) const;

  char              operator [] (size_t index) const  { return m_data[index]; }

  const char*       begin() const                     { return m_data; }
  const char*       end() const                       { return m_data + m_length; }

  // NOTE: Unlike std::string, SimpleString's c_str() is NOT guaranteed to be zero-terminated!
  const char*       c_str() const                     { return m_data; }
  const char*       data() const                      { return m_data; }

  bool              empty() const                     { return !m_length; }
  size_t            length() const                    { return m_length; }
  size_t            size() const                      { return m_length; }

  std::string       str() const                                   { return std::string(m_data, m_length); }
  std::string       substr(size_t pos = 0, size_t n = npos) const { return std::string(m_data + pos, std::min(m_length - pos, n)); }

  // Allocates a copy of the string and returns it.
  SimpleStringBase  copy() const;

  static const size_t npos = static_cast<size_t>(-1);

protected:
  const char* m_data;
  size_t      m_length;
};

// Conversion helper class, we don't want constructors
// in the base class to be able to put it in a union.
struct LIBTORRENT_EXPORT SimpleString : public SimpleStringBase {
  typedef SimpleStringBase base_type;

  SimpleString()                        { m_data = ""; m_length = 0; }
  SimpleString(const base_type& s)      { m_data = s.c_str(); m_length = s.length(); }
  SimpleString(const std::string& s)    { m_data = s.c_str(); m_length = s.length(); }
  SimpleString(const char* s)           { m_data = s; m_length = strlen(s); }
  SimpleString(const char* s, size_t l) { m_data = s; m_length = l; }
};

inline int
SimpleStringBase::cmp(const SimpleStringBase& other) const {
  int cmp = memcmp(m_data, other.m_data, std::min(m_length, other.m_length));
  return cmp ? cmp : m_length - other.m_length;
}

inline SimpleStringBase
SimpleStringBase::copy() const {
  char* data = new char[m_length + 1];
  memcpy(data, m_data, m_length);
  data[m_length] = 0;
  return SimpleString(data, m_length);
}

inline bool operator == (const SimpleStringBase& one, const SimpleStringBase& other) { return one.cmp(other) == 0; }
inline bool operator != (const SimpleStringBase& one, const SimpleStringBase& other) { return one.cmp(other) != 0; }
inline bool operator <= (const SimpleStringBase& one, const SimpleStringBase& other) { return one.cmp(other) <= 0; }
inline bool operator <  (const SimpleStringBase& one, const SimpleStringBase& other) { return one.cmp(other) <  0; }
inline bool operator >= (const SimpleStringBase& one, const SimpleStringBase& other) { return one.cmp(other) >= 0; }
inline bool operator >  (const SimpleStringBase& one, const SimpleStringBase& other) { return one.cmp(other) >  0; }

inline bool operator == (const SimpleStringBase& one, const char* other) { return one.cmp(SimpleString(other)) == 0; }
inline bool operator != (const SimpleStringBase& one, const char* other) { return one.cmp(SimpleString(other)) != 0; }
inline bool operator <= (const SimpleStringBase& one, const char* other) { return one.cmp(SimpleString(other)) <= 0; }
inline bool operator <  (const SimpleStringBase& one, const char* other) { return one.cmp(SimpleString(other)) <  0; }
inline bool operator >= (const SimpleStringBase& one, const char* other) { return one.cmp(SimpleString(other)) >= 0; }
inline bool operator >  (const SimpleStringBase& one, const char* other) { return one.cmp(SimpleString(other)) >  0; }

}

#endif
