/*
 * Decompiled with CFR 0.152.
 */
package beast.evolution.branchratemodel;

import beast.core.Citation;
import beast.core.Description;
import beast.core.Input;
import beast.core.parameter.IntegerParameter;
import beast.core.parameter.RealParameter;
import beast.core.util.Log;
import beast.evolution.branchratemodel.BranchRateModel;
import beast.evolution.tree.Node;
import beast.evolution.tree.Tree;
import beast.math.distributions.ParametricDistribution;
import beast.util.Randomizer;
import java.util.Arrays;
import org.apache.commons.math.MathException;

@Description(value="Defines an uncorrelated relaxed molecular clock.")
@Citation(value="Drummond AJ, Ho SYW, Phillips MJ, Rambaut A (2006) Relaxed Phylogenetics and\n  Dating with Confidence. PLoS Biol 4(5): e88", DOI="10.1371/journal.pbio.0040088", year=2006, firstAuthorSurname="drummond")
public class UCRelaxedClockModel
extends BranchRateModel.Base {
    public final Input<ParametricDistribution> rateDistInput = new Input("distr", "the distribution governing the rates among branches. Must have mean of 1. The clock.rate parameter can be used to change the mean rate.", Input.Validate.REQUIRED);
    public final Input<IntegerParameter> categoryInput = new Input("rateCategories", "the rate categories associated with nodes in the tree for sampling of individual rates among branches.", Input.Validate.REQUIRED);
    public final Input<Integer> numberOfDiscreteRates = new Input<Integer>("numberOfDiscreteRates", "the number of discrete rate categories to approximate the rate distribution by. A value <= 0 will cause the number of categories to be set equal to the number of branches in the tree. (default = -1)", -1);
    public final Input<RealParameter> quantileInput = new Input("rateQuantiles", "the rate quantiles associated with nodes in the tree for sampling of individual rates among branches.", Input.Validate.XOR, this.categoryInput);
    public final Input<Tree> treeInput = new Input("tree", "the tree this relaxed clock is associated with.", Input.Validate.REQUIRED);
    public final Input<Boolean> normalizeInput = new Input<Boolean>("normalize", "Whether to normalize the average rate (default false).", false);
    RealParameter meanRate;
    int LATTICE_SIZE_FOR_DISCRETIZED_RATES = 100;
    boolean usingQuantiles;
    private int branchCount;
    ParametricDistribution distribution;
    IntegerParameter categories;
    RealParameter quantiles;
    Tree tree;
    private boolean normalize = false;
    private boolean recompute = true;
    private boolean renormalize = true;
    private double[] rates;
    private double[] storedRates;
    private double scaleFactor = 1.0;
    private double storedScaleFactor = 1.0;

    @Override
    public void initAndValidate() {
        Number[] numberArray;
        this.tree = this.treeInput.get();
        this.branchCount = this.tree.getNodeCount() - 1;
        this.categories = this.categoryInput.get();
        boolean bl = this.usingQuantiles = this.categories == null;
        if (!this.usingQuantiles) {
            this.LATTICE_SIZE_FOR_DISCRETIZED_RATES = this.numberOfDiscreteRates.get();
            if (this.LATTICE_SIZE_FOR_DISCRETIZED_RATES <= 0) {
                this.LATTICE_SIZE_FOR_DISCRETIZED_RATES = this.branchCount;
            }
            Log.info.println("  UCRelaxedClockModel: using " + this.LATTICE_SIZE_FOR_DISCRETIZED_RATES + " rate " + "categories to approximate rate distribution across branches.");
        } else {
            if (this.numberOfDiscreteRates.get() != -1) {
                throw new RuntimeException("Can't specify both numberOfDiscreteRates and rateQuantiles inputs.");
            }
            Log.info.println("  UCRelaxedClockModel: using quantiles for rate distribution across branches.");
        }
        if (this.usingQuantiles) {
            this.quantiles = this.quantileInput.get();
            this.quantiles.setDimension(this.branchCount);
            numberArray = new Double[this.branchCount];
            for (int i = 0; i < this.branchCount; ++i) {
                numberArray[i] = Randomizer.nextDouble();
            }
            RealParameter realParameter = new RealParameter((Double[])numberArray);
            this.quantiles.assignFromWithoutID(realParameter);
            this.quantiles.setLower(0.0);
            this.quantiles.setUpper(1.0);
        } else {
            this.categories.setDimension(this.branchCount);
            numberArray = new Integer[this.branchCount];
            for (int i = 0; i < this.branchCount; ++i) {
                numberArray[i] = Randomizer.nextInt(this.LATTICE_SIZE_FOR_DISCRETIZED_RATES);
            }
            IntegerParameter integerParameter = new IntegerParameter((Integer[])numberArray);
            this.categories.assignFromWithoutID(integerParameter);
            this.categories.setLower(0);
            this.categories.setUpper(this.LATTICE_SIZE_FOR_DISCRETIZED_RATES - 1);
        }
        this.distribution = this.rateDistInput.get();
        if (!this.usingQuantiles) {
            this.rates = new double[this.LATTICE_SIZE_FOR_DISCRETIZED_RATES];
            this.storedRates = new double[this.LATTICE_SIZE_FOR_DISCRETIZED_RATES];
        }
        this.normalize = this.normalizeInput.get();
        this.meanRate = (RealParameter)this.meanRateInput.get();
        if (this.meanRate == null) {
            this.meanRate = new RealParameter("1.0");
        }
        try {
            double d = this.rateDistInput.get().getMean();
            if (Math.abs(d - 1.0) > 1.0E-6) {
                Log.warning.println("WARNING: mean of distribution for relaxed clock model is not 1.0.");
            }
        }
        catch (RuntimeException runtimeException) {
            // empty catch block
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public double getRateForBranch(Node node) {
        UCRelaxedClockModel uCRelaxedClockModel;
        if (node.isRoot()) {
            return 1.0;
        }
        if (this.recompute) {
            uCRelaxedClockModel = this;
            synchronized (uCRelaxedClockModel) {
                this.prepare();
                this.recompute = false;
            }
        }
        if (this.renormalize) {
            if (this.normalize) {
                uCRelaxedClockModel = this;
                synchronized (uCRelaxedClockModel) {
                    this.computeFactor();
                }
            }
            this.renormalize = false;
        }
        return this.getRawRate(node) * this.scaleFactor * this.meanRate.getValue();
    }

    private void computeFactor() {
        double d = 0.0;
        double d2 = 0.0;
        if (!this.usingQuantiles) {
            for (int i = 0; i < this.tree.getNodeCount(); ++i) {
                Node node = this.tree.getNode(i);
                if (node.isRoot()) continue;
                d += this.getRawRateForCategory(node) * node.getLength();
                d2 += node.getLength();
            }
        } else {
            for (int i = 0; i < this.tree.getNodeCount(); ++i) {
                Node node = this.tree.getNode(i);
                if (node.isRoot()) continue;
                d += this.getRawRateForQuantile(node) * node.getLength();
                d2 += node.getLength();
            }
        }
        this.scaleFactor = 1.0 / (d / d2);
    }

    private double getRawRate(Node node) {
        if (this.usingQuantiles) {
            return this.getRawRateForQuantile(node);
        }
        return this.getRawRateForCategory(node);
    }

    private double getRawRateForCategory(Node node) {
        int n;
        int n2 = node.getNr();
        if (n2 == this.branchCount) {
            n2 = node.getTree().getRoot().getNr();
        }
        if (this.rates[n = ((Integer)this.categories.getValue(n2)).intValue()] == 0.0) {
            try {
                this.rates[n] = this.distribution.inverseCumulativeProbability(((double)n + 0.5) / (double)this.rates.length);
            }
            catch (MathException mathException) {
                throw new RuntimeException("Failed to compute inverse cumulative probability!");
            }
        }
        return this.rates[n];
    }

    private double getRawRateForQuantile(Node node) {
        int n = node.getNr();
        if (n == this.branchCount) {
            n = node.getTree().getRoot().getNr();
        }
        try {
            return this.distribution.inverseCumulativeProbability((Double)this.quantiles.getValue(n));
        }
        catch (MathException mathException) {
            throw new RuntimeException("Failed to compute inverse cumulative probability!");
        }
    }

    private void prepare() {
        this.categories = this.categoryInput.get();
        this.usingQuantiles = this.categories == null;
        this.distribution = this.rateDistInput.get();
        this.tree = this.treeInput.get();
        if (!this.usingQuantiles) {
            Arrays.fill(this.rates, 0.0);
        }
    }

    @Override
    protected boolean requiresRecalculation() {
        this.recompute = false;
        this.renormalize = true;
        if (this.rateDistInput.get().isDirtyCalculation()) {
            this.recompute = true;
            return true;
        }
        if (this.categoryInput.get() != null && this.categoryInput.get().somethingIsDirty()) {
            return true;
        }
        if (this.quantileInput.get() != null && this.quantileInput.get().somethingIsDirty()) {
            return true;
        }
        if (this.meanRate.somethingIsDirty()) {
            return true;
        }
        return this.recompute;
    }

    @Override
    public void store() {
        if (!this.usingQuantiles) {
            System.arraycopy(this.rates, 0, this.storedRates, 0, this.rates.length);
        }
        this.storedScaleFactor = this.scaleFactor;
        super.store();
    }

    @Override
    public void restore() {
        if (!this.usingQuantiles) {
            double[] dArray = this.rates;
            this.rates = this.storedRates;
            this.storedRates = dArray;
        }
        this.scaleFactor = this.storedScaleFactor;
        super.restore();
    }
}

