/*
 * Copyright (C) 2021 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1.  Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 * 2.  Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "WebCookieManager.h"

#include "ArgumentCoders.h"
#include "Decoder.h"
#include "HandleMessage.h"
#if USE(SOUP)
#include "SoupCookiePersistentStorageType.h"
#endif
#include "WebCookieManagerMessages.h"
#include "WebCoreArgumentCoders.h"
#include <WebCore/Cookie.h>
#include <WebCore/HTTPCookieAcceptPolicy.h>
#include <pal/SessionID.h>
#include <wtf/URLHash.h>
#include <wtf/Vector.h>
#include <wtf/WallTime.h>
#include <wtf/text/WTFString.h>

namespace Messages {

namespace WebCookieManager {

void GetHostnamesWithCookies::callReply(IPC::Decoder& decoder, CompletionHandler<void(Vector<String>&&)>&& completionHandler)
{
    std::optional<Vector<String>> hostnames;
    decoder >> hostnames;
    if (!hostnames) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*hostnames));
}

void GetHostnamesWithCookies::cancelReply(CompletionHandler<void(Vector<String>&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<Vector<String>>::create());
}

void GetHostnamesWithCookies::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Vector<String>& hostnames)
{
    encoder.get() << hostnames;
    connection.sendSyncReply(WTFMove(encoder));
}

void SetCookie::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetCookie::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetCookie::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

void SetCookies::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetCookies::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetCookies::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

void GetAllCookies::callReply(IPC::Decoder& decoder, CompletionHandler<void(Vector<WebCore::Cookie>&&)>&& completionHandler)
{
    std::optional<Vector<WebCore::Cookie>> cookies;
    decoder >> cookies;
    if (!cookies) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*cookies));
}

void GetAllCookies::cancelReply(CompletionHandler<void(Vector<WebCore::Cookie>&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<Vector<WebCore::Cookie>>::create());
}

void GetAllCookies::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Vector<WebCore::Cookie>& cookies)
{
    encoder.get() << cookies;
    connection.sendSyncReply(WTFMove(encoder));
}

void GetCookies::callReply(IPC::Decoder& decoder, CompletionHandler<void(Vector<WebCore::Cookie>&&)>&& completionHandler)
{
    std::optional<Vector<WebCore::Cookie>> cookies;
    decoder >> cookies;
    if (!cookies) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*cookies));
}

void GetCookies::cancelReply(CompletionHandler<void(Vector<WebCore::Cookie>&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<Vector<WebCore::Cookie>>::create());
}

void GetCookies::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Vector<WebCore::Cookie>& cookies)
{
    encoder.get() << cookies;
    connection.sendSyncReply(WTFMove(encoder));
}

void DeleteCookie::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void DeleteCookie::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void DeleteCookie::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

void DeleteAllCookiesModifiedSince::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void DeleteAllCookiesModifiedSince::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void DeleteAllCookiesModifiedSince::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

void SetHTTPCookieAcceptPolicy::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetHTTPCookieAcceptPolicy::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetHTTPCookieAcceptPolicy::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

void GetHTTPCookieAcceptPolicy::callReply(IPC::Decoder& decoder, CompletionHandler<void(WebCore::HTTPCookieAcceptPolicy&&)>&& completionHandler)
{
    std::optional<WebCore::HTTPCookieAcceptPolicy> policy;
    decoder >> policy;
    if (!policy) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*policy));
}

void GetHTTPCookieAcceptPolicy::cancelReply(CompletionHandler<void(WebCore::HTTPCookieAcceptPolicy&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<WebCore::HTTPCookieAcceptPolicy>::create());
}

void GetHTTPCookieAcceptPolicy::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, WebCore::HTTPCookieAcceptPolicy policy)
{
    encoder.get() << policy;
    connection.sendSyncReply(WTFMove(encoder));
}

} // namespace WebCookieManager

} // namespace Messages

namespace WebKit {

void WebCookieManager::didReceiveMessage(IPC::Connection& connection, IPC::Decoder& decoder)
{
    if (decoder.messageName() == Messages::WebCookieManager::GetHostnamesWithCookies::name())
        return IPC::handleMessageAsync<Messages::WebCookieManager::GetHostnamesWithCookies>(connection, decoder, this, &WebCookieManager::getHostnamesWithCookies);
    if (decoder.messageName() == Messages::WebCookieManager::DeleteCookiesForHostnames::name())
        return IPC::handleMessage<Messages::WebCookieManager::DeleteCookiesForHostnames>(connection, decoder, this, &WebCookieManager::deleteCookiesForHostnames);
    if (decoder.messageName() == Messages::WebCookieManager::DeleteAllCookies::name())
        return IPC::handleMessage<Messages::WebCookieManager::DeleteAllCookies>(connection, decoder, this, &WebCookieManager::deleteAllCookies);
    if (decoder.messageName() == Messages::WebCookieManager::SetCookie::name())
        return IPC::handleMessageAsync<Messages::WebCookieManager::SetCookie>(connection, decoder, this, &WebCookieManager::setCookie);
    if (decoder.messageName() == Messages::WebCookieManager::SetCookies::name())
        return IPC::handleMessageAsync<Messages::WebCookieManager::SetCookies>(connection, decoder, this, &WebCookieManager::setCookies);
    if (decoder.messageName() == Messages::WebCookieManager::GetAllCookies::name())
        return IPC::handleMessageAsync<Messages::WebCookieManager::GetAllCookies>(connection, decoder, this, &WebCookieManager::getAllCookies);
    if (decoder.messageName() == Messages::WebCookieManager::GetCookies::name())
        return IPC::handleMessageAsync<Messages::WebCookieManager::GetCookies>(connection, decoder, this, &WebCookieManager::getCookies);
    if (decoder.messageName() == Messages::WebCookieManager::DeleteCookie::name())
        return IPC::handleMessageAsync<Messages::WebCookieManager::DeleteCookie>(connection, decoder, this, &WebCookieManager::deleteCookie);
    if (decoder.messageName() == Messages::WebCookieManager::DeleteAllCookiesModifiedSince::name())
        return IPC::handleMessageAsync<Messages::WebCookieManager::DeleteAllCookiesModifiedSince>(connection, decoder, this, &WebCookieManager::deleteAllCookiesModifiedSince);
    if (decoder.messageName() == Messages::WebCookieManager::SetHTTPCookieAcceptPolicy::name())
        return IPC::handleMessageAsync<Messages::WebCookieManager::SetHTTPCookieAcceptPolicy>(connection, decoder, this, &WebCookieManager::setHTTPCookieAcceptPolicy);
    if (decoder.messageName() == Messages::WebCookieManager::GetHTTPCookieAcceptPolicy::name())
        return IPC::handleMessageAsync<Messages::WebCookieManager::GetHTTPCookieAcceptPolicy>(connection, decoder, this, &WebCookieManager::getHTTPCookieAcceptPolicy);
    if (decoder.messageName() == Messages::WebCookieManager::StartObservingCookieChanges::name())
        return IPC::handleMessage<Messages::WebCookieManager::StartObservingCookieChanges>(connection, decoder, this, &WebCookieManager::startObservingCookieChanges);
    if (decoder.messageName() == Messages::WebCookieManager::StopObservingCookieChanges::name())
        return IPC::handleMessage<Messages::WebCookieManager::StopObservingCookieChanges>(connection, decoder, this, &WebCookieManager::stopObservingCookieChanges);
#if USE(SOUP)
    if (decoder.messageName() == Messages::WebCookieManager::SetCookiePersistentStorage::name())
        return IPC::handleMessage<Messages::WebCookieManager::SetCookiePersistentStorage>(connection, decoder, this, &WebCookieManager::setCookiePersistentStorage);
#endif
    UNUSED_PARAM(connection);
    UNUSED_PARAM(decoder);
#if ENABLE(IPC_TESTING_API)
    if (connection.ignoreInvalidMessageForTesting())
        return;
#endif // ENABLE(IPC_TESTING_API)
    ASSERT_NOT_REACHED_WITH_MESSAGE("Unhandled message %s to %" PRIu64, IPC::description(decoder.messageName()), decoder.destinationID());
}

} // namespace WebKit
