#!/bin/sh
set -eux
IMG_PATH="$1"
TAG=$(basename "$IMG_PATH" | sed -nr 's/tails-amd64-([0-9a-z~\.]+)\.[^\]+/\1/p' | sed 's/~/-/g')
REPO_URL='https://gitlab.tails.boum.org/tails/tails.git'

export TAILS_BUILD_OPTIONS="nomergebasebranch forcecleanup"

# cleanup possible leftovers
virsh vol-list default | awk '{print $1}' | grep ^tails-builder- | xargs -rL1 virsh vol-delete --pool default

# setup temporary directory
WORK_DIR=$(mktemp -d -t tails.XXXXXX)
trap '{ rm -rf -- "$WORK_DIR"; }' EXIT
# set the folder public so libvirt user can access it
chmod 0711 "$WORK_DIR"

# import gpg keys to authenticate source code
export HOME="$WORK_DIR/home"
mkdir -m 0700 -- "$HOME"
# https://tails.boum.org/tails-signing.key
gpg --import <<EOF
-----BEGIN PGP PUBLIC KEY BLOCK-----
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=jIrU
-----END PGP PUBLIC KEY BLOCK-----
EOF

# clone repo
REPO_DEST="$WORK_DIR/tails"
# doesn't work even with nomergebasebranch
#git clone --depth=1 --branch "$TAG" -- "$REPO_URL" "$REPO_DEST"
git clone --branch "$TAG" -- "$REPO_URL" "$REPO_DEST"
cd "$REPO_DEST"
git verify-tag -v -- "$TAG"
git submodule update --init

# read and export SOURCE_DATE_EPOCH to normalize the build time
SOURCE_DATE_EPOCH=$(date --utc --date="$(dpkg-parsechangelog --show-field=Date)" '+%s')
export SOURCE_DATE_EPOCH

# build the image
# libvirtd needs to be started, /var/run/libvirt needs to be mounted if run inside a container
ARTIFACTS="$REBUILDERD_OUTDIR" rake build

# list build outputs
ls -la "$REBUILDERD_OUTDIR"
