"use strict";
var __read = (this && this.__read) || function (o, n) {
    var m = typeof Symbol === "function" && o[Symbol.iterator];
    if (!m) return o;
    var i = m.call(o), r, ar = [], e;
    try {
        while ((n === void 0 || n-- > 0) && !(r = i.next()).done) ar.push(r.value);
    }
    catch (error) { e = { error: error }; }
    finally {
        try {
            if (r && !r.done && (m = i["return"])) m.call(i);
        }
        finally { if (e) throw e.error; }
    }
    return ar;
};
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
var d3_array_1 = require("d3-array");
var d3_scale_1 = require("d3-scale");
var commons_1 = require("../utils/commons");
var date_time_1 = require("../utils/data/date_time");
var constants_1 = require("./constants");
var SCALES = (_a = {},
    _a[constants_1.ScaleType.Linear] = d3_scale_1.scaleLinear,
    _a[constants_1.ScaleType.Log] = d3_scale_1.scaleLog,
    _a[constants_1.ScaleType.Sqrt] = d3_scale_1.scaleSqrt,
    _a[constants_1.ScaleType.Time] = d3_scale_1.scaleUtc,
    _a);
function limitLogScaleDomain(domain) {
    if (domain[0] === 0) {
        if (domain[1] > 0) {
            return [constants_1.LOG_MIN_ABS_DOMAIN, domain[1]];
        }
        if (domain[1] < 0) {
            return [-constants_1.LOG_MIN_ABS_DOMAIN, domain[1]];
        }
        return [constants_1.LOG_MIN_ABS_DOMAIN, constants_1.LOG_MIN_ABS_DOMAIN];
    }
    if (domain[1] === 0) {
        if (domain[0] > 0) {
            return [domain[0], constants_1.LOG_MIN_ABS_DOMAIN];
        }
        if (domain[0] < 0) {
            return [domain[0], -constants_1.LOG_MIN_ABS_DOMAIN];
        }
        return [constants_1.LOG_MIN_ABS_DOMAIN, constants_1.LOG_MIN_ABS_DOMAIN];
    }
    if (domain[0] < 0 && domain[1] > 0) {
        var isD0Min = Math.abs(domain[1]) - Math.abs(domain[0]) >= 0;
        if (isD0Min) {
            return [constants_1.LOG_MIN_ABS_DOMAIN, domain[1]];
        }
        return [domain[0], -constants_1.LOG_MIN_ABS_DOMAIN];
    }
    if (domain[0] > 0 && domain[1] < 0) {
        var isD0Max = Math.abs(domain[0]) - Math.abs(domain[1]) >= 0;
        if (isD0Max) {
            return [domain[0], constants_1.LOG_MIN_ABS_DOMAIN];
        }
        return [-constants_1.LOG_MIN_ABS_DOMAIN, domain[1]];
    }
    return domain;
}
exports.limitLogScaleDomain = limitLogScaleDomain;
var defaultScaleOptions = {
    bandwidth: 0,
    minInterval: 0,
    timeZone: 'utc',
    totalBarsInCluster: 1,
    barsPadding: 0,
    ticks: 10,
    isSingleValueHistogram: false,
    integersOnly: false,
};
var ScaleContinuous = (function () {
    function ScaleContinuous(scaleData, options) {
        var _this = this;
        var type = scaleData.type, domain = scaleData.domain, range = scaleData.range;
        var _a = commons_1.mergePartial(defaultScaleOptions, options), bandwidth = _a.bandwidth, minInterval = _a.minInterval, timeZone = _a.timeZone, totalBarsInCluster = _a.totalBarsInCluster, barsPadding = _a.barsPadding, ticks = _a.ticks, isSingleValueHistogram = _a.isSingleValueHistogram, integersOnly = _a.integersOnly;
        this.d3Scale = SCALES[type]();
        var cleanDomain = type === constants_1.ScaleType.Log ? limitLogScaleDomain(domain) : domain;
        this.domain = cleanDomain;
        this.d3Scale.domain(cleanDomain);
        var safeBarPadding = commons_1.maxValueWithUpperLimit(barsPadding, 0, 1);
        this.barsPadding = safeBarPadding;
        this.bandwidth = bandwidth * (1 - safeBarPadding);
        this.bandwidthPadding = bandwidth * safeBarPadding;
        this.d3Scale.range(range);
        this.step = this.bandwidth + this.barsPadding + this.bandwidthPadding;
        this.type = type;
        this.range = range;
        this.minInterval = minInterval;
        this.isInverted = this.domain[0] > this.domain[1];
        this.timeZone = timeZone;
        this.totalBarsInCluster = totalBarsInCluster;
        this.isSingleValueHistogram = isSingleValueHistogram;
        if (type === constants_1.ScaleType.Time) {
            var startDomain = date_time_1.getMomentWithTz(this.domain[0], this.timeZone);
            var endDomain = date_time_1.getMomentWithTz(this.domain[1], this.timeZone);
            var offset_1 = startDomain.utcOffset();
            var shiftedDomainMin = startDomain.add(offset_1, 'minutes').valueOf();
            var shiftedDomainMax = endDomain.add(offset_1, 'minutes').valueOf();
            var tzShiftedScale = d3_scale_1.scaleUtc().domain([shiftedDomainMin, shiftedDomainMax]);
            var rawTicks = tzShiftedScale.ticks(ticks);
            var timePerTick = (shiftedDomainMax - shiftedDomainMin) / rawTicks.length;
            var hasHourTicks_1 = timePerTick < 1000 * 60 * 60 * 12;
            this.tickValues = rawTicks.map(function (d) {
                var currentDateTime = date_time_1.getMomentWithTz(d, _this.timeZone);
                var currentOffset = hasHourTicks_1 ? offset_1 : currentDateTime.utcOffset();
                return currentDateTime.subtract(currentOffset, 'minutes').valueOf();
            });
        }
        else {
            if (minInterval > 0 && bandwidth > 0) {
                var intervalCount = Math.floor((this.domain[1] - this.domain[0]) / this.minInterval);
                this.tickValues = new Array(intervalCount + 1).fill(0).map(function (_, i) { return _this.domain[0] + i * _this.minInterval; });
            }
            else {
                this.tickValues = this.getTicks(ticks, integersOnly);
            }
        }
    }
    ScaleContinuous.prototype.getScaledValue = function (value) {
        if (typeof value !== 'number' || isNaN(value)) {
            return null;
        }
        var scaledValue = this.d3Scale(value);
        return isNaN(scaledValue) ? null : scaledValue;
    };
    ScaleContinuous.prototype.getTicks = function (ticks, integersOnly) {
        return integersOnly
            ? this.d3Scale
                .ticks(ticks)
                .filter(function (item) { return typeof item === 'number' && item % 1 === 0; })
                .map(function (item) { return parseInt(item.toFixed(0), 10); })
            : this.d3Scale.ticks(ticks);
    };
    ScaleContinuous.prototype.scaleOrThrow = function (value) {
        var scaleValue = this.scale(value);
        if (scaleValue === null) {
            throw new Error("Unable to scale value: " + scaleValue + ")");
        }
        return scaleValue;
    };
    ScaleContinuous.prototype.scale = function (value) {
        var scaledValue = this.getScaledValue(value);
        return scaledValue === null ? null : scaledValue + (this.bandwidthPadding / 2) * this.totalBarsInCluster;
    };
    ScaleContinuous.prototype.pureScale = function (value) {
        if (this.bandwidth === 0) {
            return this.getScaledValue(value);
        }
        if (typeof value !== 'number' || isNaN(value)) {
            return null;
        }
        return this.getScaledValue(value + this.minInterval / 2);
    };
    ScaleContinuous.prototype.ticks = function () {
        return this.tickValues;
    };
    ScaleContinuous.prototype.invert = function (value) {
        var invertedValue = this.d3Scale.invert(value);
        if (this.type === constants_1.ScaleType.Time) {
            invertedValue = date_time_1.getMomentWithTz(invertedValue, this.timeZone).valueOf();
        }
        return invertedValue;
    };
    ScaleContinuous.prototype.invertWithStep = function (value, data) {
        if (data.length === 0) {
            return null;
        }
        var invertedValue = this.invert(value);
        var bisectValue = this.bandwidth === 0 ? invertedValue + this.minInterval / 2 : invertedValue;
        var leftIndex = d3_array_1.bisectLeft(data, bisectValue);
        if (leftIndex === 0) {
            if (invertedValue < data[0]) {
                return {
                    value: data[0] - this.minInterval * Math.ceil((data[0] - invertedValue) / this.minInterval),
                    withinBandwidth: false,
                };
            }
            return {
                value: data[0],
                withinBandwidth: true,
            };
        }
        var currentValue = data[leftIndex - 1];
        if (this.minInterval === 0) {
            var nextValue = data[leftIndex];
            var nextDiff = Math.abs(nextValue - invertedValue);
            var prevDiff = Math.abs(invertedValue - currentValue);
            return {
                value: nextDiff <= prevDiff ? nextValue : currentValue,
                withinBandwidth: true,
            };
        }
        if (invertedValue - currentValue <= this.minInterval) {
            return {
                value: currentValue,
                withinBandwidth: true,
            };
        }
        return {
            value: currentValue + this.minInterval * Math.floor((invertedValue - currentValue) / this.minInterval),
            withinBandwidth: false,
        };
    };
    ScaleContinuous.prototype.isSingleValue = function () {
        if (this.isSingleValueHistogram) {
            return true;
        }
        if (this.domain.length < 2) {
            return true;
        }
        var min = this.domain[0];
        var max = this.domain[this.domain.length - 1];
        return max === min;
    };
    ScaleContinuous.prototype.isValueInDomain = function (value) {
        return value >= this.domain[0] && value <= this.domain[1];
    };
    return ScaleContinuous;
}());
exports.ScaleContinuous = ScaleContinuous;
function getDomainPolarity(domain) {
    var _a = __read(domain, 2), min = _a[0], max = _a[1];
    if (min >= 0 && max >= 0) {
        return 1;
    }
    if (min <= 0 && max <= 0) {
        return -1;
    }
    return 0;
}
exports.getDomainPolarity = getDomainPolarity;
//# sourceMappingURL=scale_continuous.js.map