// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (C) 2017-2018, Bootlin
 */

#include <linux/delay.h>
#include <linux/device.h>
#include <linux/err.h>
#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/of.h>

#include <linux/gpio/consumer.h>
#include <linux/regulator/consumer.h>

#include <drm/drm_mipi_dsi.h>
#include <drm/drm_modes.h>
#include <drm/drm_panel.h>

#include <video/mipi_display.h>

enum ili9881c_op {
	ILI9881C_SWITCH_PAGE,
	ILI9881C_COMMAND,
};

struct ili9881c_instr {
	enum ili9881c_op	op;

	union arg {
		struct cmd {
			u8	cmd;
			u8	data;
		} cmd;
		u8	page;
	} arg;
};

struct ili9881c_desc {
	const struct ili9881c_instr *init;
	const size_t init_length;
	const struct drm_display_mode *mode;
	const unsigned long mode_flags;
	u8 default_address_mode;
	unsigned int lanes;
};

struct ili9881c {
	struct drm_panel	panel;
	struct mipi_dsi_device	*dsi;
	const struct ili9881c_desc	*desc;

	struct regulator	*power;
	struct gpio_desc	*reset;

	enum drm_panel_orientation	orientation;
	u8 address_mode;
};

#define ILI9881C_SWITCH_PAGE_INSTR(_page)	\
	{					\
		.op = ILI9881C_SWITCH_PAGE,	\
		.arg = {			\
			.page = (_page),	\
		},				\
	}

#define ILI9881C_COMMAND_INSTR(_cmd, _data)		\
	{						\
		.op = ILI9881C_COMMAND,		\
		.arg = {				\
			.cmd = {			\
				.cmd = (_cmd),		\
				.data = (_data),	\
			},				\
		},					\
	}

static const struct ili9881c_instr lhr050h41_init[] = {
	ILI9881C_SWITCH_PAGE_INSTR(3),
	ILI9881C_COMMAND_INSTR(0x01, 0x00),
	ILI9881C_COMMAND_INSTR(0x02, 0x00),
	ILI9881C_COMMAND_INSTR(0x03, 0x73),
	ILI9881C_COMMAND_INSTR(0x04, 0x03),
	ILI9881C_COMMAND_INSTR(0x05, 0x00),
	ILI9881C_COMMAND_INSTR(0x06, 0x06),
	ILI9881C_COMMAND_INSTR(0x07, 0x06),
	ILI9881C_COMMAND_INSTR(0x08, 0x00),
	ILI9881C_COMMAND_INSTR(0x09, 0x18),
	ILI9881C_COMMAND_INSTR(0x0a, 0x04),
	ILI9881C_COMMAND_INSTR(0x0b, 0x00),
	ILI9881C_COMMAND_INSTR(0x0c, 0x02),
	ILI9881C_COMMAND_INSTR(0x0d, 0x03),
	ILI9881C_COMMAND_INSTR(0x0e, 0x00),
	ILI9881C_COMMAND_INSTR(0x0f, 0x25),
	ILI9881C_COMMAND_INSTR(0x10, 0x25),
	ILI9881C_COMMAND_INSTR(0x11, 0x00),
	ILI9881C_COMMAND_INSTR(0x12, 0x00),
	ILI9881C_COMMAND_INSTR(0x13, 0x00),
	ILI9881C_COMMAND_INSTR(0x14, 0x00),
	ILI9881C_COMMAND_INSTR(0x15, 0x00),
	ILI9881C_COMMAND_INSTR(0x16, 0x0c),
	ILI9881C_COMMAND_INSTR(0x17, 0x00),
	ILI9881C_COMMAND_INSTR(0x18, 0x00),
	ILI9881C_COMMAND_INSTR(0x19, 0x00),
	ILI9881C_COMMAND_INSTR(0x1a, 0x00),
	ILI9881C_COMMAND_INSTR(0x1b, 0x00),
	ILI9881C_COMMAND_INSTR(0x1c, 0x00),
	ILI9881C_COMMAND_INSTR(0x1d, 0x00),
	ILI9881C_COMMAND_INSTR(0x1e, 0xc0),
	ILI9881C_COMMAND_INSTR(0x1f, 0x80),
	ILI9881C_COMMAND_INSTR(0x20, 0x04),
	ILI9881C_COMMAND_INSTR(0x21, 0x01),
	ILI9881C_COMMAND_INSTR(0x22, 0x00),
	ILI9881C_COMMAND_INSTR(0x23, 0x00),
	ILI9881C_COMMAND_INSTR(0x24, 0x00),
	ILI9881C_COMMAND_INSTR(0x25, 0x00),
	ILI9881C_COMMAND_INSTR(0x26, 0x00),
	ILI9881C_COMMAND_INSTR(0x27, 0x00),
	ILI9881C_COMMAND_INSTR(0x28, 0x33),
	ILI9881C_COMMAND_INSTR(0x29, 0x03),
	ILI9881C_COMMAND_INSTR(0x2a, 0x00),
	ILI9881C_COMMAND_INSTR(0x2b, 0x00),
	ILI9881C_COMMAND_INSTR(0x2c, 0x00),
	ILI9881C_COMMAND_INSTR(0x2d, 0x00),
	ILI9881C_COMMAND_INSTR(0x2e, 0x00),
	ILI9881C_COMMAND_INSTR(0x2f, 0x00),
	ILI9881C_COMMAND_INSTR(0x30, 0x00),
	ILI9881C_COMMAND_INSTR(0x31, 0x00),
	ILI9881C_COMMAND_INSTR(0x32, 0x00),
	ILI9881C_COMMAND_INSTR(0x33, 0x00),
	ILI9881C_COMMAND_INSTR(0x34, 0x04),
	ILI9881C_COMMAND_INSTR(0x35, 0x00),
	ILI9881C_COMMAND_INSTR(0x36, 0x00),
	ILI9881C_COMMAND_INSTR(0x37, 0x00),
	ILI9881C_COMMAND_INSTR(0x38, 0x3c),
	ILI9881C_COMMAND_INSTR(0x39, 0x00),
	ILI9881C_COMMAND_INSTR(0x3a, 0x00),
	ILI9881C_COMMAND_INSTR(0x3b, 0x00),
	ILI9881C_COMMAND_INSTR(0x3c, 0x00),
	ILI9881C_COMMAND_INSTR(0x3d, 0x00),
	ILI9881C_COMMAND_INSTR(0x3e, 0x00),
	ILI9881C_COMMAND_INSTR(0x3f, 0x00),
	ILI9881C_COMMAND_INSTR(0x40, 0x00),
	ILI9881C_COMMAND_INSTR(0x41, 0x00),
	ILI9881C_COMMAND_INSTR(0x42, 0x00),
	ILI9881C_COMMAND_INSTR(0x43, 0x00),
	ILI9881C_COMMAND_INSTR(0x44, 0x00),
	ILI9881C_COMMAND_INSTR(0x50, 0x01),
	ILI9881C_COMMAND_INSTR(0x51, 0x23),
	ILI9881C_COMMAND_INSTR(0x52, 0x45),
	ILI9881C_COMMAND_INSTR(0x53, 0x67),
	ILI9881C_COMMAND_INSTR(0x54, 0x89),
	ILI9881C_COMMAND_INSTR(0x55, 0xab),
	ILI9881C_COMMAND_INSTR(0x56, 0x01),
	ILI9881C_COMMAND_INSTR(0x57, 0x23),
	ILI9881C_COMMAND_INSTR(0x58, 0x45),
	ILI9881C_COMMAND_INSTR(0x59, 0x67),
	ILI9881C_COMMAND_INSTR(0x5a, 0x89),
	ILI9881C_COMMAND_INSTR(0x5b, 0xab),
	ILI9881C_COMMAND_INSTR(0x5c, 0xcd),
	ILI9881C_COMMAND_INSTR(0x5d, 0xef),
	ILI9881C_COMMAND_INSTR(0x5e, 0x11),
	ILI9881C_COMMAND_INSTR(0x5f, 0x02),
	ILI9881C_COMMAND_INSTR(0x60, 0x02),
	ILI9881C_COMMAND_INSTR(0x61, 0x02),
	ILI9881C_COMMAND_INSTR(0x62, 0x02),
	ILI9881C_COMMAND_INSTR(0x63, 0x02),
	ILI9881C_COMMAND_INSTR(0x64, 0x02),
	ILI9881C_COMMAND_INSTR(0x65, 0x02),
	ILI9881C_COMMAND_INSTR(0x66, 0x02),
	ILI9881C_COMMAND_INSTR(0x67, 0x02),
	ILI9881C_COMMAND_INSTR(0x68, 0x02),
	ILI9881C_COMMAND_INSTR(0x69, 0x02),
	ILI9881C_COMMAND_INSTR(0x6a, 0x0c),
	ILI9881C_COMMAND_INSTR(0x6b, 0x02),
	ILI9881C_COMMAND_INSTR(0x6c, 0x0f),
	ILI9881C_COMMAND_INSTR(0x6d, 0x0e),
	ILI9881C_COMMAND_INSTR(0x6e, 0x0d),
	ILI9881C_COMMAND_INSTR(0x6f, 0x06),
	ILI9881C_COMMAND_INSTR(0x70, 0x07),
	ILI9881C_COMMAND_INSTR(0x71, 0x02),
	ILI9881C_COMMAND_INSTR(0x72, 0x02),
	ILI9881C_COMMAND_INSTR(0x73, 0x02),
	ILI9881C_COMMAND_INSTR(0x74, 0x02),
	ILI9881C_COMMAND_INSTR(0x75, 0x02),
	ILI9881C_COMMAND_INSTR(0x76, 0x02),
	ILI9881C_COMMAND_INSTR(0x77, 0x02),
	ILI9881C_COMMAND_INSTR(0x78, 0x02),
	ILI9881C_COMMAND_INSTR(0x79, 0x02),
	ILI9881C_COMMAND_INSTR(0x7a, 0x02),
	ILI9881C_COMMAND_INSTR(0x7b, 0x02),
	ILI9881C_COMMAND_INSTR(0x7c, 0x02),
	ILI9881C_COMMAND_INSTR(0x7d, 0x02),
	ILI9881C_COMMAND_INSTR(0x7e, 0x02),
	ILI9881C_COMMAND_INSTR(0x7f, 0x02),
	ILI9881C_COMMAND_INSTR(0x80, 0x0c),
	ILI9881C_COMMAND_INSTR(0x81, 0x02),
	ILI9881C_COMMAND_INSTR(0x82, 0x0f),
	ILI9881C_COMMAND_INSTR(0x83, 0x0e),
	ILI9881C_COMMAND_INSTR(0x84, 0x0d),
	ILI9881C_COMMAND_INSTR(0x85, 0x06),
	ILI9881C_COMMAND_INSTR(0x86, 0x07),
	ILI9881C_COMMAND_INSTR(0x87, 0x02),
	ILI9881C_COMMAND_INSTR(0x88, 0x02),
	ILI9881C_COMMAND_INSTR(0x89, 0x02),
	ILI9881C_COMMAND_INSTR(0x8a, 0x02),
	ILI9881C_SWITCH_PAGE_INSTR(4),
	ILI9881C_COMMAND_INSTR(0x6c, 0x15),
	ILI9881C_COMMAND_INSTR(0x6e, 0x22),
	ILI9881C_COMMAND_INSTR(0x6f, 0x33),
	ILI9881C_COMMAND_INSTR(0x3a, 0xa4),
	ILI9881C_COMMAND_INSTR(0x8d, 0x0d),
	ILI9881C_COMMAND_INSTR(0x87, 0xba),
	ILI9881C_COMMAND_INSTR(0x26, 0x76),
	ILI9881C_COMMAND_INSTR(0xb2, 0xd1),
	ILI9881C_SWITCH_PAGE_INSTR(1),
	ILI9881C_COMMAND_INSTR(0x22, 0x0a),
	ILI9881C_COMMAND_INSTR(0x53, 0xdc),
	ILI9881C_COMMAND_INSTR(0x55, 0xa7),
	ILI9881C_COMMAND_INSTR(0x50, 0x78),
	ILI9881C_COMMAND_INSTR(0x51, 0x78),
	ILI9881C_COMMAND_INSTR(0x31, 0x02),
	ILI9881C_COMMAND_INSTR(0x60, 0x14),
	ILI9881C_COMMAND_INSTR(0xa0, 0x2a),
	ILI9881C_COMMAND_INSTR(0xa1, 0x39),
	ILI9881C_COMMAND_INSTR(0xa2, 0x46),
	ILI9881C_COMMAND_INSTR(0xa3, 0x0e),
	ILI9881C_COMMAND_INSTR(0xa4, 0x12),
	ILI9881C_COMMAND_INSTR(0xa5, 0x25),
	ILI9881C_COMMAND_INSTR(0xa6, 0x19),
	ILI9881C_COMMAND_INSTR(0xa7, 0x1d),
	ILI9881C_COMMAND_INSTR(0xa8, 0xa6),
	ILI9881C_COMMAND_INSTR(0xa9, 0x1c),
	ILI9881C_COMMAND_INSTR(0xaa, 0x29),
	ILI9881C_COMMAND_INSTR(0xab, 0x85),
	ILI9881C_COMMAND_INSTR(0xac, 0x1c),
	ILI9881C_COMMAND_INSTR(0xad, 0x1b),
	ILI9881C_COMMAND_INSTR(0xae, 0x51),
	ILI9881C_COMMAND_INSTR(0xaf, 0x22),
	ILI9881C_COMMAND_INSTR(0xb0, 0x2d),
	ILI9881C_COMMAND_INSTR(0xb1, 0x4f),
	ILI9881C_COMMAND_INSTR(0xb2, 0x59),
	ILI9881C_COMMAND_INSTR(0xb3, 0x3f),
	ILI9881C_COMMAND_INSTR(0xc0, 0x2a),
	ILI9881C_COMMAND_INSTR(0xc1, 0x3a),
	ILI9881C_COMMAND_INSTR(0xc2, 0x45),
	ILI9881C_COMMAND_INSTR(0xc3, 0x0e),
	ILI9881C_COMMAND_INSTR(0xc4, 0x11),
	ILI9881C_COMMAND_INSTR(0xc5, 0x24),
	ILI9881C_COMMAND_INSTR(0xc6, 0x1a),
	ILI9881C_COMMAND_INSTR(0xc7, 0x1c),
	ILI9881C_COMMAND_INSTR(0xc8, 0xaa),
	ILI9881C_COMMAND_INSTR(0xc9, 0x1c),
	ILI9881C_COMMAND_INSTR(0xca, 0x29),
	ILI9881C_COMMAND_INSTR(0xcb, 0x96),
	ILI9881C_COMMAND_INSTR(0xcc, 0x1c),
	ILI9881C_COMMAND_INSTR(0xcd, 0x1b),
	ILI9881C_COMMAND_INSTR(0xce, 0x51),
	ILI9881C_COMMAND_INSTR(0xcf, 0x22),
	ILI9881C_COMMAND_INSTR(0xd0, 0x2b),
	ILI9881C_COMMAND_INSTR(0xd1, 0x4b),
	ILI9881C_COMMAND_INSTR(0xd2, 0x59),
	ILI9881C_COMMAND_INSTR(0xd3, 0x3f),
};

static const struct ili9881c_instr k101_im2byl02_init[] = {
	ILI9881C_SWITCH_PAGE_INSTR(3),
	ILI9881C_COMMAND_INSTR(0x01, 0x00),
	ILI9881C_COMMAND_INSTR(0x02, 0x00),
	ILI9881C_COMMAND_INSTR(0x03, 0x73),
	ILI9881C_COMMAND_INSTR(0x04, 0x00),
	ILI9881C_COMMAND_INSTR(0x05, 0x00),
	ILI9881C_COMMAND_INSTR(0x06, 0x08),
	ILI9881C_COMMAND_INSTR(0x07, 0x00),
	ILI9881C_COMMAND_INSTR(0x08, 0x00),
	ILI9881C_COMMAND_INSTR(0x09, 0x00),
	ILI9881C_COMMAND_INSTR(0x0a, 0x01),
	ILI9881C_COMMAND_INSTR(0x0b, 0x01),
	ILI9881C_COMMAND_INSTR(0x0c, 0x00),
	ILI9881C_COMMAND_INSTR(0x0d, 0x01),
	ILI9881C_COMMAND_INSTR(0x0e, 0x01),
	ILI9881C_COMMAND_INSTR(0x0f, 0x00),
	ILI9881C_COMMAND_INSTR(0x10, 0x00),
	ILI9881C_COMMAND_INSTR(0x11, 0x00),
	ILI9881C_COMMAND_INSTR(0x12, 0x00),
	ILI9881C_COMMAND_INSTR(0x13, 0x00),
	ILI9881C_COMMAND_INSTR(0x14, 0x00),
	ILI9881C_COMMAND_INSTR(0x15, 0x00),
	ILI9881C_COMMAND_INSTR(0x16, 0x00),
	ILI9881C_COMMAND_INSTR(0x17, 0x00),
	ILI9881C_COMMAND_INSTR(0x18, 0x00),
	ILI9881C_COMMAND_INSTR(0x19, 0x00),
	ILI9881C_COMMAND_INSTR(0x1a, 0x00),
	ILI9881C_COMMAND_INSTR(0x1b, 0x00),
	ILI9881C_COMMAND_INSTR(0x1c, 0x00),
	ILI9881C_COMMAND_INSTR(0x1d, 0x00),
	ILI9881C_COMMAND_INSTR(0x1e, 0x40),
	ILI9881C_COMMAND_INSTR(0x1f, 0xc0),
	ILI9881C_COMMAND_INSTR(0x20, 0x06),
	ILI9881C_COMMAND_INSTR(0x21, 0x01),
	ILI9881C_COMMAND_INSTR(0x22, 0x06),
	ILI9881C_COMMAND_INSTR(0x23, 0x01),
	ILI9881C_COMMAND_INSTR(0x24, 0x88),
	ILI9881C_COMMAND_INSTR(0x25, 0x88),
	ILI9881C_COMMAND_INSTR(0x26, 0x00),
	ILI9881C_COMMAND_INSTR(0x27, 0x00),
	ILI9881C_COMMAND_INSTR(0x28, 0x3b),
	ILI9881C_COMMAND_INSTR(0x29, 0x03),
	ILI9881C_COMMAND_INSTR(0x2a, 0x00),
	ILI9881C_COMMAND_INSTR(0x2b, 0x00),
	ILI9881C_COMMAND_INSTR(0x2c, 0x00),
	ILI9881C_COMMAND_INSTR(0x2d, 0x00),
	ILI9881C_COMMAND_INSTR(0x2e, 0x00),
	ILI9881C_COMMAND_INSTR(0x2f, 0x00),
	ILI9881C_COMMAND_INSTR(0x30, 0x00),
	ILI9881C_COMMAND_INSTR(0x31, 0x00),
	ILI9881C_COMMAND_INSTR(0x32, 0x00),
	ILI9881C_COMMAND_INSTR(0x33, 0x00),
	ILI9881C_COMMAND_INSTR(0x34, 0x00), /* GPWR1/2 non overlap time 2.62us */
	ILI9881C_COMMAND_INSTR(0x35, 0x00),
	ILI9881C_COMMAND_INSTR(0x36, 0x00),
	ILI9881C_COMMAND_INSTR(0x37, 0x00),
	ILI9881C_COMMAND_INSTR(0x38, 0x00),
	ILI9881C_COMMAND_INSTR(0x39, 0x00),
	ILI9881C_COMMAND_INSTR(0x3a, 0x00),
	ILI9881C_COMMAND_INSTR(0x3b, 0x00),
	ILI9881C_COMMAND_INSTR(0x3c, 0x00),
	ILI9881C_COMMAND_INSTR(0x3d, 0x00),
	ILI9881C_COMMAND_INSTR(0x3e, 0x00),
	ILI9881C_COMMAND_INSTR(0x3f, 0x00),
	ILI9881C_COMMAND_INSTR(0x40, 0x00),
	ILI9881C_COMMAND_INSTR(0x41, 0x00),
	ILI9881C_COMMAND_INSTR(0x42, 0x00),
	ILI9881C_COMMAND_INSTR(0x43, 0x00),
	ILI9881C_COMMAND_INSTR(0x44, 0x00),
	ILI9881C_COMMAND_INSTR(0x50, 0x01),
	ILI9881C_COMMAND_INSTR(0x51, 0x23),
	ILI9881C_COMMAND_INSTR(0x52, 0x45),
	ILI9881C_COMMAND_INSTR(0x53, 0x67),
	ILI9881C_COMMAND_INSTR(0x54, 0x89),
	ILI9881C_COMMAND_INSTR(0x55, 0xab),
	ILI9881C_COMMAND_INSTR(0x56, 0x01),
	ILI9881C_COMMAND_INSTR(0x57, 0x23),
	ILI9881C_COMMAND_INSTR(0x58, 0x45),
	ILI9881C_COMMAND_INSTR(0x59, 0x67),
	ILI9881C_COMMAND_INSTR(0x5a, 0x89),
	ILI9881C_COMMAND_INSTR(0x5b, 0xab),
	ILI9881C_COMMAND_INSTR(0x5c, 0xcd),
	ILI9881C_COMMAND_INSTR(0x5d, 0xef),
	ILI9881C_COMMAND_INSTR(0x5e, 0x00),
	ILI9881C_COMMAND_INSTR(0x5f, 0x01),
	ILI9881C_COMMAND_INSTR(0x60, 0x01),
	ILI9881C_COMMAND_INSTR(0x61, 0x06),
	ILI9881C_COMMAND_INSTR(0x62, 0x06),
	ILI9881C_COMMAND_INSTR(0x63, 0x07),
	ILI9881C_COMMAND_INSTR(0x64, 0x07),
	ILI9881C_COMMAND_INSTR(0x65, 0x00),
	ILI9881C_COMMAND_INSTR(0x66, 0x00),
	ILI9881C_COMMAND_INSTR(0x67, 0x02),
	ILI9881C_COMMAND_INSTR(0x68, 0x02),
	ILI9881C_COMMAND_INSTR(0x69, 0x05),
	ILI9881C_COMMAND_INSTR(0x6a, 0x05),
	ILI9881C_COMMAND_INSTR(0x6b, 0x02),
	ILI9881C_COMMAND_INSTR(0x6c, 0x0d),
	ILI9881C_COMMAND_INSTR(0x6d, 0x0d),
	ILI9881C_COMMAND_INSTR(0x6e, 0x0c),
	ILI9881C_COMMAND_INSTR(0x6f, 0x0c),
	ILI9881C_COMMAND_INSTR(0x70, 0x0f),
	ILI9881C_COMMAND_INSTR(0x71, 0x0f),
	ILI9881C_COMMAND_INSTR(0x72, 0x0e),
	ILI9881C_COMMAND_INSTR(0x73, 0x0e),
	ILI9881C_COMMAND_INSTR(0x74, 0x02),
	ILI9881C_COMMAND_INSTR(0x75, 0x01),
	ILI9881C_COMMAND_INSTR(0x76, 0x01),
	ILI9881C_COMMAND_INSTR(0x77, 0x06),
	ILI9881C_COMMAND_INSTR(0x78, 0x06),
	ILI9881C_COMMAND_INSTR(0x79, 0x07),
	ILI9881C_COMMAND_INSTR(0x7a, 0x07),
	ILI9881C_COMMAND_INSTR(0x7b, 0x00),
	ILI9881C_COMMAND_INSTR(0x7c, 0x00),
	ILI9881C_COMMAND_INSTR(0x7d, 0x02),
	ILI9881C_COMMAND_INSTR(0x7e, 0x02),
	ILI9881C_COMMAND_INSTR(0x7f, 0x05),
	ILI9881C_COMMAND_INSTR(0x80, 0x05),
	ILI9881C_COMMAND_INSTR(0x81, 0x02),
	ILI9881C_COMMAND_INSTR(0x82, 0x0d),
	ILI9881C_COMMAND_INSTR(0x83, 0x0d),
	ILI9881C_COMMAND_INSTR(0x84, 0x0c),
	ILI9881C_COMMAND_INSTR(0x85, 0x0c),
	ILI9881C_COMMAND_INSTR(0x86, 0x0f),
	ILI9881C_COMMAND_INSTR(0x87, 0x0f),
	ILI9881C_COMMAND_INSTR(0x88, 0x0e),
	ILI9881C_COMMAND_INSTR(0x89, 0x0e),
	ILI9881C_COMMAND_INSTR(0x8a, 0x02),
	ILI9881C_SWITCH_PAGE_INSTR(4),
	ILI9881C_COMMAND_INSTR(0x3b, 0xc0), /* ILI4003D sel */
	ILI9881C_COMMAND_INSTR(0x6c, 0x15), /* Set VCORE voltage = 1.5V */
	ILI9881C_COMMAND_INSTR(0x6e, 0x2a), /* di_pwr_reg=0 for power mode 2A, VGH clamp 18V */
	ILI9881C_COMMAND_INSTR(0x6f, 0x33), /* pumping ratio VGH=5x VGL=-3x */
	ILI9881C_COMMAND_INSTR(0x8d, 0x1b), /* VGL clamp -10V */
	ILI9881C_COMMAND_INSTR(0x87, 0xba), /* ESD */
	ILI9881C_COMMAND_INSTR(0x3a, 0x24), /* POWER SAVING */
	ILI9881C_COMMAND_INSTR(0x26, 0x76),
	ILI9881C_COMMAND_INSTR(0xb2, 0xd1),
	ILI9881C_SWITCH_PAGE_INSTR(1),
	ILI9881C_COMMAND_INSTR(0x22, 0x0a), /* BGR, SS */
	ILI9881C_COMMAND_INSTR(0x31, 0x00), /* Zigzag type3 inversion */
	ILI9881C_COMMAND_INSTR(0x40, 0x53), /* ILI4003D sel */
	ILI9881C_COMMAND_INSTR(0x43, 0x66),
	ILI9881C_COMMAND_INSTR(0x53, 0x4c),
	ILI9881C_COMMAND_INSTR(0x50, 0x87),
	ILI9881C_COMMAND_INSTR(0x51, 0x82),
	ILI9881C_COMMAND_INSTR(0x60, 0x15),
	ILI9881C_COMMAND_INSTR(0x61, 0x01),
	ILI9881C_COMMAND_INSTR(0x62, 0x0c),
	ILI9881C_COMMAND_INSTR(0x63, 0x00),
	ILI9881C_COMMAND_INSTR(0xa0, 0x00),
	ILI9881C_COMMAND_INSTR(0xa1, 0x13), /* VP251 */
	ILI9881C_COMMAND_INSTR(0xa2, 0x23), /* VP247 */
	ILI9881C_COMMAND_INSTR(0xa3, 0x14), /* VP243 */
	ILI9881C_COMMAND_INSTR(0xa4, 0x16), /* VP239 */
	ILI9881C_COMMAND_INSTR(0xa5, 0x29), /* VP231 */
	ILI9881C_COMMAND_INSTR(0xa6, 0x1e), /* VP219 */
	ILI9881C_COMMAND_INSTR(0xa7, 0x1d), /* VP203 */
	ILI9881C_COMMAND_INSTR(0xa8, 0x86), /* VP175 */
	ILI9881C_COMMAND_INSTR(0xa9, 0x1e), /* VP144 */
	ILI9881C_COMMAND_INSTR(0xaa, 0x29), /* VP111 */
	ILI9881C_COMMAND_INSTR(0xab, 0x74), /* VP80 */
	ILI9881C_COMMAND_INSTR(0xac, 0x19), /* VP52 */
	ILI9881C_COMMAND_INSTR(0xad, 0x17), /* VP36 */
	ILI9881C_COMMAND_INSTR(0xae, 0x4b), /* VP24 */
	ILI9881C_COMMAND_INSTR(0xaf, 0x20), /* VP16 */
	ILI9881C_COMMAND_INSTR(0xb0, 0x26), /* VP12 */
	ILI9881C_COMMAND_INSTR(0xb1, 0x4c), /* VP8 */
	ILI9881C_COMMAND_INSTR(0xb2, 0x5d), /* VP4 */
	ILI9881C_COMMAND_INSTR(0xb3, 0x3f), /* VP0 */
	ILI9881C_COMMAND_INSTR(0xc0, 0x00), /* VN255 GAMMA N */
	ILI9881C_COMMAND_INSTR(0xc1, 0x13), /* VN251 */
	ILI9881C_COMMAND_INSTR(0xc2, 0x23), /* VN247 */
	ILI9881C_COMMAND_INSTR(0xc3, 0x14), /* VN243 */
	ILI9881C_COMMAND_INSTR(0xc4, 0x16), /* VN239 */
	ILI9881C_COMMAND_INSTR(0xc5, 0x29), /* VN231 */
	ILI9881C_COMMAND_INSTR(0xc6, 0x1e), /* VN219 */
	ILI9881C_COMMAND_INSTR(0xc7, 0x1d), /* VN203 */
	ILI9881C_COMMAND_INSTR(0xc8, 0x86), /* VN175 */
	ILI9881C_COMMAND_INSTR(0xc9, 0x1e), /* VN144 */
	ILI9881C_COMMAND_INSTR(0xca, 0x29), /* VN111 */
	ILI9881C_COMMAND_INSTR(0xcb, 0x74), /* VN80 */
	ILI9881C_COMMAND_INSTR(0xcc, 0x19), /* VN52 */
	ILI9881C_COMMAND_INSTR(0xcd, 0x17), /* VN36 */
	ILI9881C_COMMAND_INSTR(0xce, 0x4b), /* VN24 */
	ILI9881C_COMMAND_INSTR(0xcf, 0x20), /* VN16 */
	ILI9881C_COMMAND_INSTR(0xd0, 0x26), /* VN12 */
	ILI9881C_COMMAND_INSTR(0xd1, 0x4c), /* VN8 */
	ILI9881C_COMMAND_INSTR(0xd2, 0x5d), /* VN4 */
	ILI9881C_COMMAND_INSTR(0xd3, 0x3f), /* VN0 */
};

static const struct ili9881c_instr kd050hdfia020_init[] = {
	ILI9881C_SWITCH_PAGE_INSTR(3),
	ILI9881C_COMMAND_INSTR(0x01, 0x00),
	ILI9881C_COMMAND_INSTR(0x02, 0x00),
	ILI9881C_COMMAND_INSTR(0x03, 0x72),
	ILI9881C_COMMAND_INSTR(0x04, 0x00),
	ILI9881C_COMMAND_INSTR(0x05, 0x00),
	ILI9881C_COMMAND_INSTR(0x06, 0x09),
	ILI9881C_COMMAND_INSTR(0x07, 0x00),
	ILI9881C_COMMAND_INSTR(0x08, 0x00),
	ILI9881C_COMMAND_INSTR(0x09, 0x01),
	ILI9881C_COMMAND_INSTR(0x0a, 0x00),
	ILI9881C_COMMAND_INSTR(0x0b, 0x00),
	ILI9881C_COMMAND_INSTR(0x0c, 0x01),
	ILI9881C_COMMAND_INSTR(0x0d, 0x00),
	ILI9881C_COMMAND_INSTR(0x0e, 0x00),
	ILI9881C_COMMAND_INSTR(0x0f, 0x00),
	ILI9881C_COMMAND_INSTR(0x10, 0x00),
	ILI9881C_COMMAND_INSTR(0x11, 0x00),
	ILI9881C_COMMAND_INSTR(0x12, 0x00),
	ILI9881C_COMMAND_INSTR(0x13, 0x00),
	ILI9881C_COMMAND_INSTR(0x14, 0x00),
	ILI9881C_COMMAND_INSTR(0x15, 0x00),
	ILI9881C_COMMAND_INSTR(0x16, 0x00),
	ILI9881C_COMMAND_INSTR(0x17, 0x00),
	ILI9881C_COMMAND_INSTR(0x18, 0x00),
	ILI9881C_COMMAND_INSTR(0x19, 0x00),
	ILI9881C_COMMAND_INSTR(0x1a, 0x00),
	ILI9881C_COMMAND_INSTR(0x1b, 0x00),
	ILI9881C_COMMAND_INSTR(0x1c, 0x00),
	ILI9881C_COMMAND_INSTR(0x1d, 0x00),
	ILI9881C_COMMAND_INSTR(0x1e, 0x40),
	ILI9881C_COMMAND_INSTR(0x1f, 0x80),
	ILI9881C_COMMAND_INSTR(0x20, 0x05),
	ILI9881C_COMMAND_INSTR(0x20, 0x05),
	ILI9881C_COMMAND_INSTR(0x21, 0x02),
	ILI9881C_COMMAND_INSTR(0x22, 0x00),
	ILI9881C_COMMAND_INSTR(0x23, 0x00),
	ILI9881C_COMMAND_INSTR(0x24, 0x00),
	ILI9881C_COMMAND_INSTR(0x25, 0x00),
	ILI9881C_COMMAND_INSTR(0x26, 0x00),
	ILI9881C_COMMAND_INSTR(0x27, 0x00),
	ILI9881C_COMMAND_INSTR(0x28, 0x33),
	ILI9881C_COMMAND_INSTR(0x29, 0x02),
	ILI9881C_COMMAND_INSTR(0x2a, 0x00),
	ILI9881C_COMMAND_INSTR(0x2b, 0x00),
	ILI9881C_COMMAND_INSTR(0x2c, 0x00),
	ILI9881C_COMMAND_INSTR(0x2d, 0x00),
	ILI9881C_COMMAND_INSTR(0x2e, 0x00),
	ILI9881C_COMMAND_INSTR(0x2f, 0x00),
	ILI9881C_COMMAND_INSTR(0x30, 0x00),
	ILI9881C_COMMAND_INSTR(0x31, 0x00),
	ILI9881C_COMMAND_INSTR(0x32, 0x00),
	ILI9881C_COMMAND_INSTR(0x32, 0x00),
	ILI9881C_COMMAND_INSTR(0x33, 0x00),
	ILI9881C_COMMAND_INSTR(0x34, 0x04),
	ILI9881C_COMMAND_INSTR(0x35, 0x00),
	ILI9881C_COMMAND_INSTR(0x36, 0x00),
	ILI9881C_COMMAND_INSTR(0x37, 0x00),
	ILI9881C_COMMAND_INSTR(0x38, 0x3c),
	ILI9881C_COMMAND_INSTR(0x39, 0x00),
	ILI9881C_COMMAND_INSTR(0x3a, 0x40),
	ILI9881C_COMMAND_INSTR(0x3b, 0x40),
	ILI9881C_COMMAND_INSTR(0x3c, 0x00),
	ILI9881C_COMMAND_INSTR(0x3d, 0x00),
	ILI9881C_COMMAND_INSTR(0x3e, 0x00),
	ILI9881C_COMMAND_INSTR(0x3f, 0x00),
	ILI9881C_COMMAND_INSTR(0x40, 0x00),
	ILI9881C_COMMAND_INSTR(0x41, 0x00),
	ILI9881C_COMMAND_INSTR(0x42, 0x00),
	ILI9881C_COMMAND_INSTR(0x43, 0x00),
	ILI9881C_COMMAND_INSTR(0x44, 0x00),
	ILI9881C_COMMAND_INSTR(0x50, 0x01),
	ILI9881C_COMMAND_INSTR(0x51, 0x23),
	ILI9881C_COMMAND_INSTR(0x52, 0x45),
	ILI9881C_COMMAND_INSTR(0x53, 0x67),
	ILI9881C_COMMAND_INSTR(0x54, 0x89),
	ILI9881C_COMMAND_INSTR(0x55, 0xab),
	ILI9881C_COMMAND_INSTR(0x56, 0x01),
	ILI9881C_COMMAND_INSTR(0x57, 0x23),
	ILI9881C_COMMAND_INSTR(0x58, 0x45),
	ILI9881C_COMMAND_INSTR(0x59, 0x67),
	ILI9881C_COMMAND_INSTR(0x5a, 0x89),
	ILI9881C_COMMAND_INSTR(0x5b, 0xab),
	ILI9881C_COMMAND_INSTR(0x5c, 0xcd),
	ILI9881C_COMMAND_INSTR(0x5d, 0xef),
	ILI9881C_COMMAND_INSTR(0x5e, 0x11),
	ILI9881C_COMMAND_INSTR(0x5f, 0x01),
	ILI9881C_COMMAND_INSTR(0x60, 0x00),
	ILI9881C_COMMAND_INSTR(0x61, 0x15),
	ILI9881C_COMMAND_INSTR(0x62, 0x14),
	ILI9881C_COMMAND_INSTR(0x63, 0x0e),
	ILI9881C_COMMAND_INSTR(0x64, 0x0f),
	ILI9881C_COMMAND_INSTR(0x65, 0x0c),
	ILI9881C_COMMAND_INSTR(0x66, 0x0d),
	ILI9881C_COMMAND_INSTR(0x67, 0x06),
	ILI9881C_COMMAND_INSTR(0x68, 0x02),
	ILI9881C_COMMAND_INSTR(0x69, 0x07),
	ILI9881C_COMMAND_INSTR(0x6a, 0x02),
	ILI9881C_COMMAND_INSTR(0x6b, 0x02),
	ILI9881C_COMMAND_INSTR(0x6c, 0x02),
	ILI9881C_COMMAND_INSTR(0x6d, 0x02),
	ILI9881C_COMMAND_INSTR(0x6e, 0x02),
	ILI9881C_COMMAND_INSTR(0x6f, 0x02),
	ILI9881C_COMMAND_INSTR(0x70, 0x02),
	ILI9881C_COMMAND_INSTR(0x71, 0x02),
	ILI9881C_COMMAND_INSTR(0x72, 0x02),
	ILI9881C_COMMAND_INSTR(0x73, 0x02),
	ILI9881C_COMMAND_INSTR(0x74, 0x02),
	ILI9881C_COMMAND_INSTR(0x75, 0x01),
	ILI9881C_COMMAND_INSTR(0x76, 0x00),
	ILI9881C_COMMAND_INSTR(0x77, 0x14),
	ILI9881C_COMMAND_INSTR(0x78, 0x15),
	ILI9881C_COMMAND_INSTR(0x79, 0x0e),
	ILI9881C_COMMAND_INSTR(0x7a, 0x0f),
	ILI9881C_COMMAND_INSTR(0x7b, 0x0c),
	ILI9881C_COMMAND_INSTR(0x7c, 0x0d),
	ILI9881C_COMMAND_INSTR(0x7d, 0x06),
	ILI9881C_COMMAND_INSTR(0x7e, 0x02),
	ILI9881C_COMMAND_INSTR(0x7f, 0x07),
	ILI9881C_COMMAND_INSTR(0x80, 0x02),
	ILI9881C_COMMAND_INSTR(0x81, 0x02),
	ILI9881C_COMMAND_INSTR(0x83, 0x02),
	ILI9881C_COMMAND_INSTR(0x84, 0x02),
	ILI9881C_COMMAND_INSTR(0x85, 0x02),
	ILI9881C_COMMAND_INSTR(0x86, 0x02),
	ILI9881C_COMMAND_INSTR(0x87, 0x02),
	ILI9881C_COMMAND_INSTR(0x88, 0x02),
	ILI9881C_COMMAND_INSTR(0x89, 0x02),
	ILI9881C_COMMAND_INSTR(0x8a, 0x02),
	ILI9881C_SWITCH_PAGE_INSTR(0x4),
	ILI9881C_COMMAND_INSTR(0x6c, 0x15),
	ILI9881C_COMMAND_INSTR(0x6e, 0x2a),
	ILI9881C_COMMAND_INSTR(0x6f, 0x33),
	ILI9881C_COMMAND_INSTR(0x3a, 0x94),
	ILI9881C_COMMAND_INSTR(0x8d, 0x15),
	ILI9881C_COMMAND_INSTR(0x87, 0xba),
	ILI9881C_COMMAND_INSTR(0x26, 0x76),
	ILI9881C_COMMAND_INSTR(0xb2, 0xd1),
	ILI9881C_COMMAND_INSTR(0xb5, 0x06),
	ILI9881C_SWITCH_PAGE_INSTR(0x1),
	ILI9881C_COMMAND_INSTR(0x22, 0x0a),
	ILI9881C_COMMAND_INSTR(0x31, 0x00),
	ILI9881C_COMMAND_INSTR(0x53, 0x90),
	ILI9881C_COMMAND_INSTR(0x55, 0xa2),
	ILI9881C_COMMAND_INSTR(0x50, 0xb7),
	ILI9881C_COMMAND_INSTR(0x51, 0xb7),
	ILI9881C_COMMAND_INSTR(0x60, 0x22),
	ILI9881C_COMMAND_INSTR(0x61, 0x00),
	ILI9881C_COMMAND_INSTR(0x62, 0x19),
	ILI9881C_COMMAND_INSTR(0x63, 0x10),
	ILI9881C_COMMAND_INSTR(0xa0, 0x08),
	ILI9881C_COMMAND_INSTR(0xa1, 0x1a),
	ILI9881C_COMMAND_INSTR(0xa2, 0x27),
	ILI9881C_COMMAND_INSTR(0xa3, 0x15),
	ILI9881C_COMMAND_INSTR(0xa4, 0x17),
	ILI9881C_COMMAND_INSTR(0xa5, 0x2a),
	ILI9881C_COMMAND_INSTR(0xa6, 0x1e),
	ILI9881C_COMMAND_INSTR(0xa7, 0x1f),
	ILI9881C_COMMAND_INSTR(0xa8, 0x8b),
	ILI9881C_COMMAND_INSTR(0xa9, 0x1b),
	ILI9881C_COMMAND_INSTR(0xaa, 0x27),
	ILI9881C_COMMAND_INSTR(0xab, 0x78),
	ILI9881C_COMMAND_INSTR(0xac, 0x18),
	ILI9881C_COMMAND_INSTR(0xad, 0x18),
	ILI9881C_COMMAND_INSTR(0xae, 0x4c),
	ILI9881C_COMMAND_INSTR(0xaf, 0x21),
	ILI9881C_COMMAND_INSTR(0xb0, 0x27),
	ILI9881C_COMMAND_INSTR(0xb1, 0x54),
	ILI9881C_COMMAND_INSTR(0xb2, 0x67),
	ILI9881C_COMMAND_INSTR(0xb3, 0x39),
	ILI9881C_COMMAND_INSTR(0xc0, 0x08),
	ILI9881C_COMMAND_INSTR(0xc1, 0x1a),
	ILI9881C_COMMAND_INSTR(0xc2, 0x27),
	ILI9881C_COMMAND_INSTR(0xc3, 0x15),
	ILI9881C_COMMAND_INSTR(0xc4, 0x17),
	ILI9881C_COMMAND_INSTR(0xc5, 0x2a),
	ILI9881C_COMMAND_INSTR(0xc6, 0x1e),
	ILI9881C_COMMAND_INSTR(0xc7, 0x1f),
	ILI9881C_COMMAND_INSTR(0xc8, 0x8b),
	ILI9881C_COMMAND_INSTR(0xc9, 0x1b),
	ILI9881C_COMMAND_INSTR(0xca, 0x27),
	ILI9881C_COMMAND_INSTR(0xcb, 0x78),
	ILI9881C_COMMAND_INSTR(0xcc, 0x18),
	ILI9881C_COMMAND_INSTR(0xcd, 0x18),
	ILI9881C_COMMAND_INSTR(0xce, 0x4c),
	ILI9881C_COMMAND_INSTR(0xcf, 0x21),
	ILI9881C_COMMAND_INSTR(0xd0, 0x27),
	ILI9881C_COMMAND_INSTR(0xd1, 0x54),
	ILI9881C_COMMAND_INSTR(0xd2, 0x67),
	ILI9881C_COMMAND_INSTR(0xd3, 0x39),
	ILI9881C_SWITCH_PAGE_INSTR(0),
	ILI9881C_COMMAND_INSTR(0x35, 0x00),
	ILI9881C_COMMAND_INSTR(0x3a, 0x7),
};

static const struct ili9881c_instr tl050hdv35_init[] = {
	ILI9881C_SWITCH_PAGE_INSTR(3),
	ILI9881C_COMMAND_INSTR(0x01, 0x00),
	ILI9881C_COMMAND_INSTR(0x02, 0x00),
	ILI9881C_COMMAND_INSTR(0x03, 0x73),
	ILI9881C_COMMAND_INSTR(0x04, 0x00),
	ILI9881C_COMMAND_INSTR(0x05, 0x00),
	ILI9881C_COMMAND_INSTR(0x06, 0x0a),
	ILI9881C_COMMAND_INSTR(0x07, 0x00),
	ILI9881C_COMMAND_INSTR(0x08, 0x00),
	ILI9881C_COMMAND_INSTR(0x09, 0x01),
	ILI9881C_COMMAND_INSTR(0x0a, 0x00),
	ILI9881C_COMMAND_INSTR(0x0b, 0x00),
	ILI9881C_COMMAND_INSTR(0x0c, 0x01),
	ILI9881C_COMMAND_INSTR(0x0d, 0x00),
	ILI9881C_COMMAND_INSTR(0x0e, 0x00),
	ILI9881C_COMMAND_INSTR(0x0f, 0x1d),
	ILI9881C_COMMAND_INSTR(0x10, 0x1d),
	ILI9881C_COMMAND_INSTR(0x15, 0x00),
	ILI9881C_COMMAND_INSTR(0x16, 0x00),
	ILI9881C_COMMAND_INSTR(0x17, 0x00),
	ILI9881C_COMMAND_INSTR(0x18, 0x00),
	ILI9881C_COMMAND_INSTR(0x19, 0x00),
	ILI9881C_COMMAND_INSTR(0x1a, 0x00),
	ILI9881C_COMMAND_INSTR(0x1b, 0x00),
	ILI9881C_COMMAND_INSTR(0x1c, 0x00),
	ILI9881C_COMMAND_INSTR(0x1d, 0x00),
	ILI9881C_COMMAND_INSTR(0x1e, 0x40),
	ILI9881C_COMMAND_INSTR(0x1f, 0x80),
	ILI9881C_COMMAND_INSTR(0x20, 0x06),
	ILI9881C_COMMAND_INSTR(0x21, 0x02),
	ILI9881C_COMMAND_INSTR(0x28, 0x33),
	ILI9881C_COMMAND_INSTR(0x29, 0x03),
	ILI9881C_COMMAND_INSTR(0x2a, 0x00),
	ILI9881C_COMMAND_INSTR(0x2b, 0x00),
	ILI9881C_COMMAND_INSTR(0x2c, 0x00),
	ILI9881C_COMMAND_INSTR(0x2d, 0x00),
	ILI9881C_COMMAND_INSTR(0x2e, 0x00),
	ILI9881C_COMMAND_INSTR(0x2f, 0x00),
	ILI9881C_COMMAND_INSTR(0x35, 0x00),
	ILI9881C_COMMAND_INSTR(0x36, 0x00),
	ILI9881C_COMMAND_INSTR(0x37, 0x00),
	ILI9881C_COMMAND_INSTR(0x38, 0x3c),
	ILI9881C_COMMAND_INSTR(0x39, 0x00),
	ILI9881C_COMMAND_INSTR(0x3a, 0x40),
	ILI9881C_COMMAND_INSTR(0x3b, 0x40),
	ILI9881C_COMMAND_INSTR(0x3c, 0x00),
	ILI9881C_COMMAND_INSTR(0x3d, 0x00),
	ILI9881C_COMMAND_INSTR(0x3e, 0x00),
	ILI9881C_COMMAND_INSTR(0x3f, 0x00),
	ILI9881C_COMMAND_INSTR(0x40, 0x00),
	ILI9881C_COMMAND_INSTR(0x41, 0x00),
	ILI9881C_COMMAND_INSTR(0x42, 0x00),
	ILI9881C_COMMAND_INSTR(0x43, 0x00),
	ILI9881C_COMMAND_INSTR(0x44, 0x00),
	ILI9881C_COMMAND_INSTR(0x55, 0xab),
	ILI9881C_COMMAND_INSTR(0x5a, 0x89),
	ILI9881C_COMMAND_INSTR(0x5b, 0xab),
	ILI9881C_COMMAND_INSTR(0x5c, 0xcd),
	ILI9881C_COMMAND_INSTR(0x5d, 0xef),
	ILI9881C_COMMAND_INSTR(0x5e, 0x11),
	ILI9881C_COMMAND_INSTR(0x5f, 0x01),
	ILI9881C_COMMAND_INSTR(0x60, 0x00),
	ILI9881C_COMMAND_INSTR(0x61, 0x15),
	ILI9881C_COMMAND_INSTR(0x62, 0x14),
	ILI9881C_COMMAND_INSTR(0x63, 0x0e),
	ILI9881C_COMMAND_INSTR(0x64, 0x0f),
	ILI9881C_COMMAND_INSTR(0x65, 0x0c),
	ILI9881C_COMMAND_INSTR(0x66, 0x0d),
	ILI9881C_COMMAND_INSTR(0x67, 0x06),
	ILI9881C_COMMAND_INSTR(0x68, 0x02),
	ILI9881C_COMMAND_INSTR(0x69, 0x07),
	ILI9881C_COMMAND_INSTR(0x6a, 0x02),
	ILI9881C_COMMAND_INSTR(0x6b, 0x02),
	ILI9881C_COMMAND_INSTR(0x6c, 0x02),
	ILI9881C_COMMAND_INSTR(0x6d, 0x02),
	ILI9881C_COMMAND_INSTR(0x6e, 0x02),
	ILI9881C_COMMAND_INSTR(0x6f, 0x02),
	ILI9881C_COMMAND_INSTR(0x70, 0x02),
	ILI9881C_COMMAND_INSTR(0x71, 0x02),
	ILI9881C_COMMAND_INSTR(0x72, 0x02),
	ILI9881C_COMMAND_INSTR(0x73, 0x02),
	ILI9881C_COMMAND_INSTR(0x74, 0x02),
	ILI9881C_COMMAND_INSTR(0x75, 0x01),
	ILI9881C_COMMAND_INSTR(0x76, 0x00),
	ILI9881C_COMMAND_INSTR(0x77, 0x14),
	ILI9881C_COMMAND_INSTR(0x78, 0x15),
	ILI9881C_COMMAND_INSTR(0x79, 0x0e),
	ILI9881C_COMMAND_INSTR(0x7a, 0x0f),
	ILI9881C_COMMAND_INSTR(0x7b, 0x0c),
	ILI9881C_COMMAND_INSTR(0x7c, 0x0d),
	ILI9881C_COMMAND_INSTR(0x7d, 0x06),
	ILI9881C_COMMAND_INSTR(0x7e, 0x02),
	ILI9881C_COMMAND_INSTR(0x7f, 0x07),
	ILI9881C_COMMAND_INSTR(0x88, 0x02),
	ILI9881C_COMMAND_INSTR(0x89, 0x02),
	ILI9881C_COMMAND_INSTR(0x8a, 0x02),
	ILI9881C_SWITCH_PAGE_INSTR(4),
	ILI9881C_COMMAND_INSTR(0x38, 0x01),
	ILI9881C_COMMAND_INSTR(0x39, 0x00),
	ILI9881C_COMMAND_INSTR(0x6c, 0x15),
	ILI9881C_COMMAND_INSTR(0x6e, 0x2b),
	ILI9881C_COMMAND_INSTR(0x6f, 0x33),
	ILI9881C_COMMAND_INSTR(0x8d, 0x18),
	ILI9881C_COMMAND_INSTR(0x87, 0xba),
	ILI9881C_COMMAND_INSTR(0x26, 0x76),
	ILI9881C_COMMAND_INSTR(0xb2, 0xd1),
	ILI9881C_COMMAND_INSTR(0xb5, 0x06),
	ILI9881C_COMMAND_INSTR(0x3a, 0x24),
	ILI9881C_COMMAND_INSTR(0x35, 0x1f),
	ILI9881C_COMMAND_INSTR(0x33, 0x14),
	ILI9881C_COMMAND_INSTR(0x3b, 0x98),
	ILI9881C_SWITCH_PAGE_INSTR(1),
	ILI9881C_COMMAND_INSTR(0x22, 0x0a),
	ILI9881C_COMMAND_INSTR(0x31, 0x00),
	ILI9881C_COMMAND_INSTR(0x40, 0x33),
	ILI9881C_COMMAND_INSTR(0x53, 0xa2),
	ILI9881C_COMMAND_INSTR(0x55, 0x92),
	ILI9881C_COMMAND_INSTR(0x50, 0x96),
	ILI9881C_COMMAND_INSTR(0x51, 0x96),
	ILI9881C_COMMAND_INSTR(0x60, 0x22),
	ILI9881C_COMMAND_INSTR(0x61, 0x00),
	ILI9881C_COMMAND_INSTR(0x62, 0x19),
	ILI9881C_COMMAND_INSTR(0x63, 0x00),
	ILI9881C_COMMAND_INSTR(0xa0, 0x08),
	ILI9881C_COMMAND_INSTR(0xa1, 0x11),
	ILI9881C_COMMAND_INSTR(0xa2, 0x19),
	ILI9881C_COMMAND_INSTR(0xa3, 0x0d),
	ILI9881C_COMMAND_INSTR(0xa4, 0x0d),
	ILI9881C_COMMAND_INSTR(0xa5, 0x1e),
	ILI9881C_COMMAND_INSTR(0xa6, 0x14),
	ILI9881C_COMMAND_INSTR(0xa7, 0x17),
	ILI9881C_COMMAND_INSTR(0xa8, 0x4f),
	ILI9881C_COMMAND_INSTR(0xa9, 0x1a),
	ILI9881C_COMMAND_INSTR(0xaa, 0x27),
	ILI9881C_COMMAND_INSTR(0xab, 0x49),
	ILI9881C_COMMAND_INSTR(0xac, 0x1a),
	ILI9881C_COMMAND_INSTR(0xad, 0x18),
	ILI9881C_COMMAND_INSTR(0xae, 0x4c),
	ILI9881C_COMMAND_INSTR(0xaf, 0x22),
	ILI9881C_COMMAND_INSTR(0xb0, 0x27),
	ILI9881C_COMMAND_INSTR(0xb1, 0x4b),
	ILI9881C_COMMAND_INSTR(0xb2, 0x60),
	ILI9881C_COMMAND_INSTR(0xb3, 0x39),
	ILI9881C_COMMAND_INSTR(0xc0, 0x08),
	ILI9881C_COMMAND_INSTR(0xc1, 0x11),
	ILI9881C_COMMAND_INSTR(0xc2, 0x19),
	ILI9881C_COMMAND_INSTR(0xc3, 0x0d),
	ILI9881C_COMMAND_INSTR(0xc4, 0x0d),
	ILI9881C_COMMAND_INSTR(0xc5, 0x1e),
	ILI9881C_COMMAND_INSTR(0xc6, 0x14),
	ILI9881C_COMMAND_INSTR(0xc7, 0x17),
	ILI9881C_COMMAND_INSTR(0xc8, 0x4f),
	ILI9881C_COMMAND_INSTR(0xc9, 0x1a),
	ILI9881C_COMMAND_INSTR(0xca, 0x27),
	ILI9881C_COMMAND_INSTR(0xcb, 0x49),
	ILI9881C_COMMAND_INSTR(0xcc, 0x1a),
	ILI9881C_COMMAND_INSTR(0xcd, 0x18),
	ILI9881C_COMMAND_INSTR(0xce, 0x4c),
	ILI9881C_COMMAND_INSTR(0xcf, 0x33),
	ILI9881C_COMMAND_INSTR(0xd0, 0x27),
	ILI9881C_COMMAND_INSTR(0xd1, 0x4b),
	ILI9881C_COMMAND_INSTR(0xd2, 0x60),
	ILI9881C_COMMAND_INSTR(0xd3, 0x39),
};

static const struct ili9881c_instr w552946aaa_init[] = {
	ILI9881C_SWITCH_PAGE_INSTR(3),
	ILI9881C_COMMAND_INSTR(0x01, 0x00),
	ILI9881C_COMMAND_INSTR(0x02, 0x00),
	ILI9881C_COMMAND_INSTR(0x03, 0x53),
	ILI9881C_COMMAND_INSTR(0x04, 0x53),
	ILI9881C_COMMAND_INSTR(0x05, 0x13),
	ILI9881C_COMMAND_INSTR(0x06, 0x04),
	ILI9881C_COMMAND_INSTR(0x07, 0x02),
	ILI9881C_COMMAND_INSTR(0x08, 0x02),
	ILI9881C_COMMAND_INSTR(0x09, 0x00),
	ILI9881C_COMMAND_INSTR(0x0a, 0x00),
	ILI9881C_COMMAND_INSTR(0x0b, 0x00),
	ILI9881C_COMMAND_INSTR(0x0c, 0x00),
	ILI9881C_COMMAND_INSTR(0x0d, 0x00),
	ILI9881C_COMMAND_INSTR(0x0e, 0x00),
	ILI9881C_COMMAND_INSTR(0x0f, 0x00),
	ILI9881C_COMMAND_INSTR(0x10, 0x00),
	ILI9881C_COMMAND_INSTR(0x11, 0x00),
	ILI9881C_COMMAND_INSTR(0x12, 0x00),
	ILI9881C_COMMAND_INSTR(0x13, 0x00),
	ILI9881C_COMMAND_INSTR(0x14, 0x00),
	ILI9881C_COMMAND_INSTR(0x15, 0x08),
	ILI9881C_COMMAND_INSTR(0x16, 0x10),
	ILI9881C_COMMAND_INSTR(0x17, 0x00),
	ILI9881C_COMMAND_INSTR(0x18, 0x08),
	ILI9881C_COMMAND_INSTR(0x19, 0x00),
	ILI9881C_COMMAND_INSTR(0x1a, 0x00),
	ILI9881C_COMMAND_INSTR(0x1b, 0x00),
	ILI9881C_COMMAND_INSTR(0x1c, 0x00),
	ILI9881C_COMMAND_INSTR(0x1d, 0x00),
	ILI9881C_COMMAND_INSTR(0x1e, 0xc0),
	ILI9881C_COMMAND_INSTR(0x1f, 0x80),
	ILI9881C_COMMAND_INSTR(0x20, 0x02),
	ILI9881C_COMMAND_INSTR(0x21, 0x09),
	ILI9881C_COMMAND_INSTR(0x22, 0x00),
	ILI9881C_COMMAND_INSTR(0x23, 0x00),
	ILI9881C_COMMAND_INSTR(0x24, 0x00),
	ILI9881C_COMMAND_INSTR(0x25, 0x00),
	ILI9881C_COMMAND_INSTR(0x26, 0x00),
	ILI9881C_COMMAND_INSTR(0x27, 0x00),
	ILI9881C_COMMAND_INSTR(0x28, 0x55),
	ILI9881C_COMMAND_INSTR(0x29, 0x03),
	ILI9881C_COMMAND_INSTR(0x2a, 0x00),
	ILI9881C_COMMAND_INSTR(0x2b, 0x00),
	ILI9881C_COMMAND_INSTR(0x2c, 0x00),
	ILI9881C_COMMAND_INSTR(0x2d, 0x00),
	ILI9881C_COMMAND_INSTR(0x2e, 0x00),
	ILI9881C_COMMAND_INSTR(0x2f, 0x00),
	ILI9881C_COMMAND_INSTR(0x30, 0x00),
	ILI9881C_COMMAND_INSTR(0x31, 0x00),
	ILI9881C_COMMAND_INSTR(0x32, 0x00),
	ILI9881C_COMMAND_INSTR(0x33, 0x00),
	ILI9881C_COMMAND_INSTR(0x34, 0x04),
	ILI9881C_COMMAND_INSTR(0x35, 0x05),
	ILI9881C_COMMAND_INSTR(0x36, 0x05),
	ILI9881C_COMMAND_INSTR(0x37, 0x00),
	ILI9881C_COMMAND_INSTR(0x38, 0x3c),
	ILI9881C_COMMAND_INSTR(0x39, 0x35),
	ILI9881C_COMMAND_INSTR(0x3a, 0x00),
	ILI9881C_COMMAND_INSTR(0x3b, 0x40),
	ILI9881C_COMMAND_INSTR(0x3c, 0x00),
	ILI9881C_COMMAND_INSTR(0x3d, 0x00),
	ILI9881C_COMMAND_INSTR(0x3e, 0x00),
	ILI9881C_COMMAND_INSTR(0x3f, 0x00),
	ILI9881C_COMMAND_INSTR(0x40, 0x00),
	ILI9881C_COMMAND_INSTR(0x41, 0x88),
	ILI9881C_COMMAND_INSTR(0x42, 0x00),
	ILI9881C_COMMAND_INSTR(0x43, 0x00),
	ILI9881C_COMMAND_INSTR(0x44, 0x1f),
	ILI9881C_COMMAND_INSTR(0x50, 0x01),
	ILI9881C_COMMAND_INSTR(0x51, 0x23),
	ILI9881C_COMMAND_INSTR(0x52, 0x45),
	ILI9881C_COMMAND_INSTR(0x53, 0x67),
	ILI9881C_COMMAND_INSTR(0x54, 0x89),
	ILI9881C_COMMAND_INSTR(0x55, 0xab),
	ILI9881C_COMMAND_INSTR(0x56, 0x01),
	ILI9881C_COMMAND_INSTR(0x57, 0x23),
	ILI9881C_COMMAND_INSTR(0x58, 0x45),
	ILI9881C_COMMAND_INSTR(0x59, 0x67),
	ILI9881C_COMMAND_INSTR(0x5a, 0x89),
	ILI9881C_COMMAND_INSTR(0x5b, 0xab),
	ILI9881C_COMMAND_INSTR(0x5c, 0xcd),
	ILI9881C_COMMAND_INSTR(0x5d, 0xef),
	ILI9881C_COMMAND_INSTR(0x5e, 0x03),
	ILI9881C_COMMAND_INSTR(0x5f, 0x14),
	ILI9881C_COMMAND_INSTR(0x60, 0x15),
	ILI9881C_COMMAND_INSTR(0x61, 0x0c),
	ILI9881C_COMMAND_INSTR(0x62, 0x0d),
	ILI9881C_COMMAND_INSTR(0x63, 0x0e),
	ILI9881C_COMMAND_INSTR(0x64, 0x0f),
	ILI9881C_COMMAND_INSTR(0x65, 0x10),
	ILI9881C_COMMAND_INSTR(0x66, 0x11),
	ILI9881C_COMMAND_INSTR(0x67, 0x08),
	ILI9881C_COMMAND_INSTR(0x68, 0x02),
	ILI9881C_COMMAND_INSTR(0x69, 0x0a),
	ILI9881C_COMMAND_INSTR(0x6a, 0x02),
	ILI9881C_COMMAND_INSTR(0x6b, 0x02),
	ILI9881C_COMMAND_INSTR(0x6c, 0x02),
	ILI9881C_COMMAND_INSTR(0x6d, 0x02),
	ILI9881C_COMMAND_INSTR(0x6e, 0x02),
	ILI9881C_COMMAND_INSTR(0x6f, 0x02),
	ILI9881C_COMMAND_INSTR(0x70, 0x02),
	ILI9881C_COMMAND_INSTR(0x71, 0x02),
	ILI9881C_COMMAND_INSTR(0x72, 0x06),
	ILI9881C_COMMAND_INSTR(0x73, 0x02),
	ILI9881C_COMMAND_INSTR(0x74, 0x02),
	ILI9881C_COMMAND_INSTR(0x75, 0x14),
	ILI9881C_COMMAND_INSTR(0x76, 0x15),
	ILI9881C_COMMAND_INSTR(0x77, 0x0f),
	ILI9881C_COMMAND_INSTR(0x78, 0x0e),
	ILI9881C_COMMAND_INSTR(0x79, 0x0d),
	ILI9881C_COMMAND_INSTR(0x7a, 0x0c),
	ILI9881C_COMMAND_INSTR(0x7b, 0x11),
	ILI9881C_COMMAND_INSTR(0x7c, 0x10),
	ILI9881C_COMMAND_INSTR(0x7d, 0x06),
	ILI9881C_COMMAND_INSTR(0x7e, 0x02),
	ILI9881C_COMMAND_INSTR(0x7f, 0x0a),
	ILI9881C_COMMAND_INSTR(0x80, 0x02),
	ILI9881C_COMMAND_INSTR(0x81, 0x02),
	ILI9881C_COMMAND_INSTR(0x82, 0x02),
	ILI9881C_COMMAND_INSTR(0x83, 0x02),
	ILI9881C_COMMAND_INSTR(0x84, 0x02),
	ILI9881C_COMMAND_INSTR(0x85, 0x02),
	ILI9881C_COMMAND_INSTR(0x86, 0x02),
	ILI9881C_COMMAND_INSTR(0x87, 0x02),
	ILI9881C_COMMAND_INSTR(0x88, 0x08),
	ILI9881C_COMMAND_INSTR(0x89, 0x02),
	ILI9881C_COMMAND_INSTR(0x8a, 0x02),
	ILI9881C_SWITCH_PAGE_INSTR(4),
	ILI9881C_COMMAND_INSTR(0x00, 0x80),
	ILI9881C_COMMAND_INSTR(0x70, 0x00),
	ILI9881C_COMMAND_INSTR(0x71, 0x00),
	ILI9881C_COMMAND_INSTR(0x66, 0xfe),
	ILI9881C_COMMAND_INSTR(0x82, 0x15),
	ILI9881C_COMMAND_INSTR(0x84, 0x15),
	ILI9881C_COMMAND_INSTR(0x85, 0x15),
	ILI9881C_COMMAND_INSTR(0x3a, 0x24),
	ILI9881C_COMMAND_INSTR(0x32, 0xac),
	ILI9881C_COMMAND_INSTR(0x8c, 0x80),
	ILI9881C_COMMAND_INSTR(0x3c, 0xf5),
	ILI9881C_COMMAND_INSTR(0x88, 0x33),
	ILI9881C_SWITCH_PAGE_INSTR(1),
	ILI9881C_COMMAND_INSTR(0x22, 0x0a),
	ILI9881C_COMMAND_INSTR(0x31, 0x00),
	ILI9881C_COMMAND_INSTR(0x53, 0x78),
	ILI9881C_COMMAND_INSTR(0x55, 0x7b),
	ILI9881C_COMMAND_INSTR(0x60, 0x20),
	ILI9881C_COMMAND_INSTR(0x61, 0x00),
	ILI9881C_COMMAND_INSTR(0x62, 0x0d),
	ILI9881C_COMMAND_INSTR(0x63, 0x00),
	ILI9881C_COMMAND_INSTR(0xa0, 0x00),
	ILI9881C_COMMAND_INSTR(0xa1, 0x10),
	ILI9881C_COMMAND_INSTR(0xa2, 0x1c),
	ILI9881C_COMMAND_INSTR(0xa3, 0x13),
	ILI9881C_COMMAND_INSTR(0xa4, 0x15),
	ILI9881C_COMMAND_INSTR(0xa5, 0x26),
	ILI9881C_COMMAND_INSTR(0xa6, 0x1a),
	ILI9881C_COMMAND_INSTR(0xa7, 0x1d),
	ILI9881C_COMMAND_INSTR(0xa8, 0x67),
	ILI9881C_COMMAND_INSTR(0xa9, 0x1c),
	ILI9881C_COMMAND_INSTR(0xaa, 0x29),
	ILI9881C_COMMAND_INSTR(0xab, 0x5b),
	ILI9881C_COMMAND_INSTR(0xac, 0x26),
	ILI9881C_COMMAND_INSTR(0xad, 0x28),
	ILI9881C_COMMAND_INSTR(0xae, 0x5c),
	ILI9881C_COMMAND_INSTR(0xaf, 0x30),
	ILI9881C_COMMAND_INSTR(0xb0, 0x31),
	ILI9881C_COMMAND_INSTR(0xb1, 0x32),
	ILI9881C_COMMAND_INSTR(0xb2, 0x00),
	ILI9881C_COMMAND_INSTR(0xb1, 0x2e),
	ILI9881C_COMMAND_INSTR(0xb2, 0x32),
	ILI9881C_COMMAND_INSTR(0xb3, 0x00),
	ILI9881C_COMMAND_INSTR(0xb6, 0x02),
	ILI9881C_COMMAND_INSTR(0xb7, 0x03),
	ILI9881C_COMMAND_INSTR(0xc0, 0x00),
	ILI9881C_COMMAND_INSTR(0xc1, 0x10),
	ILI9881C_COMMAND_INSTR(0xc2, 0x1c),
	ILI9881C_COMMAND_INSTR(0xc3, 0x13),
	ILI9881C_COMMAND_INSTR(0xc4, 0x15),
	ILI9881C_COMMAND_INSTR(0xc5, 0x26),
	ILI9881C_COMMAND_INSTR(0xc6, 0x1a),
	ILI9881C_COMMAND_INSTR(0xc7, 0x1d),
	ILI9881C_COMMAND_INSTR(0xc8, 0x67),
	ILI9881C_COMMAND_INSTR(0xc9, 0x1c),
	ILI9881C_COMMAND_INSTR(0xca, 0x29),
	ILI9881C_COMMAND_INSTR(0xcb, 0x5b),
	ILI9881C_COMMAND_INSTR(0xcc, 0x26),
	ILI9881C_COMMAND_INSTR(0xcd, 0x28),
	ILI9881C_COMMAND_INSTR(0xce, 0x5c),
	ILI9881C_COMMAND_INSTR(0xcf, 0x30),
	ILI9881C_COMMAND_INSTR(0xd0, 0x31),
	ILI9881C_COMMAND_INSTR(0xd1, 0x2e),
	ILI9881C_COMMAND_INSTR(0xd2, 0x32),
	ILI9881C_COMMAND_INSTR(0xd3, 0x00),
	ILI9881C_SWITCH_PAGE_INSTR(0),
};

static const struct ili9881c_instr w552946ab_init[] = {
	ILI9881C_SWITCH_PAGE_INSTR(3),
	ILI9881C_COMMAND_INSTR(0x01, 0x00),
	ILI9881C_COMMAND_INSTR(0x02, 0x00),
	ILI9881C_COMMAND_INSTR(0x03, 0x53),
	ILI9881C_COMMAND_INSTR(0x04, 0x53),
	ILI9881C_COMMAND_INSTR(0x05, 0x13),
	ILI9881C_COMMAND_INSTR(0x06, 0x04),
	ILI9881C_COMMAND_INSTR(0x07, 0x02),
	ILI9881C_COMMAND_INSTR(0x08, 0x02),
	ILI9881C_COMMAND_INSTR(0x09, 0x00),
	ILI9881C_COMMAND_INSTR(0x0a, 0x00),
	ILI9881C_COMMAND_INSTR(0x0b, 0x00),
	ILI9881C_COMMAND_INSTR(0x0c, 0x00),
	ILI9881C_COMMAND_INSTR(0x0d, 0x00),
	ILI9881C_COMMAND_INSTR(0x0e, 0x00),
	ILI9881C_COMMAND_INSTR(0x0f, 0x00),

	ILI9881C_COMMAND_INSTR(0x10, 0x00),
	ILI9881C_COMMAND_INSTR(0x11, 0x00),
	ILI9881C_COMMAND_INSTR(0x12, 0x00),
	ILI9881C_COMMAND_INSTR(0x13, 0x00),
	ILI9881C_COMMAND_INSTR(0x14, 0x00),
	ILI9881C_COMMAND_INSTR(0x15, 0x08),
	ILI9881C_COMMAND_INSTR(0x16, 0x10),
	ILI9881C_COMMAND_INSTR(0x17, 0x00),
	ILI9881C_COMMAND_INSTR(0x18, 0x08),
	ILI9881C_COMMAND_INSTR(0x19, 0x00),
	ILI9881C_COMMAND_INSTR(0x1a, 0x00),
	ILI9881C_COMMAND_INSTR(0x1b, 0x00),
	ILI9881C_COMMAND_INSTR(0x1c, 0x00),
	ILI9881C_COMMAND_INSTR(0x1d, 0x00),
	ILI9881C_COMMAND_INSTR(0x1e, 0xc0),
	ILI9881C_COMMAND_INSTR(0x1f, 0x80),

	ILI9881C_COMMAND_INSTR(0x20, 0x02),
	ILI9881C_COMMAND_INSTR(0x21, 0x09),
	ILI9881C_COMMAND_INSTR(0x22, 0x00),
	ILI9881C_COMMAND_INSTR(0x23, 0x00),
	ILI9881C_COMMAND_INSTR(0x24, 0x00),
	ILI9881C_COMMAND_INSTR(0x25, 0x00),
	ILI9881C_COMMAND_INSTR(0x26, 0x00),
	ILI9881C_COMMAND_INSTR(0x27, 0x00),
	ILI9881C_COMMAND_INSTR(0x28, 0x55),
	ILI9881C_COMMAND_INSTR(0x29, 0x03),
	ILI9881C_COMMAND_INSTR(0x2a, 0x00),
	ILI9881C_COMMAND_INSTR(0x2b, 0x00),
	ILI9881C_COMMAND_INSTR(0x2c, 0x00),
	ILI9881C_COMMAND_INSTR(0x2d, 0x00),
	ILI9881C_COMMAND_INSTR(0x2e, 0x00),
	ILI9881C_COMMAND_INSTR(0x2f, 0x00),

	ILI9881C_COMMAND_INSTR(0x30, 0x00),
	ILI9881C_COMMAND_INSTR(0x31, 0x00),
	ILI9881C_COMMAND_INSTR(0x32, 0x00),
	ILI9881C_COMMAND_INSTR(0x33, 0x00),
	ILI9881C_COMMAND_INSTR(0x34, 0x04),
	ILI9881C_COMMAND_INSTR(0x35, 0x05),
	ILI9881C_COMMAND_INSTR(0x36, 0x05),
	ILI9881C_COMMAND_INSTR(0x37, 0x00),
	ILI9881C_COMMAND_INSTR(0x38, 0x3c),
	ILI9881C_COMMAND_INSTR(0x39, 0x35),
	ILI9881C_COMMAND_INSTR(0x3a, 0x00),
	ILI9881C_COMMAND_INSTR(0x3b, 0x40),
	ILI9881C_COMMAND_INSTR(0x3c, 0x00),
	ILI9881C_COMMAND_INSTR(0x3d, 0x00),
	ILI9881C_COMMAND_INSTR(0x3e, 0x00),
	ILI9881C_COMMAND_INSTR(0x3f, 0x00),

	ILI9881C_COMMAND_INSTR(0x40, 0x00),
	ILI9881C_COMMAND_INSTR(0x41, 0x88),
	ILI9881C_COMMAND_INSTR(0x42, 0x00),
	ILI9881C_COMMAND_INSTR(0x43, 0x00),
	ILI9881C_COMMAND_INSTR(0x44, 0x1f),

	ILI9881C_COMMAND_INSTR(0x50, 0x01),
	ILI9881C_COMMAND_INSTR(0x51, 0x23),
	ILI9881C_COMMAND_INSTR(0x52, 0x45),
	ILI9881C_COMMAND_INSTR(0x53, 0x67),
	ILI9881C_COMMAND_INSTR(0x54, 0x89),
	ILI9881C_COMMAND_INSTR(0x55, 0xab),
	ILI9881C_COMMAND_INSTR(0x56, 0x01),
	ILI9881C_COMMAND_INSTR(0x57, 0x23),
	ILI9881C_COMMAND_INSTR(0x58, 0x45),
	ILI9881C_COMMAND_INSTR(0x59, 0x67),
	ILI9881C_COMMAND_INSTR(0x5a, 0x89),
	ILI9881C_COMMAND_INSTR(0x5b, 0xab),
	ILI9881C_COMMAND_INSTR(0x5c, 0xcd),
	ILI9881C_COMMAND_INSTR(0x5d, 0xef),
	ILI9881C_COMMAND_INSTR(0x5e, 0x03),
	ILI9881C_COMMAND_INSTR(0x5f, 0x14),

	ILI9881C_COMMAND_INSTR(0x60, 0x15),
	ILI9881C_COMMAND_INSTR(0x61, 0x0c),
	ILI9881C_COMMAND_INSTR(0x62, 0x0d),
	ILI9881C_COMMAND_INSTR(0x63, 0x0e),
	ILI9881C_COMMAND_INSTR(0x64, 0x0f),
	ILI9881C_COMMAND_INSTR(0x65, 0x10),
	ILI9881C_COMMAND_INSTR(0x66, 0x11),
	ILI9881C_COMMAND_INSTR(0x67, 0x08),
	ILI9881C_COMMAND_INSTR(0x68, 0x02),
	ILI9881C_COMMAND_INSTR(0x69, 0x0a),
	ILI9881C_COMMAND_INSTR(0x6a, 0x02),
	ILI9881C_COMMAND_INSTR(0x6b, 0x02),
	ILI9881C_COMMAND_INSTR(0x6c, 0x02),
	ILI9881C_COMMAND_INSTR(0x6d, 0x02),
	ILI9881C_COMMAND_INSTR(0x6e, 0x02),
	ILI9881C_COMMAND_INSTR(0x6f, 0x02),

	ILI9881C_COMMAND_INSTR(0x70, 0x02),
	ILI9881C_COMMAND_INSTR(0x71, 0x02),
	ILI9881C_COMMAND_INSTR(0x72, 0x06),
	ILI9881C_COMMAND_INSTR(0x73, 0x02),
	ILI9881C_COMMAND_INSTR(0x74, 0x02),
	ILI9881C_COMMAND_INSTR(0x75, 0x14),
	ILI9881C_COMMAND_INSTR(0x76, 0x15),
	ILI9881C_COMMAND_INSTR(0x77, 0x0f),
	ILI9881C_COMMAND_INSTR(0x78, 0x0e),
	ILI9881C_COMMAND_INSTR(0x79, 0x0d),
	ILI9881C_COMMAND_INSTR(0x7a, 0x0c),
	ILI9881C_COMMAND_INSTR(0x7b, 0x11),
	ILI9881C_COMMAND_INSTR(0x7c, 0x10),
	ILI9881C_COMMAND_INSTR(0x7d, 0x06),
	ILI9881C_COMMAND_INSTR(0x7e, 0x02),
	ILI9881C_COMMAND_INSTR(0x7f, 0x0a),

	ILI9881C_COMMAND_INSTR(0x80, 0x02),
	ILI9881C_COMMAND_INSTR(0x81, 0x02),
	ILI9881C_COMMAND_INSTR(0x82, 0x02),
	ILI9881C_COMMAND_INSTR(0x83, 0x02),
	ILI9881C_COMMAND_INSTR(0x84, 0x02),
	ILI9881C_COMMAND_INSTR(0x85, 0x02),
	ILI9881C_COMMAND_INSTR(0x86, 0x02),
	ILI9881C_COMMAND_INSTR(0x87, 0x02),
	ILI9881C_COMMAND_INSTR(0x88, 0x08),
	ILI9881C_COMMAND_INSTR(0x89, 0x02),
	ILI9881C_COMMAND_INSTR(0x8a, 0x02),

	ILI9881C_SWITCH_PAGE_INSTR(4),
	ILI9881C_COMMAND_INSTR(0x00, 0x80),
	ILI9881C_COMMAND_INSTR(0x70, 0x00),
	ILI9881C_COMMAND_INSTR(0x71, 0x00),
	ILI9881C_COMMAND_INSTR(0x66, 0xfe),
	ILI9881C_COMMAND_INSTR(0x82, 0x15),
	ILI9881C_COMMAND_INSTR(0x84, 0x15),
	ILI9881C_COMMAND_INSTR(0x85, 0x15),
	ILI9881C_COMMAND_INSTR(0x3a, 0x24),
	ILI9881C_COMMAND_INSTR(0x32, 0xac),
	ILI9881C_COMMAND_INSTR(0x8c, 0x80),
	ILI9881C_COMMAND_INSTR(0x3c, 0xf5),
	ILI9881C_COMMAND_INSTR(0x88, 0x33),

	ILI9881C_SWITCH_PAGE_INSTR(1),
	ILI9881C_COMMAND_INSTR(0x22, 0x0a),
	ILI9881C_COMMAND_INSTR(0x31, 0x00),
	ILI9881C_COMMAND_INSTR(0x53, 0x78),
	ILI9881C_COMMAND_INSTR(0x50, 0x5b),
	ILI9881C_COMMAND_INSTR(0x51, 0x5b),
	ILI9881C_COMMAND_INSTR(0x60, 0x20),
	ILI9881C_COMMAND_INSTR(0x61, 0x00),
	ILI9881C_COMMAND_INSTR(0x62, 0x0d),
	ILI9881C_COMMAND_INSTR(0x63, 0x00),

	ILI9881C_COMMAND_INSTR(0xa0, 0x00),
	ILI9881C_COMMAND_INSTR(0xa1, 0x10),
	ILI9881C_COMMAND_INSTR(0xa2, 0x1c),
	ILI9881C_COMMAND_INSTR(0xa3, 0x13),
	ILI9881C_COMMAND_INSTR(0xa4, 0x15),
	ILI9881C_COMMAND_INSTR(0xa5, 0x26),
	ILI9881C_COMMAND_INSTR(0xa6, 0x1a),
	ILI9881C_COMMAND_INSTR(0xa7, 0x1d),
	ILI9881C_COMMAND_INSTR(0xa8, 0x67),
	ILI9881C_COMMAND_INSTR(0xa9, 0x1c),
	ILI9881C_COMMAND_INSTR(0xaa, 0x29),
	ILI9881C_COMMAND_INSTR(0xab, 0x5b),
	ILI9881C_COMMAND_INSTR(0xac, 0x26),
	ILI9881C_COMMAND_INSTR(0xad, 0x28),
	ILI9881C_COMMAND_INSTR(0xae, 0x5c),
	ILI9881C_COMMAND_INSTR(0xaf, 0x30),
	ILI9881C_COMMAND_INSTR(0xb0, 0x31),
	ILI9881C_COMMAND_INSTR(0xb1, 0x2e),
	ILI9881C_COMMAND_INSTR(0xb2, 0x32),
	ILI9881C_COMMAND_INSTR(0xb3, 0x00),

	ILI9881C_COMMAND_INSTR(0xc0, 0x00),
	ILI9881C_COMMAND_INSTR(0xc1, 0x10),
	ILI9881C_COMMAND_INSTR(0xc2, 0x1c),
	ILI9881C_COMMAND_INSTR(0xc3, 0x13),
	ILI9881C_COMMAND_INSTR(0xc4, 0x15),
	ILI9881C_COMMAND_INSTR(0xc5, 0x26),
	ILI9881C_COMMAND_INSTR(0xc6, 0x1a),
	ILI9881C_COMMAND_INSTR(0xc7, 0x1d),
	ILI9881C_COMMAND_INSTR(0xc8, 0x67),
	ILI9881C_COMMAND_INSTR(0xc9, 0x1c),
	ILI9881C_COMMAND_INSTR(0xca, 0x29),
	ILI9881C_COMMAND_INSTR(0xcb, 0x5b),
	ILI9881C_COMMAND_INSTR(0xcc, 0x26),
	ILI9881C_COMMAND_INSTR(0xcd, 0x28),
	ILI9881C_COMMAND_INSTR(0xce, 0x5c),
	ILI9881C_COMMAND_INSTR(0xcf, 0x30),
	ILI9881C_COMMAND_INSTR(0xd0, 0x31),
	ILI9881C_COMMAND_INSTR(0xd1, 0x2e),
	ILI9881C_COMMAND_INSTR(0xd2, 0x32),
	ILI9881C_COMMAND_INSTR(0xd3, 0x00),
	ILI9881C_SWITCH_PAGE_INSTR(0),
};

static const struct ili9881c_instr am8001280g_init[] = {
	ILI9881C_SWITCH_PAGE_INSTR(3),
	ILI9881C_COMMAND_INSTR(0x01, 0x00),
	ILI9881C_COMMAND_INSTR(0x02, 0x00),
	ILI9881C_COMMAND_INSTR(0x03, 0x73),
	ILI9881C_COMMAND_INSTR(0x04, 0xd3),
	ILI9881C_COMMAND_INSTR(0x05, 0x00),
	ILI9881C_COMMAND_INSTR(0x06, 0x0a),
	ILI9881C_COMMAND_INSTR(0x07, 0x0e),
	ILI9881C_COMMAND_INSTR(0x08, 0x00),
	ILI9881C_COMMAND_INSTR(0x09, 0x01),
	ILI9881C_COMMAND_INSTR(0x0a, 0x01),
	ILI9881C_COMMAND_INSTR(0x0b, 0x01),
	ILI9881C_COMMAND_INSTR(0x0c, 0x01),
	ILI9881C_COMMAND_INSTR(0x0d, 0x01),
	ILI9881C_COMMAND_INSTR(0x0e, 0x01),
	ILI9881C_COMMAND_INSTR(0x0f, 0x01),
	ILI9881C_COMMAND_INSTR(0x10, 0x01),
	ILI9881C_COMMAND_INSTR(0x11, 0x00),
	ILI9881C_COMMAND_INSTR(0x12, 0x00),
	ILI9881C_COMMAND_INSTR(0x13, 0x00),
	ILI9881C_COMMAND_INSTR(0x14, 0x00),
	ILI9881C_COMMAND_INSTR(0x15, 0x00),
	ILI9881C_COMMAND_INSTR(0x16, 0x00),
	ILI9881C_COMMAND_INSTR(0x17, 0x00),
	ILI9881C_COMMAND_INSTR(0x18, 0x00),
	ILI9881C_COMMAND_INSTR(0x19, 0x00),
	ILI9881C_COMMAND_INSTR(0x1a, 0x00),
	ILI9881C_COMMAND_INSTR(0x1b, 0x00),
	ILI9881C_COMMAND_INSTR(0x1c, 0x00),
	ILI9881C_COMMAND_INSTR(0x1d, 0x00),
	ILI9881C_COMMAND_INSTR(0x1e, 0x40),
	ILI9881C_COMMAND_INSTR(0x1f, 0x80),
	ILI9881C_COMMAND_INSTR(0x20, 0x06),
	ILI9881C_COMMAND_INSTR(0x21, 0x01),
	ILI9881C_COMMAND_INSTR(0x22, 0x00),
	ILI9881C_COMMAND_INSTR(0x23, 0x00),
	ILI9881C_COMMAND_INSTR(0x24, 0x00),
	ILI9881C_COMMAND_INSTR(0x25, 0x00),
	ILI9881C_COMMAND_INSTR(0x26, 0x00),
	ILI9881C_COMMAND_INSTR(0x27, 0x00),
	ILI9881C_COMMAND_INSTR(0x28, 0x33),
	ILI9881C_COMMAND_INSTR(0x29, 0x03),
	ILI9881C_COMMAND_INSTR(0x2a, 0x00),
	ILI9881C_COMMAND_INSTR(0x2b, 0x00),
	ILI9881C_COMMAND_INSTR(0x2c, 0x00),
	ILI9881C_COMMAND_INSTR(0x2d, 0x00),
	ILI9881C_COMMAND_INSTR(0x2e, 0x00),
	ILI9881C_COMMAND_INSTR(0x2f, 0x00),
	ILI9881C_COMMAND_INSTR(0x30, 0x00),
	ILI9881C_COMMAND_INSTR(0x31, 0x00),
	ILI9881C_COMMAND_INSTR(0x32, 0x00),
	ILI9881C_COMMAND_INSTR(0x33, 0x00),
	ILI9881C_COMMAND_INSTR(0x34, 0x03),
	ILI9881C_COMMAND_INSTR(0x35, 0x00),
	ILI9881C_COMMAND_INSTR(0x36, 0x03),
	ILI9881C_COMMAND_INSTR(0x37, 0x00),
	ILI9881C_COMMAND_INSTR(0x38, 0x00),
	ILI9881C_COMMAND_INSTR(0x39, 0x00),
	ILI9881C_COMMAND_INSTR(0x3a, 0x40),
	ILI9881C_COMMAND_INSTR(0x3b, 0x40),
	ILI9881C_COMMAND_INSTR(0x3c, 0x00),
	ILI9881C_COMMAND_INSTR(0x3d, 0x00),
	ILI9881C_COMMAND_INSTR(0x3e, 0x00),
	ILI9881C_COMMAND_INSTR(0x3f, 0x00),
	ILI9881C_COMMAND_INSTR(0x40, 0x00),
	ILI9881C_COMMAND_INSTR(0x41, 0x00),
	ILI9881C_COMMAND_INSTR(0x42, 0x00),
	ILI9881C_COMMAND_INSTR(0x43, 0x00),
	ILI9881C_COMMAND_INSTR(0x44, 0x00),

	ILI9881C_COMMAND_INSTR(0x50, 0x01),
	ILI9881C_COMMAND_INSTR(0x51, 0x23),
	ILI9881C_COMMAND_INSTR(0x52, 0x45),
	ILI9881C_COMMAND_INSTR(0x53, 0x67),
	ILI9881C_COMMAND_INSTR(0x54, 0x89),
	ILI9881C_COMMAND_INSTR(0x55, 0xab),
	ILI9881C_COMMAND_INSTR(0x56, 0x01),
	ILI9881C_COMMAND_INSTR(0x57, 0x23),
	ILI9881C_COMMAND_INSTR(0x58, 0x45),
	ILI9881C_COMMAND_INSTR(0x59, 0x67),
	ILI9881C_COMMAND_INSTR(0x5a, 0x89),
	ILI9881C_COMMAND_INSTR(0x5b, 0xab),
	ILI9881C_COMMAND_INSTR(0x5c, 0xcd),
	ILI9881C_COMMAND_INSTR(0x5d, 0xef),

	ILI9881C_COMMAND_INSTR(0x5e, 0x11),
	ILI9881C_COMMAND_INSTR(0x5f, 0x02),
	ILI9881C_COMMAND_INSTR(0x60, 0x00),
	ILI9881C_COMMAND_INSTR(0x61, 0x01),
	ILI9881C_COMMAND_INSTR(0x62, 0x0d),
	ILI9881C_COMMAND_INSTR(0x63, 0x0c),
	ILI9881C_COMMAND_INSTR(0x64, 0x0f),
	ILI9881C_COMMAND_INSTR(0x65, 0x0e),
	ILI9881C_COMMAND_INSTR(0x66, 0x06),
	ILI9881C_COMMAND_INSTR(0x67, 0x07),
	ILI9881C_COMMAND_INSTR(0x68, 0x02),
	ILI9881C_COMMAND_INSTR(0x69, 0x02),
	ILI9881C_COMMAND_INSTR(0x6a, 0x08),
	ILI9881C_COMMAND_INSTR(0x6b, 0x02),
	ILI9881C_COMMAND_INSTR(0x6c, 0x02),
	ILI9881C_COMMAND_INSTR(0x6d, 0x02),
	ILI9881C_COMMAND_INSTR(0x6e, 0x02),
	ILI9881C_COMMAND_INSTR(0x6f, 0x02),
	ILI9881C_COMMAND_INSTR(0x70, 0x02),
	ILI9881C_COMMAND_INSTR(0x71, 0x02),
	ILI9881C_COMMAND_INSTR(0x72, 0x02),
	ILI9881C_COMMAND_INSTR(0x73, 0x02),
	ILI9881C_COMMAND_INSTR(0x74, 0x02),
	ILI9881C_COMMAND_INSTR(0x75, 0x02),
	ILI9881C_COMMAND_INSTR(0x76, 0x00),
	ILI9881C_COMMAND_INSTR(0x77, 0x01),
	ILI9881C_COMMAND_INSTR(0x78, 0x0d),
	ILI9881C_COMMAND_INSTR(0x79, 0x0c),
	ILI9881C_COMMAND_INSTR(0x7a, 0x0f),
	ILI9881C_COMMAND_INSTR(0x7b, 0x0e),
	ILI9881C_COMMAND_INSTR(0x7c, 0x06),
	ILI9881C_COMMAND_INSTR(0x7d, 0x07),
	ILI9881C_COMMAND_INSTR(0x7e, 0x02),
	ILI9881C_COMMAND_INSTR(0x7f, 0x02),
	ILI9881C_COMMAND_INSTR(0x80, 0x08),
	ILI9881C_COMMAND_INSTR(0x81, 0x02),
	ILI9881C_COMMAND_INSTR(0x82, 0x02),
	ILI9881C_COMMAND_INSTR(0x83, 0x02),
	ILI9881C_COMMAND_INSTR(0x84, 0x02),
	ILI9881C_COMMAND_INSTR(0x85, 0x02),
	ILI9881C_COMMAND_INSTR(0x86, 0x02),
	ILI9881C_COMMAND_INSTR(0x87, 0x02),
	ILI9881C_COMMAND_INSTR(0x88, 0x02),
	ILI9881C_COMMAND_INSTR(0x89, 0x02),
	ILI9881C_COMMAND_INSTR(0x8a, 0x02),

	ILI9881C_SWITCH_PAGE_INSTR(4),
	ILI9881C_COMMAND_INSTR(0x6c, 0x15),
	ILI9881C_COMMAND_INSTR(0x6e, 0x30),
	ILI9881C_COMMAND_INSTR(0x6f, 0x33),
	ILI9881C_COMMAND_INSTR(0x8d, 0x15),
	ILI9881C_COMMAND_INSTR(0x3a, 0xa4),
	ILI9881C_COMMAND_INSTR(0x87, 0xba),
	ILI9881C_COMMAND_INSTR(0x26, 0x76),
	ILI9881C_COMMAND_INSTR(0xb2, 0xd1),

	ILI9881C_SWITCH_PAGE_INSTR(1),
	ILI9881C_COMMAND_INSTR(0x22, 0x0a),
	ILI9881C_COMMAND_INSTR(0x31, 0x0b),
	ILI9881C_COMMAND_INSTR(0x50, 0xa5),
	ILI9881C_COMMAND_INSTR(0x51, 0xa0),
	ILI9881C_COMMAND_INSTR(0x53, 0x70),
	ILI9881C_COMMAND_INSTR(0x55, 0x7a),
	ILI9881C_COMMAND_INSTR(0x60, 0x14),

	ILI9881C_COMMAND_INSTR(0xa0, 0x00),
	ILI9881C_COMMAND_INSTR(0xa1, 0x53),
	ILI9881C_COMMAND_INSTR(0xa2, 0x50),
	ILI9881C_COMMAND_INSTR(0xa3, 0x20),
	ILI9881C_COMMAND_INSTR(0xa4, 0x27),
	ILI9881C_COMMAND_INSTR(0xa5, 0x33),
	ILI9881C_COMMAND_INSTR(0xa6, 0x25),
	ILI9881C_COMMAND_INSTR(0xa7, 0x25),
	ILI9881C_COMMAND_INSTR(0xa8, 0xd4),
	ILI9881C_COMMAND_INSTR(0xa9, 0x1a),
	ILI9881C_COMMAND_INSTR(0xaa, 0x2b),
	ILI9881C_COMMAND_INSTR(0xab, 0xb5),
	ILI9881C_COMMAND_INSTR(0xac, 0x19),
	ILI9881C_COMMAND_INSTR(0xad, 0x18),
	ILI9881C_COMMAND_INSTR(0xae, 0x53),
	ILI9881C_COMMAND_INSTR(0xaf, 0x1a),
	ILI9881C_COMMAND_INSTR(0xb0, 0x25),
	ILI9881C_COMMAND_INSTR(0xb1, 0x62),
	ILI9881C_COMMAND_INSTR(0xb2, 0x6a),
	ILI9881C_COMMAND_INSTR(0xb3, 0x31),

	ILI9881C_COMMAND_INSTR(0xc0, 0x00),
	ILI9881C_COMMAND_INSTR(0xc1, 0x53),
	ILI9881C_COMMAND_INSTR(0xc2, 0x50),
	ILI9881C_COMMAND_INSTR(0xc3, 0x20),
	ILI9881C_COMMAND_INSTR(0xc4, 0x27),
	ILI9881C_COMMAND_INSTR(0xc5, 0x33),
	ILI9881C_COMMAND_INSTR(0xc6, 0x25),
	ILI9881C_COMMAND_INSTR(0xc7, 0x25),
	ILI9881C_COMMAND_INSTR(0xc8, 0xd4),
	ILI9881C_COMMAND_INSTR(0xc9, 0x1a),
	ILI9881C_COMMAND_INSTR(0xca, 0x2b),
	ILI9881C_COMMAND_INSTR(0xcb, 0xb5),
	ILI9881C_COMMAND_INSTR(0xcc, 0x19),
	ILI9881C_COMMAND_INSTR(0xcd, 0x18),
	ILI9881C_COMMAND_INSTR(0xce, 0x53),
	ILI9881C_COMMAND_INSTR(0xcf, 0x1a),
	ILI9881C_COMMAND_INSTR(0xd0, 0x25),
	ILI9881C_COMMAND_INSTR(0xd1, 0x62),
	ILI9881C_COMMAND_INSTR(0xd2, 0x6a),
	ILI9881C_COMMAND_INSTR(0xd3, 0x31),
	ILI9881C_SWITCH_PAGE_INSTR(0),
	ILI9881C_COMMAND_INSTR(MIPI_DCS_WRITE_CONTROL_DISPLAY, 0x2c),
	ILI9881C_COMMAND_INSTR(MIPI_DCS_WRITE_POWER_SAVE, 0x00),
};

static const struct ili9881c_instr rpi_5inch_init[] = {
	ILI9881C_SWITCH_PAGE_INSTR(3),
	ILI9881C_COMMAND_INSTR(0x01, 0x00),
	ILI9881C_COMMAND_INSTR(0x02, 0x00),
	ILI9881C_COMMAND_INSTR(0x03, 0x73),
	ILI9881C_COMMAND_INSTR(0x04, 0x73),
	ILI9881C_COMMAND_INSTR(0x05, 0x00),
	ILI9881C_COMMAND_INSTR(0x06, 0x06),
	ILI9881C_COMMAND_INSTR(0x07, 0x02),
	ILI9881C_COMMAND_INSTR(0x08, 0x00),
	ILI9881C_COMMAND_INSTR(0x09, 0x01),
	ILI9881C_COMMAND_INSTR(0x0a, 0x01),
	ILI9881C_COMMAND_INSTR(0x0b, 0x01),
	ILI9881C_COMMAND_INSTR(0x0c, 0x01),
	ILI9881C_COMMAND_INSTR(0x0d, 0x01),
	ILI9881C_COMMAND_INSTR(0x0e, 0x01),
	ILI9881C_COMMAND_INSTR(0x0f, 0x01),
	ILI9881C_COMMAND_INSTR(0x10, 0x01),
	ILI9881C_COMMAND_INSTR(0x11, 0x00),
	ILI9881C_COMMAND_INSTR(0x12, 0x00),
	ILI9881C_COMMAND_INSTR(0x13, 0x01),
	ILI9881C_COMMAND_INSTR(0x14, 0x00),
	ILI9881C_COMMAND_INSTR(0x15, 0x00),
	ILI9881C_COMMAND_INSTR(0x16, 0x00),
	ILI9881C_COMMAND_INSTR(0x17, 0x00),
	ILI9881C_COMMAND_INSTR(0x18, 0x00),
	ILI9881C_COMMAND_INSTR(0x19, 0x00),
	ILI9881C_COMMAND_INSTR(0x1a, 0x00),
	ILI9881C_COMMAND_INSTR(0x1b, 0x00),
	ILI9881C_COMMAND_INSTR(0x1c, 0x00),
	ILI9881C_COMMAND_INSTR(0x1d, 0x00),
	ILI9881C_COMMAND_INSTR(0x1e, 0xc0),
	ILI9881C_COMMAND_INSTR(0x1f, 0x80),
	ILI9881C_COMMAND_INSTR(0x20, 0x04),
	ILI9881C_COMMAND_INSTR(0x21, 0x03),
	ILI9881C_COMMAND_INSTR(0x22, 0x00),
	ILI9881C_COMMAND_INSTR(0x23, 0x00),
	ILI9881C_COMMAND_INSTR(0x24, 0x00),
	ILI9881C_COMMAND_INSTR(0x25, 0x00),
	ILI9881C_COMMAND_INSTR(0x26, 0x00),
	ILI9881C_COMMAND_INSTR(0x27, 0x00),
	ILI9881C_COMMAND_INSTR(0x28, 0x33),
	ILI9881C_COMMAND_INSTR(0x29, 0x03),
	ILI9881C_COMMAND_INSTR(0x2a, 0x00),
	ILI9881C_COMMAND_INSTR(0x2b, 0x00),
	ILI9881C_COMMAND_INSTR(0x2c, 0x00),
	ILI9881C_COMMAND_INSTR(0x2d, 0x00),
	ILI9881C_COMMAND_INSTR(0x2e, 0x00),
	ILI9881C_COMMAND_INSTR(0x2f, 0x00),
	ILI9881C_COMMAND_INSTR(0x30, 0x00),
	ILI9881C_COMMAND_INSTR(0x31, 0x00),
	ILI9881C_COMMAND_INSTR(0x32, 0x00),
	ILI9881C_COMMAND_INSTR(0x33, 0x00),
	ILI9881C_COMMAND_INSTR(0x34, 0x03),
	ILI9881C_COMMAND_INSTR(0x35, 0x00),
	ILI9881C_COMMAND_INSTR(0x36, 0x03),
	ILI9881C_COMMAND_INSTR(0x37, 0x00),
	ILI9881C_COMMAND_INSTR(0x38, 0x00),
	ILI9881C_COMMAND_INSTR(0x39, 0x00),
	ILI9881C_COMMAND_INSTR(0x3a, 0x00),
	ILI9881C_COMMAND_INSTR(0x3b, 0x00),
	ILI9881C_COMMAND_INSTR(0x3c, 0x00),
	ILI9881C_COMMAND_INSTR(0x3d, 0x00),
	ILI9881C_COMMAND_INSTR(0x3e, 0x00),
	ILI9881C_COMMAND_INSTR(0x3f, 0x00),
	ILI9881C_COMMAND_INSTR(0x40, 0x00),
	ILI9881C_COMMAND_INSTR(0x41, 0x00),
	ILI9881C_COMMAND_INSTR(0x42, 0x00),
	ILI9881C_COMMAND_INSTR(0x43, 0x00),
	ILI9881C_COMMAND_INSTR(0x44, 0x00),
	ILI9881C_COMMAND_INSTR(0x50, 0x01),
	ILI9881C_COMMAND_INSTR(0x51, 0x23),
	ILI9881C_COMMAND_INSTR(0x52, 0x45),
	ILI9881C_COMMAND_INSTR(0x53, 0x67),
	ILI9881C_COMMAND_INSTR(0x54, 0x89),
	ILI9881C_COMMAND_INSTR(0x55, 0xab),
	ILI9881C_COMMAND_INSTR(0x56, 0x01),
	ILI9881C_COMMAND_INSTR(0x57, 0x23),
	ILI9881C_COMMAND_INSTR(0x58, 0x45),
	ILI9881C_COMMAND_INSTR(0x59, 0x67),
	ILI9881C_COMMAND_INSTR(0x5a, 0x89),
	ILI9881C_COMMAND_INSTR(0x5b, 0xab),
	ILI9881C_COMMAND_INSTR(0x5c, 0xcd),
	ILI9881C_COMMAND_INSTR(0x5d, 0xef),
	ILI9881C_COMMAND_INSTR(0x5e, 0x10),
	ILI9881C_COMMAND_INSTR(0x5f, 0x09),
	ILI9881C_COMMAND_INSTR(0x60, 0x08),
	ILI9881C_COMMAND_INSTR(0x61, 0x0f),
	ILI9881C_COMMAND_INSTR(0x62, 0x0e),
	ILI9881C_COMMAND_INSTR(0x63, 0x0d),
	ILI9881C_COMMAND_INSTR(0x64, 0x0c),
	ILI9881C_COMMAND_INSTR(0x65, 0x02),
	ILI9881C_COMMAND_INSTR(0x66, 0x02),
	ILI9881C_COMMAND_INSTR(0x67, 0x02),
	ILI9881C_COMMAND_INSTR(0x68, 0x02),
	ILI9881C_COMMAND_INSTR(0x69, 0x02),
	ILI9881C_COMMAND_INSTR(0x6a, 0x02),
	ILI9881C_COMMAND_INSTR(0x6b, 0x02),
	ILI9881C_COMMAND_INSTR(0x6c, 0x02),
	ILI9881C_COMMAND_INSTR(0x6d, 0x02),
	ILI9881C_COMMAND_INSTR(0x6e, 0x02),
	ILI9881C_COMMAND_INSTR(0x6f, 0x02),
	ILI9881C_COMMAND_INSTR(0x70, 0x02),
	ILI9881C_COMMAND_INSTR(0x71, 0x06),
	ILI9881C_COMMAND_INSTR(0x72, 0x07),
	ILI9881C_COMMAND_INSTR(0x73, 0x02),
	ILI9881C_COMMAND_INSTR(0x74, 0x02),
	ILI9881C_COMMAND_INSTR(0x75, 0x06),
	ILI9881C_COMMAND_INSTR(0x76, 0x07),
	ILI9881C_COMMAND_INSTR(0x77, 0x0e),
	ILI9881C_COMMAND_INSTR(0x78, 0x0f),
	ILI9881C_COMMAND_INSTR(0x79, 0x0c),
	ILI9881C_COMMAND_INSTR(0x7a, 0x0d),
	ILI9881C_COMMAND_INSTR(0x7b, 0x02),
	ILI9881C_COMMAND_INSTR(0x7c, 0x02),
	ILI9881C_COMMAND_INSTR(0x7d, 0x02),
	ILI9881C_COMMAND_INSTR(0x7e, 0x02),
	ILI9881C_COMMAND_INSTR(0x7f, 0x02),
	ILI9881C_COMMAND_INSTR(0x80, 0x02),
	ILI9881C_COMMAND_INSTR(0x81, 0x02),
	ILI9881C_COMMAND_INSTR(0x82, 0x02),
	ILI9881C_COMMAND_INSTR(0x83, 0x02),
	ILI9881C_COMMAND_INSTR(0x84, 0x02),
	ILI9881C_COMMAND_INSTR(0x85, 0x02),
	ILI9881C_COMMAND_INSTR(0x86, 0x02),
	ILI9881C_COMMAND_INSTR(0x87, 0x09),
	ILI9881C_COMMAND_INSTR(0x88, 0x08),
	ILI9881C_COMMAND_INSTR(0x89, 0x02),
	ILI9881C_COMMAND_INSTR(0x8a, 0x02),
	ILI9881C_SWITCH_PAGE_INSTR(4),
	ILI9881C_COMMAND_INSTR(0x6c, 0x15),
	ILI9881C_COMMAND_INSTR(0x6e, 0x2a),
	ILI9881C_COMMAND_INSTR(0x6f, 0x57),
	ILI9881C_COMMAND_INSTR(0x3a, 0xa4),
	ILI9881C_COMMAND_INSTR(0x8d, 0x1a),
	ILI9881C_COMMAND_INSTR(0x87, 0xba),
	ILI9881C_COMMAND_INSTR(0x26, 0x76),
	ILI9881C_COMMAND_INSTR(0xb2, 0xd1),
	ILI9881C_SWITCH_PAGE_INSTR(1),
	ILI9881C_COMMAND_INSTR(0x22, 0x0a),
	ILI9881C_COMMAND_INSTR(0x31, 0x00),
	ILI9881C_COMMAND_INSTR(0x53, 0x35),
	ILI9881C_COMMAND_INSTR(0x55, 0x50),
	ILI9881C_COMMAND_INSTR(0x50, 0xaf),
	ILI9881C_COMMAND_INSTR(0x51, 0xaf),
	ILI9881C_COMMAND_INSTR(0x60, 0x14),
	ILI9881C_COMMAND_INSTR(0xa0, 0x08),
	ILI9881C_COMMAND_INSTR(0xa1, 0x1d),
	ILI9881C_COMMAND_INSTR(0xa2, 0x2c),
	ILI9881C_COMMAND_INSTR(0xa3, 0x14),
	ILI9881C_COMMAND_INSTR(0xa4, 0x19),
	ILI9881C_COMMAND_INSTR(0xa5, 0x2e),
	ILI9881C_COMMAND_INSTR(0xa6, 0x22),
	ILI9881C_COMMAND_INSTR(0xa7, 0x23),
	ILI9881C_COMMAND_INSTR(0xa8, 0x97),
	ILI9881C_COMMAND_INSTR(0xa9, 0x1e),
	ILI9881C_COMMAND_INSTR(0xaa, 0x29),
	ILI9881C_COMMAND_INSTR(0xab, 0x7b),
	ILI9881C_COMMAND_INSTR(0xac, 0x18),
	ILI9881C_COMMAND_INSTR(0xad, 0x17),
	ILI9881C_COMMAND_INSTR(0xae, 0x4b),
	ILI9881C_COMMAND_INSTR(0xaf, 0x1f),
	ILI9881C_COMMAND_INSTR(0xb0, 0x27),
	ILI9881C_COMMAND_INSTR(0xb1, 0x52),
	ILI9881C_COMMAND_INSTR(0xb2, 0x63),
	ILI9881C_COMMAND_INSTR(0xb3, 0x39),
	ILI9881C_COMMAND_INSTR(0xc0, 0x08),
	ILI9881C_COMMAND_INSTR(0xc1, 0x1d),
	ILI9881C_COMMAND_INSTR(0xc2, 0x2c),
	ILI9881C_COMMAND_INSTR(0xc3, 0x14),
	ILI9881C_COMMAND_INSTR(0xc4, 0x19),
	ILI9881C_COMMAND_INSTR(0xc5, 0x2e),
	ILI9881C_COMMAND_INSTR(0xc6, 0x22),
	ILI9881C_COMMAND_INSTR(0xc7, 0x23),
	ILI9881C_COMMAND_INSTR(0xc8, 0x97),
	ILI9881C_COMMAND_INSTR(0xc9, 0x1e),
	ILI9881C_COMMAND_INSTR(0xca, 0x29),
	ILI9881C_COMMAND_INSTR(0xcb, 0x7b),
	ILI9881C_COMMAND_INSTR(0xcc, 0x18),
	ILI9881C_COMMAND_INSTR(0xcd, 0x17),
	ILI9881C_COMMAND_INSTR(0xce, 0x4b),
	ILI9881C_COMMAND_INSTR(0xcf, 0x1f),
	ILI9881C_COMMAND_INSTR(0xd0, 0x27),
	ILI9881C_COMMAND_INSTR(0xd1, 0x52),
	ILI9881C_COMMAND_INSTR(0xd2, 0x63),
	ILI9881C_COMMAND_INSTR(0xd3, 0x39),
};

static const struct ili9881c_instr rpi_7inch_init[] = {
	ILI9881C_SWITCH_PAGE_INSTR(3),
	ILI9881C_COMMAND_INSTR(0x01, 0x00),
	ILI9881C_COMMAND_INSTR(0x02, 0x00),
	ILI9881C_COMMAND_INSTR(0x03, 0x73),
	ILI9881C_COMMAND_INSTR(0x04, 0x00),
	ILI9881C_COMMAND_INSTR(0x05, 0x00),
	ILI9881C_COMMAND_INSTR(0x06, 0x0a),
	ILI9881C_COMMAND_INSTR(0x07, 0x00),
	ILI9881C_COMMAND_INSTR(0x08, 0x00),
	ILI9881C_COMMAND_INSTR(0x09, 0x61),
	ILI9881C_COMMAND_INSTR(0x0a, 0x00),
	ILI9881C_COMMAND_INSTR(0x0b, 0x00),
	ILI9881C_COMMAND_INSTR(0x0c, 0x01),
	ILI9881C_COMMAND_INSTR(0x0d, 0x00),
	ILI9881C_COMMAND_INSTR(0x0e, 0x00),
	ILI9881C_COMMAND_INSTR(0x0f, 0x61),
	ILI9881C_COMMAND_INSTR(0x10, 0x61),
	ILI9881C_COMMAND_INSTR(0x11, 0x00),
	ILI9881C_COMMAND_INSTR(0x12, 0x00),
	ILI9881C_COMMAND_INSTR(0x13, 0x00),
	ILI9881C_COMMAND_INSTR(0x14, 0x00),
	ILI9881C_COMMAND_INSTR(0x15, 0x00),
	ILI9881C_COMMAND_INSTR(0x16, 0x00),
	ILI9881C_COMMAND_INSTR(0x17, 0x00),
	ILI9881C_COMMAND_INSTR(0x18, 0x00),
	ILI9881C_COMMAND_INSTR(0x19, 0x00),
	ILI9881C_COMMAND_INSTR(0x1a, 0x00),
	ILI9881C_COMMAND_INSTR(0x1b, 0x00),
	ILI9881C_COMMAND_INSTR(0x1c, 0x00),
	ILI9881C_COMMAND_INSTR(0x1d, 0x00),
	ILI9881C_COMMAND_INSTR(0x1e, 0x40),
	ILI9881C_COMMAND_INSTR(0x1f, 0x80),
	ILI9881C_COMMAND_INSTR(0x20, 0x06),
	ILI9881C_COMMAND_INSTR(0x21, 0x01),
	ILI9881C_COMMAND_INSTR(0x22, 0x00),
	ILI9881C_COMMAND_INSTR(0x23, 0x00),
	ILI9881C_COMMAND_INSTR(0x24, 0x00),
	ILI9881C_COMMAND_INSTR(0x25, 0x00),
	ILI9881C_COMMAND_INSTR(0x26, 0x00),
	ILI9881C_COMMAND_INSTR(0x27, 0x00),
	ILI9881C_COMMAND_INSTR(0x28, 0x33),
	ILI9881C_COMMAND_INSTR(0x29, 0x03),
	ILI9881C_COMMAND_INSTR(0x2a, 0x00),
	ILI9881C_COMMAND_INSTR(0x2b, 0x00),
	ILI9881C_COMMAND_INSTR(0x2c, 0x00),
	ILI9881C_COMMAND_INSTR(0x2d, 0x00),
	ILI9881C_COMMAND_INSTR(0x2e, 0x00),
	ILI9881C_COMMAND_INSTR(0x2f, 0x00),
	ILI9881C_COMMAND_INSTR(0x30, 0x00),
	ILI9881C_COMMAND_INSTR(0x31, 0x00),
	ILI9881C_COMMAND_INSTR(0x32, 0x00),
	ILI9881C_COMMAND_INSTR(0x33, 0x00),
	ILI9881C_COMMAND_INSTR(0x34, 0x04),
	ILI9881C_COMMAND_INSTR(0x35, 0x00),
	ILI9881C_COMMAND_INSTR(0x36, 0x00),
	ILI9881C_COMMAND_INSTR(0x37, 0x00),
	ILI9881C_COMMAND_INSTR(0x38, 0x3c),
	ILI9881C_COMMAND_INSTR(0x39, 0x00),
	ILI9881C_COMMAND_INSTR(0x3a, 0x00),
	ILI9881C_COMMAND_INSTR(0x3b, 0x00),
	ILI9881C_COMMAND_INSTR(0x3c, 0x00),
	ILI9881C_COMMAND_INSTR(0x3d, 0x00),
	ILI9881C_COMMAND_INSTR(0x3e, 0x00),
	ILI9881C_COMMAND_INSTR(0x3f, 0x00),
	ILI9881C_COMMAND_INSTR(0x40, 0x00),
	ILI9881C_COMMAND_INSTR(0x41, 0x00),
	ILI9881C_COMMAND_INSTR(0x42, 0x00),
	ILI9881C_COMMAND_INSTR(0x43, 0x00),
	ILI9881C_COMMAND_INSTR(0x44, 0x00),
	ILI9881C_COMMAND_INSTR(0x50, 0x10),
	ILI9881C_COMMAND_INSTR(0x51, 0x32),
	ILI9881C_COMMAND_INSTR(0x52, 0x54),
	ILI9881C_COMMAND_INSTR(0x53, 0x76),
	ILI9881C_COMMAND_INSTR(0x54, 0x98),
	ILI9881C_COMMAND_INSTR(0x55, 0xba),
	ILI9881C_COMMAND_INSTR(0x56, 0x10),
	ILI9881C_COMMAND_INSTR(0x57, 0x32),
	ILI9881C_COMMAND_INSTR(0x58, 0x54),
	ILI9881C_COMMAND_INSTR(0x59, 0x76),
	ILI9881C_COMMAND_INSTR(0x5a, 0x98),
	ILI9881C_COMMAND_INSTR(0x5b, 0xba),
	ILI9881C_COMMAND_INSTR(0x5c, 0xdc),
	ILI9881C_COMMAND_INSTR(0x5d, 0xfe),
	ILI9881C_COMMAND_INSTR(0x5e, 0x00),
	ILI9881C_COMMAND_INSTR(0x5f, 0x0e),
	ILI9881C_COMMAND_INSTR(0x60, 0x0f),
	ILI9881C_COMMAND_INSTR(0x61, 0x0c),
	ILI9881C_COMMAND_INSTR(0x62, 0x0d),
	ILI9881C_COMMAND_INSTR(0x63, 0x06),
	ILI9881C_COMMAND_INSTR(0x64, 0x07),
	ILI9881C_COMMAND_INSTR(0x65, 0x02),
	ILI9881C_COMMAND_INSTR(0x66, 0x02),
	ILI9881C_COMMAND_INSTR(0x67, 0x02),
	ILI9881C_COMMAND_INSTR(0x68, 0x02),
	ILI9881C_COMMAND_INSTR(0x69, 0x01),
	ILI9881C_COMMAND_INSTR(0x6a, 0x00),
	ILI9881C_COMMAND_INSTR(0x6b, 0x02),
	ILI9881C_COMMAND_INSTR(0x6c, 0x15),
	ILI9881C_COMMAND_INSTR(0x6d, 0x14),
	ILI9881C_COMMAND_INSTR(0x6e, 0x02),
	ILI9881C_COMMAND_INSTR(0x6f, 0x02),
	ILI9881C_COMMAND_INSTR(0x70, 0x02),
	ILI9881C_COMMAND_INSTR(0x71, 0x02),
	ILI9881C_COMMAND_INSTR(0x72, 0x02),
	ILI9881C_COMMAND_INSTR(0x73, 0x02),
	ILI9881C_COMMAND_INSTR(0x74, 0x02),
	ILI9881C_COMMAND_INSTR(0x75, 0x0e),
	ILI9881C_COMMAND_INSTR(0x76, 0x0f),
	ILI9881C_COMMAND_INSTR(0x77, 0x0c),
	ILI9881C_COMMAND_INSTR(0x78, 0x0d),
	ILI9881C_COMMAND_INSTR(0x79, 0x06),
	ILI9881C_COMMAND_INSTR(0x7a, 0x07),
	ILI9881C_COMMAND_INSTR(0x7b, 0x02),
	ILI9881C_COMMAND_INSTR(0x7c, 0x02),
	ILI9881C_COMMAND_INSTR(0x7d, 0x02),
	ILI9881C_COMMAND_INSTR(0x7e, 0x02),
	ILI9881C_COMMAND_INSTR(0x7f, 0x01),
	ILI9881C_COMMAND_INSTR(0x80, 0x00),
	ILI9881C_COMMAND_INSTR(0x81, 0x02),
	ILI9881C_COMMAND_INSTR(0x82, 0x14),
	ILI9881C_COMMAND_INSTR(0x83, 0x15),
	ILI9881C_COMMAND_INSTR(0x84, 0x02),
	ILI9881C_COMMAND_INSTR(0x85, 0x02),
	ILI9881C_COMMAND_INSTR(0x86, 0x02),
	ILI9881C_COMMAND_INSTR(0x87, 0x02),
	ILI9881C_COMMAND_INSTR(0x88, 0x02),
	ILI9881C_COMMAND_INSTR(0x89, 0x02),
	ILI9881C_COMMAND_INSTR(0x8a, 0x02),
	ILI9881C_SWITCH_PAGE_INSTR(4),
	ILI9881C_COMMAND_INSTR(0x6c, 0x15),
	ILI9881C_COMMAND_INSTR(0x6e, 0x2a),
	ILI9881C_COMMAND_INSTR(0x6f, 0x33),
	ILI9881C_COMMAND_INSTR(0x3b, 0x98),
	ILI9881C_COMMAND_INSTR(0x3a, 0x94),
	ILI9881C_COMMAND_INSTR(0x8d, 0x14),
	ILI9881C_COMMAND_INSTR(0x87, 0xba),
	ILI9881C_COMMAND_INSTR(0x26, 0x76),
	ILI9881C_COMMAND_INSTR(0xb2, 0xd1),
	ILI9881C_COMMAND_INSTR(0xb5, 0x06),
	ILI9881C_COMMAND_INSTR(0x38, 0x01),
	ILI9881C_COMMAND_INSTR(0x39, 0x00),
	ILI9881C_SWITCH_PAGE_INSTR(1),
	ILI9881C_COMMAND_INSTR(0x22, 0x0a),
	ILI9881C_COMMAND_INSTR(0x31, 0x00),
	ILI9881C_COMMAND_INSTR(0x53, 0x7d),
	ILI9881C_COMMAND_INSTR(0x55, 0x8f),
	ILI9881C_COMMAND_INSTR(0x40, 0x33),
	ILI9881C_COMMAND_INSTR(0x50, 0x96),
	ILI9881C_COMMAND_INSTR(0x51, 0x96),
	ILI9881C_COMMAND_INSTR(0x60, 0x23),
	ILI9881C_COMMAND_INSTR(0xa0, 0x08),
	ILI9881C_COMMAND_INSTR(0xa1, 0x1d),
	ILI9881C_COMMAND_INSTR(0xa2, 0x2a),
	ILI9881C_COMMAND_INSTR(0xa3, 0x10),
	ILI9881C_COMMAND_INSTR(0xa4, 0x15),
	ILI9881C_COMMAND_INSTR(0xa5, 0x28),
	ILI9881C_COMMAND_INSTR(0xa6, 0x1c),
	ILI9881C_COMMAND_INSTR(0xa7, 0x1d),
	ILI9881C_COMMAND_INSTR(0xa8, 0x7e),
	ILI9881C_COMMAND_INSTR(0xa9, 0x1d),
	ILI9881C_COMMAND_INSTR(0xaa, 0x29),
	ILI9881C_COMMAND_INSTR(0xab, 0x6b),
	ILI9881C_COMMAND_INSTR(0xac, 0x1a),
	ILI9881C_COMMAND_INSTR(0xad, 0x18),
	ILI9881C_COMMAND_INSTR(0xae, 0x4b),
	ILI9881C_COMMAND_INSTR(0xaf, 0x20),
	ILI9881C_COMMAND_INSTR(0xb0, 0x27),
	ILI9881C_COMMAND_INSTR(0xb1, 0x50),
	ILI9881C_COMMAND_INSTR(0xb2, 0x64),
	ILI9881C_COMMAND_INSTR(0xb3, 0x39),
	ILI9881C_COMMAND_INSTR(0xc0, 0x08),
	ILI9881C_COMMAND_INSTR(0xc1, 0x1d),
	ILI9881C_COMMAND_INSTR(0xc2, 0x2a),
	ILI9881C_COMMAND_INSTR(0xc3, 0x10),
	ILI9881C_COMMAND_INSTR(0xc4, 0x15),
	ILI9881C_COMMAND_INSTR(0xc5, 0x28),
	ILI9881C_COMMAND_INSTR(0xc6, 0x1c),
	ILI9881C_COMMAND_INSTR(0xc7, 0x1d),
	ILI9881C_COMMAND_INSTR(0xc8, 0x7e),
	ILI9881C_COMMAND_INSTR(0xc9, 0x1d),
	ILI9881C_COMMAND_INSTR(0xca, 0x29),
	ILI9881C_COMMAND_INSTR(0xcb, 0x6b),
	ILI9881C_COMMAND_INSTR(0xcc, 0x1a),
	ILI9881C_COMMAND_INSTR(0xcd, 0x18),
	ILI9881C_COMMAND_INSTR(0xce, 0x4b),
	ILI9881C_COMMAND_INSTR(0xcf, 0x20),
	ILI9881C_COMMAND_INSTR(0xd0, 0x27),
	ILI9881C_COMMAND_INSTR(0xd1, 0x50),
	ILI9881C_COMMAND_INSTR(0xd2, 0x64),
	ILI9881C_COMMAND_INSTR(0xd3, 0x39),
};

static const struct ili9881c_instr bsd1218_a101kl68_init[] = {
	ILI9881C_SWITCH_PAGE_INSTR(3),
	ILI9881C_COMMAND_INSTR(0x01, 0x00),
	ILI9881C_COMMAND_INSTR(0x02, 0x00),
	ILI9881C_COMMAND_INSTR(0x03, 0x55),
	ILI9881C_COMMAND_INSTR(0x04, 0x55),
	ILI9881C_COMMAND_INSTR(0x05, 0x03),
	ILI9881C_COMMAND_INSTR(0x06, 0x06),
	ILI9881C_COMMAND_INSTR(0x07, 0x00),
	ILI9881C_COMMAND_INSTR(0x08, 0x07),
	ILI9881C_COMMAND_INSTR(0x09, 0x00),
	ILI9881C_COMMAND_INSTR(0x0a, 0x00),
	ILI9881C_COMMAND_INSTR(0x0b, 0x00),
	ILI9881C_COMMAND_INSTR(0x0c, 0x00),
	ILI9881C_COMMAND_INSTR(0x0d, 0x00),
	ILI9881C_COMMAND_INSTR(0x0e, 0x00),
	ILI9881C_COMMAND_INSTR(0x0f, 0x00),
	ILI9881C_COMMAND_INSTR(0x10, 0x00),
	ILI9881C_COMMAND_INSTR(0x11, 0x00),
	ILI9881C_COMMAND_INSTR(0x12, 0x00),
	ILI9881C_COMMAND_INSTR(0x13, 0x00),
	ILI9881C_COMMAND_INSTR(0x14, 0x00),
	ILI9881C_COMMAND_INSTR(0x15, 0x00),
	ILI9881C_COMMAND_INSTR(0x16, 0x00),
	ILI9881C_COMMAND_INSTR(0x17, 0x00),
	ILI9881C_COMMAND_INSTR(0x18, 0x00),
	ILI9881C_COMMAND_INSTR(0x19, 0x00),
	ILI9881C_COMMAND_INSTR(0x1a, 0x00),
	ILI9881C_COMMAND_INSTR(0x1b, 0x00),
	ILI9881C_COMMAND_INSTR(0x1c, 0x00),
	ILI9881C_COMMAND_INSTR(0x1d, 0x00),
	ILI9881C_COMMAND_INSTR(0x1e, 0xc0),
	ILI9881C_COMMAND_INSTR(0x1f, 0x80),
	ILI9881C_COMMAND_INSTR(0x20, 0x04),
	ILI9881C_COMMAND_INSTR(0x21, 0x03),
	ILI9881C_COMMAND_INSTR(0x22, 0x00),
	ILI9881C_COMMAND_INSTR(0x23, 0x00),
	ILI9881C_COMMAND_INSTR(0x24, 0x00),
	ILI9881C_COMMAND_INSTR(0x25, 0x00),
	ILI9881C_COMMAND_INSTR(0x26, 0x00),
	ILI9881C_COMMAND_INSTR(0x27, 0x00),
	ILI9881C_COMMAND_INSTR(0x28, 0x33),
	ILI9881C_COMMAND_INSTR(0x29, 0x33),
	ILI9881C_COMMAND_INSTR(0x2a, 0x00),
	ILI9881C_COMMAND_INSTR(0x2b, 0x00),
	ILI9881C_COMMAND_INSTR(0x2c, 0x00),
	ILI9881C_COMMAND_INSTR(0x2d, 0x00),
	ILI9881C_COMMAND_INSTR(0x2e, 0x00),
	ILI9881C_COMMAND_INSTR(0x2f, 0x00),
	ILI9881C_COMMAND_INSTR(0x30, 0x00),
	ILI9881C_COMMAND_INSTR(0x31, 0x00),
	ILI9881C_COMMAND_INSTR(0x32, 0x00),
	ILI9881C_COMMAND_INSTR(0x33, 0x00),
	ILI9881C_COMMAND_INSTR(0x34, 0x04),
	ILI9881C_COMMAND_INSTR(0x35, 0x00),
	ILI9881C_COMMAND_INSTR(0x36, 0x00),
	ILI9881C_COMMAND_INSTR(0x37, 0x00),
	ILI9881C_COMMAND_INSTR(0x38, 0x3c),
	ILI9881C_COMMAND_INSTR(0x39, 0x00),
	ILI9881C_COMMAND_INSTR(0x3a, 0x00),
	ILI9881C_COMMAND_INSTR(0x3b, 0x00),
	ILI9881C_COMMAND_INSTR(0x3c, 0x00),
	ILI9881C_COMMAND_INSTR(0x3d, 0x00),
	ILI9881C_COMMAND_INSTR(0x3e, 0x00),
	ILI9881C_COMMAND_INSTR(0x3f, 0x00),
	ILI9881C_COMMAND_INSTR(0x40, 0x00),
	ILI9881C_COMMAND_INSTR(0x41, 0x00),
	ILI9881C_COMMAND_INSTR(0x42, 0x00),
	ILI9881C_COMMAND_INSTR(0x43, 0x00),
	ILI9881C_COMMAND_INSTR(0x44, 0x00),
	ILI9881C_COMMAND_INSTR(0x50, 0x00),
	ILI9881C_COMMAND_INSTR(0x51, 0x11),
	ILI9881C_COMMAND_INSTR(0x52, 0x44),
	ILI9881C_COMMAND_INSTR(0x53, 0x55),
	ILI9881C_COMMAND_INSTR(0x54, 0x88),
	ILI9881C_COMMAND_INSTR(0x55, 0xab),
	ILI9881C_COMMAND_INSTR(0x56, 0x00),
	ILI9881C_COMMAND_INSTR(0x57, 0x11),
	ILI9881C_COMMAND_INSTR(0x58, 0x22),
	ILI9881C_COMMAND_INSTR(0x59, 0x33),
	ILI9881C_COMMAND_INSTR(0x5a, 0x44),
	ILI9881C_COMMAND_INSTR(0x5b, 0x55),
	ILI9881C_COMMAND_INSTR(0x5c, 0x66),
	ILI9881C_COMMAND_INSTR(0x5d, 0x77),
	ILI9881C_COMMAND_INSTR(0x5e, 0x00),
	ILI9881C_COMMAND_INSTR(0x5f, 0x02),
	ILI9881C_COMMAND_INSTR(0x60, 0x02),
	ILI9881C_COMMAND_INSTR(0x61, 0x0a),
	ILI9881C_COMMAND_INSTR(0x62, 0x09),
	ILI9881C_COMMAND_INSTR(0x63, 0x08),
	ILI9881C_COMMAND_INSTR(0x64, 0x13),
	ILI9881C_COMMAND_INSTR(0x65, 0x12),
	ILI9881C_COMMAND_INSTR(0x66, 0x11),
	ILI9881C_COMMAND_INSTR(0x67, 0x10),
	ILI9881C_COMMAND_INSTR(0x68, 0x0f),
	ILI9881C_COMMAND_INSTR(0x69, 0x0e),
	ILI9881C_COMMAND_INSTR(0x6a, 0x0d),
	ILI9881C_COMMAND_INSTR(0x6b, 0x0c),
	ILI9881C_COMMAND_INSTR(0x6c, 0x06),
	ILI9881C_COMMAND_INSTR(0x6d, 0x07),
	ILI9881C_COMMAND_INSTR(0x6e, 0x02),
	ILI9881C_COMMAND_INSTR(0x6f, 0x02),
	ILI9881C_COMMAND_INSTR(0x70, 0x02),
	ILI9881C_COMMAND_INSTR(0x71, 0x02),
	ILI9881C_COMMAND_INSTR(0x72, 0x02),
	ILI9881C_COMMAND_INSTR(0x73, 0x02),
	ILI9881C_COMMAND_INSTR(0x74, 0x02),
	ILI9881C_COMMAND_INSTR(0x75, 0x02),
	ILI9881C_COMMAND_INSTR(0x76, 0x02),
	ILI9881C_COMMAND_INSTR(0x77, 0x0a),
	ILI9881C_COMMAND_INSTR(0x78, 0x06),
	ILI9881C_COMMAND_INSTR(0x79, 0x07),
	ILI9881C_COMMAND_INSTR(0x7a, 0x10),
	ILI9881C_COMMAND_INSTR(0x7b, 0x11),
	ILI9881C_COMMAND_INSTR(0x7c, 0x12),
	ILI9881C_COMMAND_INSTR(0x7d, 0x13),
	ILI9881C_COMMAND_INSTR(0x7e, 0x0c),
	ILI9881C_COMMAND_INSTR(0x7f, 0x0d),
	ILI9881C_COMMAND_INSTR(0x80, 0x0e),
	ILI9881C_COMMAND_INSTR(0x81, 0x0f),
	ILI9881C_COMMAND_INSTR(0x82, 0x09),
	ILI9881C_COMMAND_INSTR(0x83, 0x08),
	ILI9881C_COMMAND_INSTR(0x84, 0x02),
	ILI9881C_COMMAND_INSTR(0x85, 0x02),
	ILI9881C_COMMAND_INSTR(0x86, 0x02),
	ILI9881C_COMMAND_INSTR(0x87, 0x02),
	ILI9881C_COMMAND_INSTR(0x88, 0x02),
	ILI9881C_COMMAND_INSTR(0x89, 0x02),
	ILI9881C_COMMAND_INSTR(0x8a, 0x02),

	ILI9881C_SWITCH_PAGE_INSTR(4),
	ILI9881C_COMMAND_INSTR(0x6e, 0x2a),
	ILI9881C_COMMAND_INSTR(0x6f, 0x37),
	ILI9881C_COMMAND_INSTR(0x3a, 0x24),
	ILI9881C_COMMAND_INSTR(0x8d, 0x19),
	ILI9881C_COMMAND_INSTR(0x87, 0xba),
	ILI9881C_COMMAND_INSTR(0xb2, 0xd1),
	ILI9881C_COMMAND_INSTR(0x88, 0x0b),
	ILI9881C_COMMAND_INSTR(0x38, 0x01),
	ILI9881C_COMMAND_INSTR(0x39, 0x00),
	ILI9881C_COMMAND_INSTR(0xb5, 0x02),
	ILI9881C_COMMAND_INSTR(0x31, 0x25),
	ILI9881C_COMMAND_INSTR(0x3b, 0x98),

	ILI9881C_SWITCH_PAGE_INSTR(1),
	ILI9881C_COMMAND_INSTR(0x22, 0x0a),
	ILI9881C_COMMAND_INSTR(0x31, 0x0c),
	ILI9881C_COMMAND_INSTR(0x53, 0x40),
	ILI9881C_COMMAND_INSTR(0x55, 0x45),
	ILI9881C_COMMAND_INSTR(0x50, 0xb7),
	ILI9881C_COMMAND_INSTR(0x51, 0xb2),
	ILI9881C_COMMAND_INSTR(0x60, 0x07),
	ILI9881C_COMMAND_INSTR(0xa0, 0x22),
	ILI9881C_COMMAND_INSTR(0xa1, 0x3f),
	ILI9881C_COMMAND_INSTR(0xa2, 0x4e),
	ILI9881C_COMMAND_INSTR(0xa3, 0x17),
	ILI9881C_COMMAND_INSTR(0xa4, 0x1a),
	ILI9881C_COMMAND_INSTR(0xa5, 0x2d),
	ILI9881C_COMMAND_INSTR(0xa6, 0x21),
	ILI9881C_COMMAND_INSTR(0xa7, 0x22),
	ILI9881C_COMMAND_INSTR(0xa8, 0xc4),
	ILI9881C_COMMAND_INSTR(0xa9, 0x1b),
	ILI9881C_COMMAND_INSTR(0xaa, 0x25),
	ILI9881C_COMMAND_INSTR(0xab, 0xa7),
	ILI9881C_COMMAND_INSTR(0xac, 0x1a),
	ILI9881C_COMMAND_INSTR(0xad, 0x19),
	ILI9881C_COMMAND_INSTR(0xae, 0x4b),
	ILI9881C_COMMAND_INSTR(0xaf, 0x1f),
	ILI9881C_COMMAND_INSTR(0xb0, 0x2a),
	ILI9881C_COMMAND_INSTR(0xb1, 0x59),
	ILI9881C_COMMAND_INSTR(0xb2, 0x64),
	ILI9881C_COMMAND_INSTR(0xb3, 0x3f),
	ILI9881C_COMMAND_INSTR(0xc0, 0x22),
	ILI9881C_COMMAND_INSTR(0xc1, 0x48),
	ILI9881C_COMMAND_INSTR(0xc2, 0x59),
	ILI9881C_COMMAND_INSTR(0xc3, 0x15),
	ILI9881C_COMMAND_INSTR(0xc4, 0x15),
	ILI9881C_COMMAND_INSTR(0xc5, 0x28),
	ILI9881C_COMMAND_INSTR(0xc6, 0x1c),
	ILI9881C_COMMAND_INSTR(0xc7, 0x1e),
	ILI9881C_COMMAND_INSTR(0xc8, 0xc4),
	ILI9881C_COMMAND_INSTR(0xc9, 0x1c),
	ILI9881C_COMMAND_INSTR(0xca, 0x2b),
	ILI9881C_COMMAND_INSTR(0xcb, 0xa3),
	ILI9881C_COMMAND_INSTR(0xcc, 0x1f),
	ILI9881C_COMMAND_INSTR(0xcd, 0x1e),
	ILI9881C_COMMAND_INSTR(0xce, 0x52),
	ILI9881C_COMMAND_INSTR(0xcf, 0x24),
	ILI9881C_COMMAND_INSTR(0xd0, 0x2a),
	ILI9881C_COMMAND_INSTR(0xd1, 0x58),
	ILI9881C_COMMAND_INSTR(0xd2, 0x68),
	ILI9881C_COMMAND_INSTR(0xd3, 0x3f),
};

static inline struct ili9881c *panel_to_ili9881c(struct drm_panel *panel)
{
	return container_of(panel, struct ili9881c, panel);
}

/*
 * The panel seems to accept some private DCS commands that map
 * directly to registers.
 *
 * It is organised by page, with each page having its own set of
 * registers, and the first page looks like it's holding the standard
 * DCS commands.
 *
 * So before any attempt at sending a command or data, we have to be
 * sure if we're in the right page or not.
 */
static void ili9881c_switch_page(struct mipi_dsi_multi_context *mctx, u8 page)
{
	u8 buf[4] = { 0xff, 0x98, 0x81, page };

	mipi_dsi_dcs_write_buffer_multi(mctx, buf, sizeof(buf));
}

static void ili9881c_send_cmd_data(struct mipi_dsi_multi_context *mctx, u8 cmd, u8 data)
{
	u8 buf[2] = { cmd, data };

	mipi_dsi_dcs_write_buffer_multi(mctx, buf, sizeof(buf));
}

static int ili9881c_prepare(struct drm_panel *panel)
{
	struct ili9881c *ctx = panel_to_ili9881c(panel);
	struct mipi_dsi_multi_context mctx = { .dsi = ctx->dsi };
	unsigned int i;
	int ret;

	/* Power the panel */
	ret = regulator_enable(ctx->power);
	if (ret)
		return ret;
	msleep(5);

	/* And reset it */
	gpiod_set_value_cansleep(ctx->reset, 1);
	msleep(20);

	gpiod_set_value_cansleep(ctx->reset, 0);
	msleep(20);

	for (i = 0; i < ctx->desc->init_length; i++) {
		const struct ili9881c_instr *instr = &ctx->desc->init[i];

		if (instr->op == ILI9881C_SWITCH_PAGE)
			ili9881c_switch_page(&mctx, instr->arg.page);
		else if (instr->op == ILI9881C_COMMAND)
			ili9881c_send_cmd_data(&mctx, instr->arg.cmd.cmd,
					       instr->arg.cmd.data);
	}

	ili9881c_switch_page(&mctx, 0);

	if (ctx->address_mode)
		ili9881c_send_cmd_data(&mctx, MIPI_DCS_SET_ADDRESS_MODE,
				       ctx->address_mode);

	mipi_dsi_dcs_set_tear_on_multi(&mctx, MIPI_DSI_DCS_TEAR_MODE_VBLANK);
	mipi_dsi_dcs_exit_sleep_mode_multi(&mctx);
	mipi_dsi_msleep(&mctx, 120);
	mipi_dsi_dcs_set_display_on_multi(&mctx);
	if (mctx.accum_err)
		goto disable_power;

	return 0;

disable_power:
	regulator_disable(ctx->power);
	return mctx.accum_err;
}

static int ili9881c_unprepare(struct drm_panel *panel)
{
	struct ili9881c *ctx = panel_to_ili9881c(panel);
	struct mipi_dsi_multi_context mctx = { .dsi = ctx->dsi };

	mipi_dsi_dcs_set_display_off_multi(&mctx);
	mipi_dsi_dcs_enter_sleep_mode_multi(&mctx);
	regulator_disable(ctx->power);
	gpiod_set_value_cansleep(ctx->reset, 1);

	return 0;
}

static const struct drm_display_mode lhr050h41_default_mode = {
	.clock		= 62000,

	.hdisplay	= 720,
	.hsync_start	= 720 + 10,
	.hsync_end	= 720 + 10 + 20,
	.htotal		= 720 + 10 + 20 + 30,

	.vdisplay	= 1280,
	.vsync_start	= 1280 + 10,
	.vsync_end	= 1280 + 10 + 10,
	.vtotal		= 1280 + 10 + 10 + 20,

	.width_mm	= 62,
	.height_mm	= 110,
};

static const struct drm_display_mode k101_im2byl02_default_mode = {
	.clock		= 69700,

	.hdisplay	= 800,
	.hsync_start	= 800 + 52,
	.hsync_end	= 800 + 52 + 8,
	.htotal		= 800 + 52 + 8 + 48,

	.vdisplay	= 1280,
	.vsync_start	= 1280 + 16,
	.vsync_end	= 1280 + 16 + 6,
	.vtotal		= 1280 + 16 + 6 + 15,

	.width_mm	= 135,
	.height_mm	= 217,
};

static const struct drm_display_mode kd050hdfia020_default_mode = {
	.clock		= 62000,

	.hdisplay	= 720,
	.hsync_start	= 720 + 10,
	.hsync_end	= 720 + 10 + 20,
	.htotal		= 720 + 10 + 20 + 30,

	.vdisplay	= 1280,
	.vsync_start	= 1280 + 10,
	.vsync_end	= 1280 + 10 + 10,
	.vtotal		= 1280 + 10 + 10 + 20,

	.width_mm	= 62,
	.height_mm	= 110,
};

static const struct drm_display_mode tl050hdv35_default_mode = {
	.clock		= 59400,

	.hdisplay	= 720,
	.hsync_start	= 720 + 18,
	.hsync_end	= 720 + 18 + 3,
	.htotal		= 720 + 18 + 3 + 20,

	.vdisplay	= 1280,
	.vsync_start	= 1280 + 26,
	.vsync_end	= 1280 + 26 + 6,
	.vtotal		= 1280 + 26 + 6 + 28,

	.width_mm	= 62,
	.height_mm	= 110,
};

static const struct drm_display_mode w552946aaa_default_mode = {
	.clock		= 65000,

	.hdisplay	= 720,
	.hsync_start	= 720 + 52,
	.hsync_end	= 720 + 52 + 8,
	.htotal		= 720 + 52 + 8 + 48,

	.vdisplay	= 1280,
	.vsync_start	= 1280 + 16,
	.vsync_end	= 1280 + 16 + 6,
	.vtotal		= 1280 + 16 + 6 + 15,

	.width_mm	= 68,
	.height_mm	= 121,
};

static const struct drm_display_mode w552946aba_default_mode = {
	.clock		= 64000,

	.hdisplay	= 720,
	.hsync_start	= 720 + 40,
	.hsync_end	= 720 + 40 + 10,
	.htotal		= 720 + 40 + 10 + 40,

	.vdisplay	= 1280,
	.vsync_start	= 1280 + 22,
	.vsync_end	= 1280 + 22 + 4,
	.vtotal		= 1280 + 22 + 4 + 11,

	.width_mm	= 68,
	.height_mm	= 121,
};

static const struct drm_display_mode am8001280g_default_mode = {
	.clock		= 67911,

	.hdisplay	= 800,
	.hsync_start	= 800 + 20,
	.hsync_end	= 800 + 20 + 32,
	.htotal		= 800 + 20 + 32 + 20,

	.vdisplay	= 1280,
	.vsync_start	= 1280 + 6,
	.vsync_end	= 1280 + 6 + 8,
	.vtotal		= 1280 + 6 + 8 + 4,

	.width_mm	= 94,
	.height_mm	= 151,
};

static const struct drm_display_mode rpi_5inch_default_mode = {
	.clock		= 83333,

	.hdisplay	= 720,
	.hsync_start	= 720 + 110,
	.hsync_end	= 720 + 110 + 12,
	.htotal		= 720 + 110 + 12 + 95,

	.vdisplay	= 1280,
	.vsync_start	= 1280 + 100,
	.vsync_end	= 1280 + 100 + 2,
	.vtotal		= 1280 + 100 + 2 + 100,

	.width_mm	= 62,
	.height_mm	= 110,
};

static const struct drm_display_mode rpi_7inch_default_mode = {
	.clock          = 83330,

	.hdisplay	= 720,
	.hsync_start	= 720 + 239,
	.hsync_end	= 720 + 239 + 33,
	.htotal		= 720 + 239 + 33 + 50,

	.vdisplay	= 1280,
	.vsync_start	= 1280 + 20,
	.vsync_end	= 1280 + 20 + 2,
	.vtotal		= 1280 + 20 + 2 + 30,

	.width_mm	= 90,
	.height_mm	= 151,
};

static const struct drm_display_mode bsd1218_a101kl68_default_mode = {
	.clock		= 70000,

	.hdisplay	= 800,
	.hsync_start	= 800 + 40,
	.hsync_end	= 800 + 40 + 20,
	.htotal		= 800 + 40 + 20 + 20,

	.vdisplay	= 1280,
	.vsync_start	= 1280 + 20,
	.vsync_end	= 1280 + 20 + 4,
	.vtotal		= 1280 + 20 + 4 + 20,

	.width_mm	= 120,
	.height_mm	= 170,
};

static int ili9881c_get_modes(struct drm_panel *panel,
			      struct drm_connector *connector)
{
	struct ili9881c *ctx = panel_to_ili9881c(panel);
	struct drm_display_mode *mode;

	mode = drm_mode_duplicate(connector->dev, ctx->desc->mode);
	if (!mode) {
		dev_err(&ctx->dsi->dev, "failed to add mode %ux%ux@%u\n",
			ctx->desc->mode->hdisplay,
			ctx->desc->mode->vdisplay,
			drm_mode_vrefresh(ctx->desc->mode));
		return -ENOMEM;
	}

	drm_mode_set_name(mode);

	mode->type = DRM_MODE_TYPE_DRIVER | DRM_MODE_TYPE_PREFERRED;
	drm_mode_probed_add(connector, mode);

	connector->display_info.width_mm = mode->width_mm;
	connector->display_info.height_mm = mode->height_mm;
	if (ctx->address_mode == 0x3)
		connector->display_info.subpixel_order = SubPixelHorizontalBGR;
	else
		connector->display_info.subpixel_order = SubPixelHorizontalRGB;

	/*
	 * TODO: Remove once all drm drivers call
	 * drm_connector_set_orientation_from_panel()
	 */
	drm_connector_set_panel_orientation(connector, ctx->orientation);

	return 1;
}

static enum drm_panel_orientation ili9881c_get_orientation(struct drm_panel *panel)
{
	struct ili9881c *ctx = panel_to_ili9881c(panel);

	return ctx->orientation;
}

static const struct drm_panel_funcs ili9881c_funcs = {
	.prepare	= ili9881c_prepare,
	.unprepare	= ili9881c_unprepare,
	.get_modes	= ili9881c_get_modes,
	.get_orientation = ili9881c_get_orientation,
};

static int ili9881c_dsi_probe(struct mipi_dsi_device *dsi)
{
	struct ili9881c *ctx;
	int ret;

	ctx = devm_drm_panel_alloc(&dsi->dev, struct ili9881c, panel, &ili9881c_funcs,
				   DRM_MODE_CONNECTOR_DSI);
	if (IS_ERR(ctx))
		return PTR_ERR(ctx);

	mipi_dsi_set_drvdata(dsi, ctx);
	ctx->dsi = dsi;
	ctx->desc = of_device_get_match_data(&dsi->dev);

	ctx->power = devm_regulator_get(&dsi->dev, "power");
	if (IS_ERR(ctx->power))
		return dev_err_probe(&dsi->dev, PTR_ERR(ctx->power),
				     "Couldn't get our power regulator\n");

	ctx->reset = devm_gpiod_get_optional(&dsi->dev, "reset", GPIOD_OUT_LOW);
	if (IS_ERR(ctx->reset))
		return dev_err_probe(&dsi->dev, PTR_ERR(ctx->reset),
				     "Couldn't get our reset GPIO\n");

	ret = of_drm_get_panel_orientation(dsi->dev.of_node, &ctx->orientation);
	if (ret) {
		dev_err(&dsi->dev, "%pOF: failed to get orientation: %d\n",
			dsi->dev.of_node, ret);
		return ret;
	}

	ctx->address_mode = ctx->desc->default_address_mode;
	if (ctx->orientation == DRM_MODE_PANEL_ORIENTATION_BOTTOM_UP) {
		ctx->address_mode ^= 0x03;
		ctx->orientation = DRM_MODE_PANEL_ORIENTATION_NORMAL;
	}

	ctx->panel.prepare_prev_first = true;

	ret = drm_panel_of_backlight(&ctx->panel);
	if (ret)
		return ret;

	drm_panel_add(&ctx->panel);

	dsi->mode_flags = ctx->desc->mode_flags;
	dsi->format = MIPI_DSI_FMT_RGB888;
	dsi->lanes = ctx->desc->lanes;

	return mipi_dsi_attach(dsi);
}

static void ili9881c_dsi_remove(struct mipi_dsi_device *dsi)
{
	struct ili9881c *ctx = mipi_dsi_get_drvdata(dsi);

	mipi_dsi_detach(dsi);
	drm_panel_remove(&ctx->panel);
}

static const struct ili9881c_desc lhr050h41_desc = {
	.init = lhr050h41_init,
	.init_length = ARRAY_SIZE(lhr050h41_init),
	.mode = &lhr050h41_default_mode,
	.mode_flags = MIPI_DSI_MODE_VIDEO_SYNC_PULSE,
	.lanes = 4,
};

static const struct ili9881c_desc k101_im2byl02_desc = {
	.init = k101_im2byl02_init,
	.init_length = ARRAY_SIZE(k101_im2byl02_init),
	.mode = &k101_im2byl02_default_mode,
	.mode_flags = MIPI_DSI_MODE_VIDEO_SYNC_PULSE,
	.lanes = 4,
};

static const struct ili9881c_desc kd050hdfia020_desc = {
	.init = kd050hdfia020_init,
	.init_length = ARRAY_SIZE(kd050hdfia020_init),
	.mode = &kd050hdfia020_default_mode,
	.mode_flags = MIPI_DSI_MODE_VIDEO | MIPI_DSI_MODE_VIDEO_SYNC_PULSE |
		      MIPI_DSI_MODE_LPM,
};

static const struct ili9881c_desc tl050hdv35_desc = {
	.init = tl050hdv35_init,
	.init_length = ARRAY_SIZE(tl050hdv35_init),
	.mode = &tl050hdv35_default_mode,
	.mode_flags = MIPI_DSI_MODE_VIDEO | MIPI_DSI_MODE_VIDEO_SYNC_PULSE |
		      MIPI_DSI_MODE_LPM,
	.default_address_mode = 0x03,
};

static const struct ili9881c_desc w552946aaa_desc = {
	.init = w552946aaa_init,
	.init_length = ARRAY_SIZE(w552946aaa_init),
	.mode = &w552946aaa_default_mode,
	.mode_flags = MIPI_DSI_MODE_VIDEO | MIPI_DSI_MODE_VIDEO_BURST |
		      MIPI_DSI_MODE_LPM | MIPI_DSI_MODE_NO_EOT_PACKET,
	.lanes = 2,
};

static const struct ili9881c_desc w552946aba_desc = {
	.init = w552946ab_init,
	.init_length = ARRAY_SIZE(w552946ab_init),
	.mode = &w552946aba_default_mode,
	.mode_flags = MIPI_DSI_MODE_VIDEO | MIPI_DSI_MODE_VIDEO_BURST |
		      MIPI_DSI_MODE_LPM | MIPI_DSI_MODE_NO_EOT_PACKET,
	.lanes = 4,
};

static const struct ili9881c_desc am8001280g_desc = {
	.init = am8001280g_init,
	.init_length = ARRAY_SIZE(am8001280g_init),
	.mode = &am8001280g_default_mode,
	.mode_flags = MIPI_DSI_MODE_VIDEO | MIPI_DSI_MODE_VIDEO_SYNC_PULSE |
		      MIPI_DSI_CLOCK_NON_CONTINUOUS | MIPI_DSI_MODE_LPM,
};

static const struct ili9881c_desc rpi_5inch_desc = {
	.init = rpi_5inch_init,
	.init_length = ARRAY_SIZE(rpi_5inch_init),
	.mode = &rpi_5inch_default_mode,
	.mode_flags =  MIPI_DSI_MODE_VIDEO | MIPI_DSI_MODE_LPM,
	.lanes = 2,
};

static const struct ili9881c_desc rpi_7inch_desc = {
	.init = rpi_7inch_init,
	.init_length = ARRAY_SIZE(rpi_7inch_init),
	.mode = &rpi_7inch_default_mode,
	.mode_flags =  MIPI_DSI_MODE_VIDEO | MIPI_DSI_MODE_LPM,
	.lanes = 2,
};

static const struct ili9881c_desc bsd1218_a101kl68_desc = {
	.init = bsd1218_a101kl68_init,
	.init_length = ARRAY_SIZE(bsd1218_a101kl68_init),
	.mode = &bsd1218_a101kl68_default_mode,
	.mode_flags = MIPI_DSI_MODE_VIDEO | MIPI_DSI_MODE_VIDEO_BURST |
		      MIPI_DSI_MODE_LPM | MIPI_DSI_MODE_NO_EOT_PACKET,
	.lanes = 4,
};

static const struct of_device_id ili9881c_of_match[] = {
	{ .compatible = "bananapi,lhr050h41", .data = &lhr050h41_desc },
	{ .compatible = "bestar,bsd1218-a101kl68", .data = &bsd1218_a101kl68_desc },
	{ .compatible = "feixin,k101-im2byl02", .data = &k101_im2byl02_desc },
	{ .compatible = "startek,kd050hdfia020", .data = &kd050hdfia020_desc },
	{ .compatible = "tdo,tl050hdv35", .data = &tl050hdv35_desc },
	{ .compatible = "wanchanglong,w552946aaa", .data = &w552946aaa_desc },
	{ .compatible = "wanchanglong,w552946aba", .data = &w552946aba_desc },
	{ .compatible = "ampire,am8001280g", .data = &am8001280g_desc },
	{ .compatible = "raspberrypi,dsi-5inch", &rpi_5inch_desc },
	{ .compatible = "raspberrypi,dsi-7inch", &rpi_7inch_desc },
	{ }
};
MODULE_DEVICE_TABLE(of, ili9881c_of_match);

static struct mipi_dsi_driver ili9881c_dsi_driver = {
	.probe		= ili9881c_dsi_probe,
	.remove		= ili9881c_dsi_remove,
	.driver = {
		.name		= "ili9881c-dsi",
		.of_match_table	= ili9881c_of_match,
	},
};
module_mipi_dsi_driver(ili9881c_dsi_driver);

MODULE_AUTHOR("Maxime Ripard <maxime.ripard@free-electrons.com>");
MODULE_DESCRIPTION("Ilitek ILI9881C Controller Driver");
MODULE_LICENSE("GPL v2");
