/* wptMainProc.cpp - Main window procedure
 *	Copyright (C) 2000-2005 Timo Schulz
 *
 * This file is part of WinPT.
 *
 * WinPT is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with WinPT; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <windows.h>
#include <commctrl.h>
#include <io.h>

#include "../resource.h"
#include "wptTypes.h"
#include "wptGPG.h"
#include "wptW32API.h"
#include "wptVersion.h"
#include "wptCommonCtl.h"
#include "wptContext.h" /* for passphrase_s */
#include "wptDlgs.h"
#include "wptErrors.h"
#include "wptNLS.h"
#include "wptHotkey.h"
#include "wptRegistry.h"
#include "wptKeyserver.h"
#include "wptAgent.h"
#include "wptKeyManager.h"
#include "wptCard.h"
#include "wptCryptdisk.h"

extern HWND activ_hwnd;
static int cmd = 0;
static int wipe_contents = 0;


BOOL CALLBACK
elgamal_warn_dlg_proc( HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam )
{
    gpgme_keycache_t pc;
    gpgme_key_t key;

    switch( msg ) {
    case WM_INITDIALOG:
	pc = keycache_get_ctx( 1 );
	while( !gpgme_keycache_next_key( pc, 0, &key ) ) {
	    if( gpgme_key_get_ulong_attr( key, GPGME_ATTR_ALGO, NULL, 0 )
		== GPGME_PK_ELG_ES ) {
		char tmp[128+64+1];
		_snprintf( tmp, sizeof (tmp)-1, "(0x%s) %s",
		gpgme_key_get_string_attr( key, GPGME_ATTR_KEYID, NULL, 0 )+8,
		gpgme_key_get_string_attr( key, GPGME_ATTR_USERID, NULL, 0 ) );
		SendDlgItemMessage( dlg, IDC_ELGWARN_LIST, LB_ADDSTRING,
				    0, (LPARAM)(const char *) tmp );
	    }
	}
	gpgme_keycache_rewind( pc );
	center_window( dlg );
	SetForegroundWindow( dlg );
	break;

    case WM_COMMAND:
	switch( LOWORD( wparam ) ) {
	case IDOK:
	    EndDialog( dlg, TRUE );
	    break;
	}
	break;
    }
    return FALSE;
} /* elgamal_warn_dlg_proc */


static void
cleanup_tmp_files (void)
{
    struct _finddata_t dat;
    char tmp[384];
    long hd;

    if (GetTempPath (sizeof (tmp)-1, tmp) == FALSE ||
	SetCurrentDirectory (tmp) == FALSE) {
	winpt_errmsg ("GetTempPath", 0);
	return;
    }
    hd = _findfirst ("gpgmeOUT*", &dat);
    if (hd == -1)
	return;
    do {
	secure_unlink (dat.name, WIPE_MODE_SIMPLE);
    } while (_findnext (hd, &dat) == 0);
    _findclose (hd);
}


static BOOL CALLBACK
confirm_delclipboard_dlg (HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam)
{
    switch (msg) {
    case WM_INITDIALOG:
	center_window (dlg);
	SetForegroundWindow (dlg);
	break;

    case WM_COMMAND:
	switch (LOWORD (wparam)) {
	case IDYES:
	    wipe_contents = 1;
	    if (IsDlgButtonChecked (dlg, IDC_CONFDELCLIP_BRAIN))
		set_reg_winpt_flag ("WipeClipboard", 1);
	    EndDialog (dlg, TRUE);
	    break;

	case IDNO:
	    wipe_contents = 0;
	    if (IsDlgButtonChecked (dlg, IDC_CONFDELCLIP_BRAIN))
		set_reg_winpt_flag ("WipeClipboard", 0);
	    EndDialog (dlg, FALSE);
	    break;
	}
	break;
    }
    return FALSE;
} /* confirm_delclipboard_dlg */


static int
currwnd_gpg_dlg (HWND hwnd, UINT id, int *ret_set)
{
    gpgme_error_t err;
    gpgme_cliptype_t type;
    int rc = 0;
    
    if( ret_set )
	*ret_set = 1;
    switch( id ) {
    case ID_WINPT_CURRWND_SYMENC:
	gpgme_encrypt_symmetric ();
	break;

    case ID_WINPT_CURRWND_ENCRYPT:
        dialog_box_param( glob_hinst, (LPCSTR)IDD_WINPT_ENCRYPT, hwnd,
                          clip_encrypt_dlg_proc, NULL, 
                          _("Encryption"), IDS_WINPT_ENCRYPT );
        break;
        
    case ID_WINPT_CURRWND_SIGNENCRYPT:
        dialog_box_param( glob_hinst, (LPCSTR)IDD_WINPT_SIGNENC, hwnd,
                          clip_signenc_dlg_proc, NULL, 
                          _("Sign & Encrypt"), IDS_WINPT_SIGNENC );
        break;
        
    case ID_WINPT_CURRWND_SIGN:
        dialog_box_param( glob_hinst, (LPCSTR)IDD_WINPT_SIGN, hwnd,
                          clip_sign_dlg_proc, NULL, 
                          _("Signing"), IDS_WINPT_SIGN );
        break;
        
    case ID_WINPT_CURRWND_DECRYPT_VERIFY:
	err = gpgme_clip_get_pgptype( &type );
        if( err ) {
	    msg_box( hwnd, gpgme_strerror( err ), _("Clipboard"), MB_ERR );
            break;
        }
        if( (type & GPGME_CLIP_MESSAGE) && !(type & GPGME_CLIP_CLEARSIG) ) {
            /* It makes only sense to insert the data in the window if it's
               really a ciphertext. Verfied data doesn't need this. */
            rc = clip_decrypt_dlg( hwnd );
            if( !rc && reg_prefs.use_viewer ) {
                dialog_box_param( glob_hinst, (LPCSTR)IDD_WINPT_CLIPEDIT,
                                  GetDesktopWindow(), clip_edit_dlg_proc, NULL, 
                                 _("Clipboard Editor"), IDS_WINPT_CLIPEDIT );
		if( ret_set )
		    *ret_set = 0;
            }
        }
	else if ((type & GPGME_CLIP_PUBKEY) && !(type & GPGME_CLIP_CLEARSIG))
	    km_clip_import (GetDesktopWindow ());
        else {
            dialog_box_param( glob_hinst, (LPCSTR)IDD_WINPT_VERIFY, hwnd,
                              clip_verify_dlg_proc, NULL, 
                              _("Verify"), IDS_WINPT_VERIFY );
            rc = 1;
        }
        break;
    }
    
    return rc;
} /* currwnd_gpg_dlg */


static void
clip_gpg_dlg( HWND hwnd, UINT id )
{
    gpgme_error_t ec;
    gpgme_cliptype_t type;
    size_t size = 0;
    int rc;
    
    if( (id == ID_WINPT_SIGN || id == ID_WINPT_SIGNENCRYPT)
	&& gnupg_access_keyring (0) ) {
	msg_box( hwnd, _("Could not access secret keyring."), _("Sign"), MB_ERR );
	return;
    }
    
    switch( id ) {
    case ID_WINPT_SYMENC:
        gpgme_encrypt_symmetric();
        break;
        
    case ID_WINPT_ENCRYPT:
        dialog_box_param( glob_hinst, (LPCSTR)IDD_WINPT_ENCRYPT, hwnd, 
                          clip_encrypt_dlg_proc, NULL,
                          _("Encryption"), IDS_WINPT_ENCRYPT );
        break;
        
    case ID_WINPT_SIGN:
        dialog_box_param( glob_hinst, (LPCSTR)IDD_WINPT_SIGN, hwnd, 
                          clip_sign_dlg_proc, NULL, 
                          _("Signing"), IDS_WINPT_SIGN );
        break;
        
    case ID_WINPT_SIGNENCRYPT:
        dialog_box_param( glob_hinst, (LPCSTR)IDD_WINPT_SIGNENC, hwnd, 
                          clip_signenc_dlg_proc, NULL, 
                          _("Sign & Encrypt"), IDS_WINPT_SIGNENC );
        break;
        
    case ID_WINPT_DECRYPT_VERIFY:
	ec = gpgme_clip_get_pgptype( &type );
	if( ec ) {
	    msg_box( hwnd, gpgme_strerror( ec ), _("Clipboard"), MB_ERR );
	    break;
	}
        if( (type & GPGME_CLIP_MESSAGE) && !(type & GPGME_CLIP_CLEARSIG) ) {
            rc = clip_decrypt_dlg (hwnd);
            if (!rc && reg_prefs.use_viewer) {
                dialog_box_param( glob_hinst, (LPCSTR)IDD_WINPT_CLIPEDIT, 
				  GetDesktopWindow(), clip_edit_dlg_proc, NULL, 
                                 _("Clipboard Editor"), IDS_WINPT_CLIPEDIT );
            }
        }
        else if( (type & GPGME_CLIP_SIG) && !(type & GPGME_CLIP_CLEARSIG) ) {
            text_input_s input;
            gpgme_data_t sig;
            
            memset( &input, 0, sizeof (input) );
            ec = gpgme_data_new_from_clipboard (&sig);
	    if( ec ) {
		msg_box( hwnd, gpgme_strerror( ec ),_("Verify"), MB_ERR );
		return;
	    }
            input.type = 0;
            dialog_box_param( glob_hinst, (LPCSTR)IDD_WINPT_TEXT, hwnd,
                             text_input_dlg_proc, (LPARAM)&input, 
                             _("Text Input"), IDS_WINPT_TEXT );
            
            gpgme_data_release_and_set_clipboard( sig );
            if( input.length ) {
                dialog_box_param( glob_hinst, (LPCSTR)IDD_WINPT_VERIFY, hwnd,
                                 clip_verify_dlg_proc, (LPARAM)&input,
                                 _("Verify"), IDS_WINPT_VERIFY );
                input.length = 0;
		free_if_alloc( input.data );
            }
        }
        else if( type & GPGME_CLIP_CLEARSIG ) {		
            dialog_box_param( glob_hinst, (LPCSTR)IDD_WINPT_VERIFY, hwnd,
                              clip_verify_dlg_proc, NULL, 
                              _("Verify"), IDS_WINPT_VERIFY );
        }
        else if ((type & GPGME_CLIP_PUBKEY) || (type & GPGME_CLIP_SECKEY))
	    km_clip_import (GetDesktopWindow ());
        else
            msg_box (hwnd, _("Unknown OpenPGP type."), _("Clipboard"), MB_ERR);
    }
} /* clip_gpg_dlg */


gpgme_card_t
smartcard_init (void)
{
    /*
    pcsc_loadlib (scard_support);
    rc = show_card_status ();
    if( rc )    
	break;
    */
    return gpg_load_scard ();
} /* smartcard_init */


LRESULT CALLBACK 
winpt_main_proc (HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{		
    static NOTIFYICONDATA NID;
    static int use_hotkey = 0;    
    reminder_ctx_s ctx;
    LPARAM param;
    curr_wnd_ctx currwnd = {0};	
    size_t size = 0;
    int rc, set_wc = 0, has_data = 0;
    INITCOMMONCONTROLSEX cce;
    gpgme_error_t ec;
    gpgme_card_t card;

    switch( msg ) {
    case WM_CREATE:
        NID.cbSize = sizeof (NID);
        NID.uFlags = NIF_MESSAGE | NIF_ICON | NIF_TIP;
        NID.uCallbackMessage = WM_USER;
        NID.hWnd = hwnd;
        NID.hIcon = LoadIcon (glob_hinst, MAKEINTRESOURCE (IDI_WINPT));
        strcpy (NID.szTip, "Windows Privacy Tray v"PGM_VERSION);
        Shell_NotifyIcon (NIM_ADD, &NID);
        DestroyIcon (NID.hIcon);
        rc = wsock_init ();
        if (rc)
            msg_box (NULL, winpt_strerror (rc), "Winsock2 DLL", MB_ERR);
        if (!reg_prefs.no_hotkeys) {
            rc = hotkeys_register (hwnd);
            if (rc)
                msg_box (hwnd, hotkeys_strerror (), winpt_strerror (rc), MB_ERR);
        }
        rc = PTD_initialize ();
        if (!rc)
            msg_box (hwnd, _("Could not set current window mode hooks."), _("WinPT Error"), MB_OK);
	mapi_init ();
	/* init common controls: date control. */
	cce.dwSize = sizeof (INITCOMMONCONTROLSEX);
	cce.dwICC = ICC_DATE_CLASSES;
	InitCommonControlsEx (&cce);
	LoadLibrary ("RichEd32.Dll");
        break;
        
    case WM_DESTROY:
    case WM_CLOSE:
	cryptdisk_cleanup ();
	mapi_deinit ();
        wsock_end ();        
        keycache_release ();
	gnupg_backup_keyrings ();
	free_reg_prefs ();
	free_gnupg_table ();
        hotkeys_unregister (hwnd);
        release_file_lock (&mo_file);
        PTD_delete ();
	agent_flush_cache ();
	if (!gpgme_clip_istext_avail (&has_data) && has_data) {
	    int chk = get_reg_winpt_flag ("WipeClipboard");
	    if (chk == -1)
		DialogBoxParam (glob_hinst, (LPCTSTR)IDD_WINPT_CONFDELCLIP,
				GetDesktopWindow (), confirm_delclipboard_dlg, 
				NULL);
	    if (wipe_contents || chk)
		set_clip_text (NULL, " ", 1);
	    wipe_contents = 0;
	}
	gpgme_lib_cleanup ();
	cleanup_tmp_files ();
        Shell_NotifyIcon (NIM_DELETE, &NID);
        PostQuitMessage (0);
        ExitProcess (0);
        return 0;
        
    case WM_USER:
        switch (lparam) {
        case WM_RBUTTONUP:
            HMENU hm, popup;
            POINT p;

	    gpg_read_only = gpg_check_permissions (0) == 0? 0 : 1;

            SetForegroundWindow (hwnd);
            GetCursorPos (&p);
            hm = LoadMenu (glob_hinst, MAKEINTRESOURCE (IDR_WINPT));
            popup = GetSubMenu (hm, 0);
#ifndef LANG_DE
            set_menu_text( popup, ID_WINPT_FILE, _("File Manager") );
            set_menu_text( popup, ID_WINPT_KEY, _("Key Manager") );
	    set_menu_text( popup, ID_WINPT_CARD, _("Card Manager") );
	    if (!scard_support)
		set_menu_state (popup, ID_WINPT_CARD, MF_DISABLED|MF_GRAYED);
	    if (!cryptdisk_available ()) {
		set_menu_state (popup, ID_WINPT_CDISKNEW, MF_DISABLED|MF_GRAYED);
		set_menu_state (popup, ID_WINPT_CDISKMOUNT, MF_DISABLED|MF_GRAYED);
		set_menu_state (popup, ID_WINPT_CDISKUNMOUNT, MF_DISABLED|MF_GRAYED);
	    }

            set_menu_text( popup, ID_WINPT_EDIT, _("Edit Clipboard") );
            set_menu_text( popup, ID_WINPT_ABOUT, _("About...") );
            set_menu_text( popup, ID_WINPT_QUIT, _("Exit") );
            set_menu_text( popup, ID_WINPT_SYMENC, _("Symmetric") );
#if 0
            set_menu_text( popup, ID_WINPT_ENCRYPT, _("Encrypt") );
            set_menu_text( popup, ID_WINPT_SIGNENCRYPT, _("Sign && Encrypt") );
            set_menu_text( popup, ID_WINPT_DECRYPT, _("Decrypt/Verify") );
            set_menu_text( popup, ID_WINPT_VERIFY, _("Verify") );
            set_menu_text( popup, ID_WINPT_CURRWND_ENCRYPT, _("Encrypt") );
            set_menu_text( popup, ID_WINPT_CURRWND_SIGNENCRYPT, _("Sign && Encrypt") );
            set_menu_text( popup, ID_WINPT_CURRWND_DECRYPT_VERIFY, _("Decrypt/Verify") );
            set_menu_text( popup, ID_WINPT_CURRWND_SIGN, _("Sign") );
#endif
#endif
            TrackPopupMenu( popup, TPM_RIGHTALIGN, p.x, p.y, 0, hwnd, NULL );
            PostMessage( hwnd, WM_USER, 0, 0 );
            DestroyMenu( popup );
            DestroyMenu( hm );
            break;
            
        case WM_LBUTTONDBLCLK:
            SendMessage( hwnd, WM_COMMAND, ID_WINPT_KEY, 0 );
            break;
        }
        break;
        
    case WM_HOTKEY:
        cmd = 0;
        switch( wparam ) {
        case WPT_CLIP_ENCRYPT_ID:			
            SendMessage( hwnd, WM_COMMAND, ID_WINPT_ENCRYPT, 0 );
            break;
            
        case WPT_CLIP_DECRYPT_VERIFY_ID:
            SendMessage( hwnd, WM_COMMAND, ID_WINPT_DECRYPT_VERIFY, 0 );
            break;
            
        case WPT_CLIP_SIGN_ID:
            SendMessage( hwnd, WM_COMMAND, ID_WINPT_SIGN, 0 );
            break;
            
        case WPT_CLIP_SIGN_ENCRYPT_ID:
            SendMessage( hwnd, WM_COMMAND, ID_WINPT_SIGNENCRYPT, 0 );
            break;
            
        case WPT_CURRWND_ENCRYPT_ID:
            use_hotkey = 1;
            SendMessage( hwnd, WM_COMMAND, ID_WINPT_CURRWND_ENCRYPT, 0 );
            break;				
            
        case WPT_CURRWND_SIGN_ID:
            use_hotkey = 1;
            SendMessage( hwnd, WM_COMMAND, ID_WINPT_CURRWND_SIGN, 0 );
            break;
            
        case WPT_CURRWND_SIGN_ENCRYPT_ID:
	    use_hotkey = 1;
            SendMessage( hwnd, WM_COMMAND, ID_WINPT_CURRWND_SIGNENCRYPT, 0 );
            break;
            
        case WPT_CURRWND_DECRYPT_VERIFY_ID:
            use_hotkey = 1;
            SendMessage( hwnd, WM_COMMAND, ID_WINPT_CURRWND_DECRYPT_VERIFY, 0 );
            break;
            
        case WPT_AGENT_FORGET_ID:
            rc = msg_box( hwnd, _("Remove all passphrases from cache?"), 
			  _("WinPT"), MB_YESNO|MB_ICONQUESTION );
            if( rc == IDYES )
                agent_flush_cache();
            break;
        }		
        break;
        
    case WM_COMMAND:
        if( gnupg_access_keyring( 1 ) ) {
	    if( log_box( "WinPT", MB_ERR|MB_YESNO, 
		_("Could not access public keyring, exit WinPT?") ) == IDYES )
                SendMessage( hwnd, WM_DESTROY, 0, 0 );
            break;
        }
        switch( wparam ) {
            /** Clipboard operations **/
        case ID_WINPT_SYMENC:
        case ID_WINPT_ENCRYPT:
        case ID_WINPT_SIGN:
        case ID_WINPT_SIGNENCRYPT:
        case ID_WINPT_DECRYPT_VERIFY:
	    ec = gpgme_clip_istext_avail( &has_data );
	    if( ec ) {
		msg_box( hwnd, gpgme_strerror( ec ),_("Clipboard"), MB_ERR );
		break;
	    }
	    if( !has_data ) {
		msg_box( hwnd, gpgme_strerror( GPGME_Clip_Empty ),_("Clipboard"), MB_ERR );
		break;
	    }
            ctx.msecs = 500;
            window_reminder( &ctx );			
            clip_gpg_dlg( hwnd, (int)wparam );
            break;
            
            /** Current window operations **/
	case ID_WINPT_CURRWND_SYMENC:
        case ID_WINPT_CURRWND_ENCRYPT:
        case ID_WINPT_CURRWND_SIGNENCRYPT:
        case ID_WINPT_CURRWND_DECRYPT_VERIFY:
        case ID_WINPT_CURRWND_SIGN:
            rc = get_window_contents( hwnd, &currwnd, &use_hotkey );
            if( rc ) {
		log_box( _("WinPT Error"), MB_ERR, 
			 _("Make sure that the window contains text.\n"
			   "%s."),
			winpt_strerror( WPTERR_CURR_WND ) );
                break;
            }
            ctx.msecs = 500;
            window_reminder( &ctx );
            rc = currwnd_gpg_dlg( hwnd, (UINT)wparam, &set_wc );
            if( !rc && set_wc )
                set_window_contents( hwnd, &currwnd );
            break;
            
            /** File handling **/
        case ID_WINPT_FILE:
            dialog_box_param( glob_hinst, (LPCSTR)IDD_WINPT_FILE, 
		              GetDesktopWindow(), file_manager_dlg_proc, NULL,
                              _("File Manager (use drag & drop to add files)"), IDS_WINPT_FILE );
            break;
            
            /** Misc **/
        case ID_WINPT_KEY:
            param = NULL;
            if( cmd )
                param = (LPARAM)cmd;
            dialog_box_param( glob_hinst, (LPCSTR)IDD_WINPT_KEYMISC, 
			      GetDesktopWindow(), keymanager_dlg_proc, param, 
                              _("Key Manager"), IDS_WINPT_KEYMISC );
            cmd = 0;
            break;

	case ID_WINPT_CARD:
	    card = smartcard_init ();
	    if( card ) {
		dialog_box_param( glob_hinst, (LPCSTR)IDD_WINPT_CARD_EDIT,
				  GetDesktopWindow(), card_edit_dlg_proc, 
				  (LPARAM)card,
				  _("Card Edit"), IDS_WINPT_CARD_EDIT );
		gpgme_card_release( card ); card = NULL;
	    }
	    break;
            
        case ID_WINPT_EDIT:
            dialog_box_param( glob_hinst, (LPCSTR)IDD_WINPT_CLIPEDIT,
			      GetDesktopWindow(), clip_edit_dlg_proc, NULL,
                             _("Clipboard Editor"), IDS_WINPT_CLIPEDIT );
            break;
            
        case ID_WINPT_ABOUT:
            dialog_box_param( glob_hinst, (LPCSTR)IDD_WINPT_ABOUT, GetDesktopWindow(), 
                              about_winpt_dlg_proc, NULL, 
                             _("About WinPT"), IDS_WINPT_ABOUT );
            break;
            
        case ID_WINPT_PREFS:
            dialog_box_param( glob_hinst, (LPCSTR)IDD_WINPT_PREFS, GetDesktopWindow(),
                              prefs_dlg_proc, NULL,
                             _("WinPT Preferences"), IDS_WINPT_PREFS );
            break;
            
        case ID_WINPT_GPGPREFS:
            dialog_box_param( glob_hinst, (LPCSTR)IDD_WINPT_GPGPREFS, GetDesktopWindow(),
                              gpgprefs_dlg_proc, NULL,
                             _("GnuPG Preferences"), IDS_WINPT_GPGPREFS );
            break;

	case ID_WINPT_CDISKNEW:
	    DialogBoxParam (glob_hinst, (LPCTSTR)IDD_WINPT_CDISK_NEW,
			GetDesktopWindow (), cryptdisk_new_dlg_proc, NULL);
	    break;

	case ID_WINPT_CDISKMOUNT:
	    DialogBoxParam (glob_hinst, (LPCTSTR)IDD_WINPT_CDISK_MOUNT, 
			GetDesktopWindow (), cryptdisk_mount_dlg_proc, NULL);
	    break;

	case ID_WINPT_CDISKUNMOUNT:
	    DialogBoxParam (glob_hinst, (LPCTSTR)IDD_WINPT_CDISK_UMOUNT,
			GetDesktopWindow (), cryptdisk_umount_dlg_proc, NULL);
	    break;

	case ID_WINPT_CDISKUMOUNTALL:
	    cryptdisk_unmount (0, 0);
	    break;
            
        case ID_WINPT_QUIT:
            SendMessage (hwnd, WM_DESTROY, 0, 0);
            break;
        }
        break;
    }
    
    return DefWindowProc (hwnd, msg, wparam, lparam);
} /* winpt_main_proc */
