/* gpgme.h -  My GnuPG Made Easy (MyGPGME)
 *	Copyright (C) 2000 Werner Koch (dd9jn)
 *	Copyright (C) 2001-2005 Timo Schulz
 *
 * This file is part of MyGPGME.
 *
 * MyGPGME is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MyGPGME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#ifndef GPGME_H
#define GPGME_H

#include <stdio.h> /* for FILE * */
#include <time.h> /* for time_t */

#ifdef __cplusplus
extern "C" { 
#if 0 /* just to make Emacs auto-indent happy */
}
#endif
#endif


/*
 * The version of this header should match the one of the library
 * It should not be used by a program because gpgme_check_version(NULL)
 * does return the same version.  The purpose of this macro is to
 * let autoconf (using the AM_PATH_GPGME macro) check that this
 * header matches the installed library.
 * Warning: Do not edit the next line.  configure will do that for you! */
#define GPGME_VERSION "20050418ts"

struct gpgme_context_s;
typedef struct gpgme_context_s *gpgme_ctx_t;

struct gpgme_data_s;
typedef struct gpgme_data_s *gpgme_data_t;

struct gpgme_recipients_s;
typedef struct gpgme_recipients_s *gpgme_recipients_t;

struct gpgme_key_s;
typedef struct gpgme_key_s *gpgme_key_t;

struct gpgme_keycache_s;
typedef struct gpgme_keycache_s *gpgme_keycache_t;

struct gpgme_signature_s;
typedef struct gpgme_signature_s *gpgme_sig_t;

struct gpgme_editkey_s;
typedef struct gpgme_editkey_s *gpgme_editkey_t;

struct gpgme_editcard_s;
typedef struct gpgme_editcard_s *gpgme_editcard_t;

struct gpgme_card_s;
typedef struct gpgme_card_s *gpgme_card_t;

struct gpgme_user_id_info_s;
typedef struct gpgme_user_id_info_s *gpgme_uidinfo_t;

struct gpgme_rev_cert_s;
typedef struct gpgme_rev_cert_s *gpgme_revcert_t;


typedef enum {
    GPGME_EOF = -1,
    GPGME_No_Error = 0,
    GPGME_General_Error = 1,
    GPGME_Out_Of_Core = 2,
    GPGME_Invalid_Value = 3,
    GPGME_Busy = 4,
    GPGME_No_Request = 5,
    GPGME_Exec_Error = 6,
    GPGME_Too_Many_Procs = 7,
    GPGME_Pipe_Error = 8,
    GPGME_No_Recipients = 9,
    GPGME_No_Data = 10,
    GPGME_Conflict = 11,
    GPGME_Not_Implemented = 12,
    GPGME_Read_Error = 13,
    GPGME_Write_Error = 14,
    GPGME_Invalid_Type = 15,
    GPGME_Invalid_Mode = 16,
    GPGME_File_Error = 17,  /* errno is set in this case */
    GPGME_Decryption_Failed = 18,
    GPGME_Signing_Failed = 19,
    GPGME_Encryption_Failed = 29,
    GPGME_No_Passphrase = 20,	
    GPGME_No_Seckey = 21,
    GPGME_No_Pubkey = 27,
    GPGME_Canceled = 22,
    GPGME_Invalid_Key = 23,
    GPGME_Invalid_Engine = 24,
    GPGME_Bad_Signature = 25,
    GPGME_Bad_Passphrase = 26,
    GPGME_Inv_Recipients = 28,
    GPGME_Internal_GPG_Problem = 30,
    GPGME_Clip_Open = 31,
    GPGME_Clip_Empty = 32,
    GPGME_Clip_Get = 33,
    GPGME_Cipher_IDEA = 34,
    GPGME_Conf_InvOption = 35,
    GPGME_SC_Failure = 36,

    /* Recipient Errors */    
    GPGME_RecpError_NoReason      =  100,
    GPGME_RecpError_NotFound      =  101,
    GPGME_RecpError_Ambigious     =  102,
    GPGME_RecpError_WrongKeyUsage =  103,
    GPGME_RecpError_Revoked       =  104,
    GPGME_RecpError_Expired       =  105,
    GPGME_RecpError_NotSecret     =  109,
    GPGME_RecpError_Untrusted     =  110,
} gpgme_error_t;

typedef enum {
    GPGME_DATA_TYPE_NONE = 0,
    GPGME_DATA_TYPE_MEM  = 1,
    GPGME_DATA_TYPE_FD   = 2,
    GPGME_DATA_TYPE_FILE = 3,
    GPGME_DATA_TYPE_CB   = 4
} gpgme_data_type_t;

typedef enum {
    GPGME_SIG_STAT_NONE  = 0,
    GPGME_SIG_STAT_GOOD  = 1,
    GPGME_SIG_STAT_BAD   = 2,
    GPGME_SIG_STAT_NOKEY = 3,
    GPGME_SIG_STAT_NOSIG = 4,
    GPGME_SIG_STAT_ERROR = 5,
    GPGME_SIG_STAT_DIFF  = 6,
    GPGME_SIG_STAT_E_GOOD= 7,
    GPGME_SIG_STAT_R_GOOD= 8,
} gpgme_sigstat_t;

typedef enum {
    GPGME_SIGF_NONE  = 0,
    GPGME_SIGF_LOCAL = 1,
    GPGME_SIGF_NREV  = 2
} gpgme_sigflat_t;

typedef enum {
    GPGME_SIG_MODE_NORMAL = 0,
    GPGME_SIG_MODE_DETACH = 1,
    GPGME_SIG_MODE_CLEAR  = 2
} gpgme_sigmode_t;

typedef enum {
    GPGME_ATTR_KEYID        =  1,
    GPGME_ATTR_FPR          =  2,
    GPGME_ATTR_ALGO         =  3,
    GPGME_ATTR_ALGO_SHORT   =  4,
    GPGME_ATTR_LEN          =  5,
    GPGME_ATTR_CREATED      =  6,
    GPGME_ATTR_EXPIRE       =  7,
    GPGME_ATTR_EXPIRES      =  8,
    GPGME_ATTR_OTRUST       =  9,
    GPGME_ATTR_USERID       = 10,
    GPGME_ATTR_NAME         = 11,
    GPGME_ATTR_EMAIL        = 12,
    GPGME_ATTR_COMMENT      = 13,
    GPGME_ATTR_VALIDITY     = 14,
    GPGME_ATTR_LEVEL        = 15,
    GPGME_ATTR_TYPE         = 16,
    GPGME_ATTR_IS_SECRET    = 17,
    GPGME_ATTR_IS_ULTIMATE  = 18,
    GPGME_ATTR_IS_PROTECTED = 19,
    GPGME_ATTR_MDC          = 20,
    GPGME_ATTR_NOTATION     = 21,
    GPGME_ATTR_POLICY_URL   = 22,
    GPGME_ATTR_REVKEY_ALGO  = 23,
    GPGME_ATTR_REVKEY_FPR   = 24,
    GPGME_ATTR_OPAQUE       = 25,
    GPGME_ATTR_MD           = 26,
    GPGME_ATTR_PHOTO        = 27,
    GPGME_ATTR_CAN_ENCRYPT  = 28,
    GPGME_ATTR_CAN_SIGN     = 29,
    GPGME_ATTR_CAN_CERTIFY  = 30,
    GPGME_ATTR_CAN_AUTH     = 31,
    GPGME_ATTR_DIVERT_CARD  = 32,
    GPGME_ATTR_KEYSERVER    = 33,

    GPGME_ATTR_KEYDAT_NP    = 50,
    GPGME_ATTR_KEYDAT_BITS  = 51,
    GPGME_ATTR_KEYDAT_VAL   = 52,

    GPGME_ATTR_KEY_REVOKED  = 100,
    GPGME_ATTR_KEY_INVALID  = 101,
    GPGME_ATTR_KEY_EXPIRED  = 102,
    GPGME_ATTR_KEY_DISABLED = 103,
    GPGME_ATTR_KEY_CAPS     = 104,
    GPGME_ATTR_KEY_SYMPREFS = 105,
    GPGME_ATTR_KEY_VALIDITY = 106,
    GPGME_ATTR_KEY_CARDNO   = 107,
    GPGME_ATTR_KEY_USABLE   = 108,

    GPGME_ATTR_UID_REVOKED  = 150,
    GPGME_ATTR_UID_INVALID  = 151,
    GPGME_ATTR_UID_PRIMARY  = 152,
    GPGME_ATTR_UID_PREFS    = 153,    
    GPGME_ATTR_UID_CREATED  = 154,
    GPGME_ATTR_UID_HASH	    = 155,

    GPGME_ATTR_SIG_ALGO     = 200,
    GPGME_ATTR_SIG_KEYID    = 201,
    GPGME_ATTR_SIG_ISSUER   = 202,
    GPGME_ATTR_SIG_CREATED  = 203,
    GPGME_ATTR_SIG_EXPIRES  = 204,
    GPGME_ATTR_SIG_STAT     = 205,
    GPGME_ATTR_SIG_USERID   = 206,
    GPGME_ATTR_SIG_CLASS    = 206,
    GPGME_ATTR_SIG_FLAGS    = 207,

    GPGME_ATTR_CARD_AID     = 250,
    GPGME_ATTR_CARD_VER	    = 251,
    GPGME_ATTR_CARD_VENDOR  = 252,
    GPGME_ATTR_CARD_SERIAL  = 253,
    GPGME_ATTR_CARD_NAME    = 254,
    GPGME_ATTR_CARD_NAME2   = 255,
    GPGME_ATTR_CARD_LANG    = 256,
    GPGME_ATTR_CARD_SEX     = 257,
    GPGME_ATTR_CARD_URL     = 258,
    GPGME_ATTR_CARD_LOGIN   = 259,
    GPGME_ATTR_CARD_MAXPINLEN=260,
    GPGME_ATTR_CARD_SIGCOUNT= 261,
    GPGME_ATTR_CARD_FPR     = 262
} gpgme_attr_t;

typedef enum {
    GPGME_EDITKEY_SIGN    =  1,
    GPGME_EDITKEY_LSIGN   =  2,
    GPGME_EDITKEY_NRSIGN  = 19,
    GPGME_EDITKEY_NRLSIGN = 22,
    GPGME_EDITKEY_TSIGN   = 20,
    GPGME_EDITKEY_TRUST   =  3,
    GPGME_EDITKEY_ADDUID  =  4,
    GPGME_EDITKEY_DELUID  =  5,
    GPGME_EDITKEY_DELKEY  =  6,
    GPGME_EDITKEY_ADDKEY  =  7,
    GPGME_EDITKEY_PASSWD  =  8,
    GPGME_EDITKEY_PRIMARY =  9,
    GPGME_EDITKEY_EXPIRE  = 10,
    GPGME_EDITKEY_REVSIG  = 11,
    GPGME_EDITKEY_REVKEY  = 12,
    GPGME_EDITKEY_REVOKE  = 13, /* revoke the whole key */
    GPGME_EDITKEY_ADDREV  = 14,
    GPGME_EDITKEY_ADDPHOTO= 15,
    GPGME_EDITKEY_ENABLE  = 16,
    GPGME_EDITKEY_DISABLE = 17,
    GPGME_EDITKEY_SETPREF = 18,
    GPGME_EDITKEY_DELSIG  = 21,
    GPGME_EDITKEY_KEYSERV = 23
} gpgme_edit_key_t;


typedef enum {
    GPGME_EDITCARD_NAME   = 1,
    GPGME_EDITCARD_NAME2  = 2,
    GPGME_EDITCARD_KEYURL = 3,
    GPGME_EDITCARD_LOGIN  = 4,
    GPGME_EDITCARD_SEX    = 5,
    GPGME_EDITCARD_LANG   = 6,
    GPGME_EDITCARD_APIN   = 7,
    GPGME_EDITCARD_UPIN   = 8,
    GPGME_EDITCARD_GENKEY = 9,
    GPGME_EDITCARD_FETCH  = 10,
    GPGME_EDITCARD_CAFPR  = 11,
    /* change pin */
    GPGME_EDITCARD_CHAPIN = 20,
    GPGME_EDITCARD_CHUPIN = 21,
    GPGME_EDITCARD_UNBPIN = 22 /*unblock PIN*/
} gpgme_edit_card_t;

typedef enum {
    GPGME_CARDFLAG_NONE    = 0,
    GPGME_CARDFLAG_REPLACE = 1,
    GPGME_CARDFLAG_BAKENC  = 2
} gpgme_cardflag_t;

typedef enum {
    GPGME_VALIDITY_UNKNOWN   = 0,
    GPGME_VALIDITY_UNDEFINED = 1,
    GPGME_VALIDITY_NEVER     = 2,
    GPGME_VALIDITY_MARGINAL  = 3,
    GPGME_VALIDITY_FULL      = 4,
    GPGME_VALIDITY_ULTIMATE  = 5,
    GPGME_VALIDITY_REVOKED   = 6,
    GPGME_VALIDITY_EXPIRED   = 7
} gpgme_validity_t;

typedef enum {
    GPGME_TRUST_UNKNOWN   = 0,
    GPGME_TRUST_DONTKNOW  = 1,
    GPGME_TRUST_NEVER     = 2,
    GPGME_TRUST_MARGINAL  = 3,
    GPGME_TRUST_FULLY     = 4,
    GPGME_TRUST_ULTIMATE  = 5,
    GPGME_TRUST_UNDEFINED = 6,
    GPGME_TRUST_EXPIRED   = 7
} GpgmeOwnertrust;

typedef enum {
    GPGME_CTRL_ARMOR           =      1,
    GPGME_CTRL_TEXTMODE        =      2,
    GPGME_CTRL_THROWKEYID      =      4,
    GPGME_CTRL_LISTMODE        =      8,
    GPGME_CTRL_FORCETRUST      =     16,
    GPGME_CTRL_SIGMODE         =     32,
    GPGME_CTRL_CIPHER          =     64,
    GPGME_CTRL_PASSPWD         =    128,
    GPGME_CTRL_TMPFILES        =    256,
    GPGME_CTRL_WITH_SECRET_KEY =    512,
    GPGME_CTRL_FORCEOPT        =   1024,
    GPGME_CTRL_FILE	       =   2048,
    GPGME_CTRL_FPR	       =   4096,
    GPGME_CTRL_S2K	       =   8192,
    GPGME_CTRL_S2K_MODE	       =  16384,
    GPGME_CTRL_S2K_HASH	       =  32678,	
    GPGME_CTRL_INTERACTIVE     =  65536,
    GPGME_CTRL_PIPEMODE	       = 131072,
    GPGME_CTRL_LOGGING         = 262144,
    GPGME_CTRL_NO_COMPR	       = 524288,
    GPGME_CTRL_CB_VAL	       = 1048576
} gpgme_control_t;


typedef enum {
    GPGME_GLOBAL_GPGPROGRAM = 1,
    GPGME_GLOBAL_OPTFILE    = 2,
    GPGME_GLOBAL_PUBRING    = 3,
    GPGME_GLOBAL_SECRING    = 4,
} gpgme_global_t;


typedef enum {
    GPGME_LISTMODE_NORMAL  = 0,
    GPGME_LISTMODE_NOTRUST = 1,
    GPGME_LISTMODE_KEYDATA = 2,
    GPGME_LISTMODE_FAST    = 4
} gpgme_listmode_t;

typedef enum {
    GPGME_LISTOPT_NONE = 0,
    GPGME_LISTOPT_SIGSUBPKT = 1
} gpgme_listopt_t;

typedef enum {
    GPGME_KEYGEN_NONE    = 0,
    GPGME_KEYGEN_DSA_ELG = 1,
    GPGME_KEYGEN_DSA_RSA = 2,
    GPGME_KEYGEN_DSA_SIG = 3,
    GPGME_KEYGEN_RSA_SIG = 4,    
    GPGME_KEYGEN_RSA     = 5,
    GPGME_KEYGEN_RSA_RSA = 6 /*PGP*/
} gpgme_keygen_t;

typedef enum {
    GPGME_IMPSTAT_NPKEYS    =  0,
    GPGME_IMPSTAT_NOSELFSIG =  1,
    GPGME_IMPSTAT_IPKEYS    =  2,
    GPGME_IMPSTAT_NUIDS     =  5,
    GPGME_IMPSTAT_NSUBKEYS  =  6,
    GPGME_IMPSTAT_NSIGS     =  7,
    GPGME_IMPSTAT_NREV      =  8,
    GPGME_IMPSTAT_NSKEYS    =  9, 
    GPGME_IMPSTAT_ISKEYS    = 10
} gpgme_import_status_t;


typedef enum {
    GPGME_REVOKE_INSECURE   = 1,
    GPGME_REVOKE_SUPERSEDED = 2,
    GPGME_REVOKE_NOUSED     = 3
} gpgme_revoke_reason_t;


typedef enum {
    GPGME_OPFLAG_NOSECKEY = 1,
    GPGME_OPFLAG_BADMDC   = 2,
    GPGME_OPFLAG_BADARMOR = 4
} gpgme_op_flags_t;


typedef enum {
    GPGME_TYPE_MESSAGE	    =  1,
    GPGME_TYPE_CLEARSIG	    =  2,
    GPGME_TYPE_SIG	    =  4,
    GPGME_TYPE_PUBKEY	    =  8,
    GPGME_TYPE_SECKEY	    = 16,
    GPGME_TYPE_DASH_ESCAPED = 32
} gpgme_pgptype_t;

typedef enum {
    GPGME_CIPHER_3DES     = 1,
    GPGME_CIPHER_CAST5    = 2,
    GPGME_CIPHER_BLOWFISH = 3,
    GPGME_CIPHER_AES128   = 4,
    GPGME_CIPHER_AES192   = 5,
    GPGME_CIPHER_AES256   = 6,
    GPGME_CIPHER_TWOFISH  = 7
} gpgme_cipher_t;

typedef enum {
    GPGME_S2K_SIMPLE = 0,
    GPGME_S2K_SALTED = 1,
    GPGME_S2K_ITERSALTED = 3
} gpgme_s2kmode_t;


typedef enum {
    GPGME_PK_RSA    =  1,
    GPGME_PK_RSA_E  =  2,
    GPGME_PK_RSA_S  =  3,
    GPGME_PK_ELG_E  = 16,
    GPGME_PK_DSA    = 17,
    GPGME_PK_ELG_ES = 20
} gpgme_pk_cipher_t;

typedef enum {
    GPGME_MD_MD5        = 1,
    GPGME_MD_SHA1	= 2,
    GPGME_MD_RMD160	= 3,
    GPGME_MD_TIGER      = 6,
    GPGME_MD_SHA256     = 8,
    GPGME_MD_SHA384     = 9,
    GPGME_MD_SHA512     = 10
} gpgme_md_t;


#define GPGME_KEY_CANSIGN 0x01
#define GPGME_KEY_CANENCR 0x02

typedef const char *(*gpgme_passphrase_cb_t)( void*, const char *desc, 
					      void * r_hd );

typedef const char *(*gpgme_interactive_cb_t)( void * opaque, 
					      int code, const char * key,
					      const char * val );
typedef void (*gpgme_progress_cb_t)( void * opaque,
                                     const char * what, 
                                     int type, unsigned current, unsigned total );

typedef void (*gpgme_wipe_t)( const char *path, int simple );


/* Initialize the library */
gpgme_error_t gpgme_lib_init (void);
void          gpgme_lib_cleanup (void);

/* Context management */
gpgme_error_t gpgme_new( gpgme_ctx_t * r_ctx );
void          gpgme_release( gpgme_ctx_t ctx );
void	      gpgme_cancel( gpgme_ctx_t ctx );
gpgme_ctx_t   gpgme_wait( gpgme_ctx_t c, int hang );

gpgme_error_t gpgme_get_enc_to_ctx( gpgme_ctx_t c, gpgme_recipients_t *r_ctx );
void gpgme_clear_enc_to_ctx (gpgme_ctx_t ctx);
char * gpgme_get_notation (gpgme_ctx_t ctx);
char * gpgme_get_logging (gpgme_ctx_t ctx);
gpgme_error_t gpgme_check_logging (gpgme_ctx_t ctx);
void gpgme_set_comment (gpgme_ctx_t c, const char * text);
void * gpgme_control( gpgme_ctx_t c, int cmd, int val );
void * gpgme_global_control( gpgme_global_t cmd, void * val );
void gpgme_set_passphrase_cb( gpgme_ctx_t c,
                              gpgme_passphrase_cb_t cb, void *cb_value );
void gpgme_set_interactive_cb( gpgme_ctx_t ctx, 
			       gpgme_interactive_cb_t cb, void * cb_value );
void gpgme_set_progress_cb( gpgme_ctx_t c, 
			    gpgme_progress_cb_t cb, void * cb_value );
void gpgme_set_wipe_fnc( gpgme_ctx_t c, gpgme_wipe_t fnc );
void gpgme_set_local_user( gpgme_ctx_t ctx, const char * name );
int gpgme_get_process_rc( gpgme_ctx_t ctx );

void          gpgme_signers_clear( gpgme_ctx_t ctx );
gpgme_error_t gpgme_signers_add( gpgme_ctx_t ctx, const gpgme_key_t key );
gpgme_key_t   gpgme_signers_enum( const gpgme_ctx_t c, int seq );
gpgme_error_t gpgme_get_sig_key( gpgme_ctx_t ctx, int idx, gpgme_key_t *r_key );
gpgme_error_t gpgme_get_sig_ctx( gpgme_ctx_t ctx, int idx, size_t *r_ncount,
				 gpgme_sig_t *r_sig );

/* Functions to handle recipients */
gpgme_error_t   gpgme_recipients_new( gpgme_recipients_t *r_rset );
void            gpgme_recipients_release( gpgme_recipients_t rset );
gpgme_error_t   gpgme_recipients_add_name( gpgme_recipients_t rset,
                                           const char *name );
gpgme_error_t   gpgme_recipients_add_name_with_validity( gpgme_recipients_t rset,
                                                         const char *name,
                                                         gpgme_validity_t val );
unsigned int    gpgme_recipients_count (const gpgme_recipients_t rset);
const char*     gpgme_recipients_get_name (const gpgme_recipients_t rset, int pos);
gpgme_error_t   gpgme_recipients_enum_open( const gpgme_recipients_t rset,
					    void ** ctx );
const char *    gpgme_recipients_enum_read( const gpgme_recipients_t rset, 
					    void ** ctx );
gpgme_error_t   gpgme_recipients_enum_close( const gpgme_recipients_t rset,
					     void ** ctx );

int          gpgme_recperr_count_items( gpgme_ctx_t ctx );
const char * gpgme_recperr_get( gpgme_ctx_t ctx, int idx, 
			       gpgme_error_t * r_code );

/* Functions to handle data sources */
gpgme_error_t     gpgme_data_new( gpgme_data_t *r_dh );
gpgme_error_t     gpgme_data_new_from_mem( gpgme_data_t *r_dh,
                                           const char *buffer, size_t size,
                                           int copy );
gpgme_error_t     gpgme_data_new_with_read_cb( gpgme_data_t *r_dh,
                                  int (*read_cb)(void*,char *,size_t,size_t*),
                                  void *read_cb_value );

gpgme_error_t     gpgme_data_new_from_file (gpgme_data_t *r_dh,
					    const char *fname);
gpgme_error_t     gpgme_data_new_from_filepart( gpgme_data_t * r_dh,
						const char * fname, FILE * fp,
						long offset, 
						unsigned long length );
void              gpgme_data_release( gpgme_data_t dh );
char *            gpgme_data_release_and_get_mem( gpgme_data_t dh, 
						  size_t * r_len );
gpgme_data_type_t gpgme_data_get_type( gpgme_data_t dh );
gpgme_error_t     gpgme_data_rewind( gpgme_data_t dh );
gpgme_error_t     gpgme_data_read( gpgme_data_t dh,
				   char *buffer, size_t length, size_t *nread );
gpgme_error_t     gpgme_data_write( gpgme_data_t dh,
				    const char *buffer, size_t length );
#define           gpgme_data_putc( dh, c ) do { \
    unsigned char buf[2]; \
    buf[0] = c & 0xFF; buf[1] = 0x00; \
    gpgme_data_write( dh, buf, 1 ); \
} while( 0 )

/* Key and trust functions */
int           gpgme_key_append (gpgme_key_t dst, gpgme_key_t src, int idx);
void          gpgme_key_ref( gpgme_key_t key );
void          gpgme_key_unref( gpgme_key_t key );
void          gpgme_key_release( gpgme_key_t key );
const char *  gpgme_key_get_string_attr( gpgme_key_t key, gpgme_attr_t what,
                                         void **reserved, int idx );
unsigned long gpgme_key_get_ulong_attr( gpgme_key_t key, gpgme_attr_t what,
                                         void **reserved, int idx );

/* Key management functions */
gpgme_error_t gpgme_op_keylist_start( gpgme_ctx_t ctx,
                                      const char *pattern, int secret_only );
gpgme_error_t gpgme_op_keylist_next( gpgme_ctx_t ctx, gpgme_key_t * r_key );

/* Convenience functions for normal usage */
gpgme_error_t gpgme_op_encrypt ( gpgme_ctx_t c, gpgme_recipients_t recp,
                              gpgme_data_t in, gpgme_data_t out );
gpgme_error_t gpgme_op_file_encrypt( gpgme_ctx_t ctx, gpgme_recipients_t rset, 
				     const char * input, const char * output );
gpgme_error_t gpgme_op_files_encrypt( gpgme_ctx_t ctx, gpgme_recipients_t rset,
				      const char ** files, size_t nfiles );
gpgme_error_t gpgme_op_clip_encrypt( gpgme_recipients_t rset, int opts, 
				      gpgme_ctx_t * r_ctx );

gpgme_error_t gpgme_op_list_keys( gpgme_data_t ciph, const char * file, 
				  gpgme_recipients_t * r_rset );
gpgme_error_t gpgme_op_decrypt( gpgme_ctx_t ctx, gpgme_data_t ciph, 
				gpgme_data_t plain );
gpgme_error_t gpgme_op_file_decrypt( gpgme_ctx_t ctx, const char * ciph,
				     const char * plain );
gpgme_error_t gpgme_op_files_decrypt( gpgme_ctx_t ctx, const char ** files, 
				      size_t nfiles );
gpgme_error_t gpgme_op_clip_decrypt( gpgme_ctx_t ctx );

gpgme_error_t gpgme_op_sign ( gpgme_ctx_t c, gpgme_data_t in, gpgme_data_t out,
			      gpgme_sigmode_t mode );
gpgme_error_t gpgme_op_clip_sign( gpgme_ctx_t ctx, gpgme_sigmode_t mode, 
				  const char * keyid, int wraplen );
gpgme_error_t gpgme_op_file_sign( gpgme_ctx_t ctx, gpgme_sigmode_t mode, 
				  const char *input, const char *output );
gpgme_error_t gpgme_op_files_sign( gpgme_ctx_t ctx, gpgme_sigmode_t mode,
				   const char ** files, size_t nfiles );

gpgme_error_t gpgme_op_verify ( gpgme_ctx_t c, gpgme_data_t sig, 
			        gpgme_data_t text );
gpgme_error_t gpgme_op_file_verify( gpgme_ctx_t ctx, gpgme_sigmode_t sigmode, 
				    gpgme_sig_t * r_sigctx,
				    const char * sigfile, const char * datfile );
gpgme_error_t gpgme_op_files_verify( gpgme_ctx_t ctx, const char ** files, 
				     size_t nfiles, gpgme_sig_t * ret_sig );
gpgme_error_t gpgme_op_clip_verify( gpgme_keycache_t cache, 
				    gpgme_sig_t * r_sigctx );
gpgme_error_t gpgme_op_clip_verify_detached( gpgme_keycache_t cache, 
					     gpgme_sig_t * r_sigctx, 
					     const char * data, size_t len );

gpgme_error_t gpgme_op_import( gpgme_ctx_t ctx, gpgme_recipients_t selkeys, 
			       gpgme_data_t keydata );
gpgme_error_t gpgme_op_clip_import( int ctrl, gpgme_recipients_t selkeys, 
				    int * import_res );
gpgme_error_t gpgme_op_file_import( gpgme_ctx_t ctx, gpgme_recipients_t selkeys, 
				    const char *input );
gpgme_error_t gpgme_op_import_list( gpgme_ctx_t ctx, gpgme_data_t in, 
				    gpgme_data_t out );
gpgme_error_t gpgme_op_import_list_next( gpgme_data_t out, 
					 char **pending_line, 
				         gpgme_key_t *r_key );

gpgme_error_t gpgme_op_export ( gpgme_ctx_t c, gpgme_recipients_t recp,
				gpgme_data_t keydata );
gpgme_error_t gpgme_op_file_export( gpgme_ctx_t ctx, gpgme_recipients_t recp,
				    const char *output );
gpgme_error_t gpgme_op_clip_export( gpgme_recipients_t rset );

gpgme_error_t gpgme_op_genkey ( gpgme_ctx_t c, const char *parms,
				gpgme_data_t pubkey, gpgme_data_t seckey );
gpgme_error_t gpgme_op_delete ( gpgme_ctx_t c, const gpgme_key_t key, 
			        int allow_secret );

gpgme_error_t gpgme_op_symenc ( gpgme_ctx_t c, gpgme_data_t in, 
			        gpgme_data_t out );
gpgme_error_t gpgme_op_clip_symenc( gpgme_ctx_t ctx );

gpgme_error_t gpgme_op_sign_encrypt( gpgme_ctx_t ctx, gpgme_recipients_t rset, 
				     gpgme_data_t plain, gpgme_data_t ciph );
gpgme_error_t gpgme_op_clip_sign_encrypt( gpgme_ctx_t c, gpgme_recipients_t rset,
					  const char * keyid, int force_trust );
gpgme_error_t gpgme_op_file_sign_encrypt( gpgme_ctx_t ctx, 
					  gpgme_recipients_t rset,
					  const char * input, 
					  const char * output );
gpgme_error_t gpgme_op_files_sign_encrypt( gpgme_ctx_t ctx, 
					   gpgme_recipients_t recp,
					   const char ** files, size_t nfiles );

/* miscellaneous functions */
const char *gpgme_strerror( gpgme_error_t err );
gpgme_error_t gpgme_file_get_pgptype( const char * filename, gpgme_pgptype_t * r_type );

/* Extensions */
void gpgme_set_debug_mode( int val );
void gpgme_set_edit_ctx( gpgme_ctx_t c, void * edit_ctx, int edit_id );
void gpgme_set_cache_ctx( gpgme_ctx_t c, void * cache_ctx );

/* Signature API */
gpgme_error_t gpgme_sig_new( gpgme_sig_t * r_sig );
void gpgme_sig_release( gpgme_sig_t sig );
unsigned long gpgme_sig_get_ulong_attr( gpgme_sig_t sig, int idx, int what );
const char* gpgme_sig_get_string_attr( gpgme_sig_t sig, int what );
void gpgme_sig_set_key_ctx( gpgme_sig_t sig, gpgme_keycache_t cache );

/* Passphrase operations */
void gpgme_set_passphrase( gpgme_ctx_t c, const char * passphrase );

/* Data operations */
gpgme_error_t gpgme_data_new_from_clipboard (gpgme_data_t * r_dh);
void gpgme_data_release_and_set_clipboard (gpgme_data_t dh);
gpgme_error_t gpgme_data_change_version( gpgme_data_t *r_dh );
size_t gpgme_data_readline( gpgme_data_t dh, char *line, size_t nbytes );
int gpgme_data_wrap_lines( gpgme_data_t *r_dh, size_t wraplen );
gpgme_error_t gpgme_data_release_and_set_file( gpgme_data_t dh, const char *fname );
char* gpgme_data_release_and_return_string( gpgme_data_t dh );
char* gpgme_data_get_as_string (gpgme_data_t dh);
gpgme_error_t gpgme_data_extract_plaintext (gpgme_data_t sig, gpgme_data_t *r_plain);
gpgme_error_t gpgme_data_mail_quote (gpgme_data_t *r_dh);

/* Key edit operations */
gpgme_error_t gpgme_op_delete_key( const char *keyid, int allow_secret );
gpgme_error_t gpgme_op_delete_keys( gpgme_recipients_t rset, int allow_secret );

/* Key caching */
gpgme_error_t gpgme_keycache_new (gpgme_keycache_t * r_ctx);
void gpgme_keycache_release (gpgme_keycache_t ctx);
void gpgme_keycache_set_cb (gpgme_keycache_t ctx,
			    void (*cb)(void *, const char *, int, unsigned, unsigned),
			    void * cb_value1, int cb_value2);
int gpgme_keycache_add_key (gpgme_keycache_t ctx, gpgme_key_t key);
gpgme_error_t gpgme_keycache_find_key (gpgme_keycache_t ctx, const char * pattern,
				       int flags, gpgme_key_t * r_key);
gpgme_error_t gpgme_keycache_delete_key (gpgme_keycache_t ctx, 
					 const char * pattern);
gpgme_error_t gpgme_keycache_init (gpgme_keycache_t ctx, const char * pattern, 
				   int secret);
gpgme_error_t gpgme_keycache_next_key (gpgme_keycache_t ctx, int flags,
				       gpgme_key_t * r_key);
void gpgme_keycache_rewind( gpgme_keycache_t ctx );
int gpgme_keycache_count( gpgme_keycache_t ctx );
gpgme_error_t gpgme_keycache_prepare( gpgme_keycache_t ctx, 
				    const char *pubring, const char * secring );
gpgme_error_t gpgme_keycache_sync (gpgme_keycache_t pub, gpgme_keycache_t sec);
int gpgme_getkey_bykeyid( void *fd_handle, const char * keyid, gpgme_key_t *r_key );


/* Ownertrust */
gpgme_error_t gpgme_op_key_trust_import( gpgme_ctx_t c, gpgme_data_t in );
gpgme_error_t gpgme_op_key_trust_export( gpgme_ctx_t c, gpgme_data_t out );
gpgme_error_t gpgme_op_ownertrust_export_file( const char *output );
gpgme_error_t gpgme_op_ownertrust_import_file( const char *input );

gpgme_error_t gpgme_op_rebuild_cache( void );

/* Import parser */

void gpgme_implist_create_revcert( gpgme_key_t *r_key, gpgme_revcert_t cert, gpgme_key_t pubk );


/* Key generation */
char * gpgme_genkey_params( int keytype, int bits,
			    const char *user, const char *comment,const char *email,
			    const char *expdate, const char *passphrase );
gpgme_error_t gpgme_op_genkey_auto( const char *params, gpgme_progress_cb_t prog_cb, 
				 char **fpr );

/* Key attributes */
const char * gpgme_key_expand_attr( int what, unsigned long attr );

/* Key edit */
gpgme_error_t gpgme_uid_info_new( gpgme_uidinfo_t * r_inf );
void          gpgme_uid_info_release( gpgme_uidinfo_t inf );
gpgme_error_t gpgme_op_editkey_get_info( gpgme_ctx_t ctx, const char * keyid, 
					 gpgme_uidinfo_t * r_inf );
unsigned long gpgme_editkey_get_ulong_attr( gpgme_uidinfo_t inf, int what, int idx );
const char *  gpgme_editkey_get_string_attr( gpgme_uidinfo_t inf, int what, int idx );
int           gpgme_editkey_count_items( gpgme_uidinfo_t inf );
gpgme_error_t gpgme_op_editkey( gpgme_ctx_t ctx, const char *keyid );

/* Card edit */
gpgme_error_t gpgme_editcard_new( gpgme_editcard_t * r_ctx );
void          gpgme_editcard_release( gpgme_editcard_t ctx );
gpgme_error_t gpgme_editcard_control( gpgme_editcard_t ctx, int cmd,
				      const void * val );
gpgme_error_t gpgme_editcard_set_passwd (gpgme_editcard_t ctx, const char * passwd);
gpgme_error_t gpgme_editcard_set_keygen_params( gpgme_editcard_t ctx, int flags,
				  const char * name, const char * email,
				  const char * comment, const char * expdate );
void gpgme_editcard_set_callback( gpgme_editcard_t ctx, 
				const char *(*cb)( int code, void * opaque ),
				void * cb_value );

gpgme_error_t gpgme_op_changepin( gpgme_ctx_t ctx );
gpgme_error_t gpgme_changepin_set( gpgme_editcard_t chpin, int which,
				   const char * pinold, const char * pinnew );

gpgme_error_t gpgme_op_editcard( gpgme_ctx_t ctx );
gpgme_error_t gpgme_op_statuscard( gpgme_ctx_t ctx, gpgme_card_t * ret_card );

gpgme_error_t gpgme_card_new( gpgme_card_t * ret_card );
void          gpgme_card_release( gpgme_card_t card );
const char *  gpgme_card_get_string_attr( gpgme_card_t card, gpgme_attr_t what,
					  void ** reserved, int idx );
unsigned long gpgme_card_get_ulong_attr( gpgme_card_t card, gpgme_attr_t what,
					 void ** reserved, int idx );

/* Key revocation */
gpgme_error_t gpgme_revcert_new( gpgme_revcert_t *r_cert );
void gpgme_revcert_release( gpgme_revcert_t cert );
gpgme_error_t gpgme_revcert_parse( const char *data, gpgme_revcert_t cert, char * keyid );
gpgme_error_t gpgme_revoke_set( gpgme_editkey_t ctx, const char *uid, 
			     const char *text, int reason,  int pgpmode, 
			     const char *pass );
gpgme_error_t gpgme_op_revoke( gpgme_ctx_t c, gpgme_editkey_t rev, gpgme_data_t cert );

/* Misc - version */
gpgme_error_t gpgme_op_version( char **info );
void gpgme_set_gettext_handler( char *(*gettext)( const char *id ) );

/* Misc */
void gpgme_set_pgm_string( const char *name );
int gpgme_key_count_items( gpgme_key_t key, int what );
int gpgme_key_get_cability( gpgme_key_t key, int attr, int keyidx );
int gpgme_key_cability_from_algo( gpgme_pk_cipher_t algo );
gpgme_error_t gpgme_get_import_status( gpgme_ctx_t ctx, int * import_res, 
				       gpgme_recipients_t * r_keys  );
void gpgme_decrypt_get_status( gpgme_ctx_t ctx, char * keyid,
			       gpgme_op_flags_t *r_flags );
void gpgme_decrypt_get_sig_ctx( gpgme_ctx_t ctx, gpgme_sig_t *r_sig );

/* Edit key interface */
gpgme_error_t gpgme_editkey_new( gpgme_editkey_t *ctx );
void gpgme_editkey_release( gpgme_editkey_t ctx );
void gpgme_editkey_make_invalid( gpgme_editkey_t ctx );
int gpgme_editkey_is_valid( gpgme_editkey_t ctx );
int gpgme_editkey_is_secret( gpgme_editkey_t ctx, int val );


void gpgme_editkey_trust_set( gpgme_editkey_t ctx, int val );
void gpgme_editkey_delsig_set (gpgme_editkey_t ctx, int uid, int signo);
gpgme_error_t gpgme_editkey_sign_set (gpgme_editkey_t ctx,  				  
				      const char * passwd,
				      int sigclass,
				      int sigtype,
				      const char * param);
void gpgme_editkey_primary_set( gpgme_editkey_t ctx, int id, const char *passwd );
void gpgme_editkey_adduid_set( gpgme_editkey_t ctx, const char *name,
				const char *email, const char *comment,
				const char *passwd );
void gpgme_editkey_deluid_set_id( gpgme_editkey_t ctx, int id );
void gpgme_editkey_delkey_set_id( gpgme_editkey_t ctx, int id );
gpgme_error_t gpgme_editkey_addkey_set (gpgme_editkey_t ctx, 
					const char * passwd,
					int algo, int size, int valid);
void gpgme_editkey_passwd_set( gpgme_editkey_t ctx,
			       const char * old_pwd, const char * new_pwd, 
			       int allow_empty );
void gpgme_editkey_expire_set( gpgme_editkey_t ctx, int id, int days, 
			       const char *date, const char *passwd );
void gpgme_editkey_revsig_set( gpgme_editkey_t ctx, int id, const char *passwd );
void gpgme_editkey_revkey_set( gpgme_editkey_t ctx, int id, int reason, 
				const char * passwd );
void  gpgme_editkey_setpref_set (gpgme_editkey_t ctx, const char * new_prefs, 
				 int uid_idx, const char * passwd);
gpgme_error_t gpgme_editkey_addrev_set (gpgme_editkey_t ctx, const char * userid,
			       const char * passwd);
void gpgme_editkey_addphoto_set( gpgme_editkey_t ctx, const char *jpegfile,
				 const char * passwd );
void gpgme_editkey_enable_set( gpgme_editkey_t ctx );
void gpgme_editkey_disable_set( gpgme_editkey_t ctx );


/* Clipboard interface */
gpgme_error_t gpgme_clip_istext_avail( int *r_rc );
gpgme_error_t gpgme_clip_is_secured( gpgme_pgptype_t *r_type, int *r_rc );
gpgme_error_t gpgme_clip_get_pgptype( gpgme_pgptype_t *r_type );
gpgme_error_t gpgme_clip_parse_pgpid( const char *data, gpgme_pgptype_t *r_type );
gpgme_error_t gpgme_clip_do_check( int *r_rc );

#ifdef __cplusplus
}
#endif
#endif /* GPGME_H */
