/*
 * Copyright (C) 2008-2014 Free Software Foundation, Inc.
 *
 * Author: Simon Josefsson, Nikos Mavrogiannopoulos
 *
 * This file is part of GnuTLS.
 *
 * GnuTLS is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * GnuTLS is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GnuTLS; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

/* *INDENT-OFF* */

static const char *nc_bad1[] = {
"-----BEGIN CERTIFICATE-----\n"
"MIIDiTCCAkGgAwIBAgIMUwS5Fgr3eOV9V76jMA0GCSqGSIb3DQEBCwUAMA8xDTAL\n"
"BgNVBAMTBENBLTEwIhgPMjAxNDAyMTkxNDAwNTRaGA85OTk5MTIzMTIzNTk1OVow\n"
"EzERMA8GA1UEAxMIc2VydmVyLTIwggFSMA0GCSqGSIb3DQEBAQUAA4IBPwAwggE6\n"
"AoIBMQDBzpbdv7dod/G5JB6eYBtXrvKVibK+qpOlbfS0arSphkyLKDF0341yXAGF\n"
"7ia6yZz2udrzfKdcsdFzsDyByBxsneLtiUYn4WLwcaqFuoLy8AR4jOgDiPU95+bf\n"
"f8iw1BSnNEbFvshWECY3TyceLxA3De+w7cxgiie3Q0aEkuw8dZ4+MSXeXe/je/S7\n"
"OdFY5dXP7iIiqdDlpnPCB6BTfLeUrWOAT4FWM0ngkMijeMEAbPCbTbFw93DeXd2B\n"
"eQPQ1+Qx/VLbzte3tMFbMVNcQARfaO7u4hEvFEe+QOttE0VH4FqgQdx775mWJ1Kl\n"
"eDa+Wf9g9+NhSGMLsJ9dE67bK5CKqC+Z3Ff0QWFAc9wD+yeR4+cCVc1dY/MvaULQ\n"
"Rn1GQIts5amjHyxJvWavfeVTj7sbAgMBAAGjfTB7MAwGA1UdEwEB/wQCMAAwGgYD\n"
"VR0RBBMwEYIPd3d3LmV4YW1wbGUuY29tMA8GA1UdDwEB/wQFAwMHoAAwHQYDVR0O\n"
"BBYEFIonB7QoBQsSXVVzggEQYQWf/rjQMB8GA1UdIwQYMBaAFJ+td6XNJkYpP6B/\n"
"VLp6TDmzEhqnMA0GCSqGSIb3DQEBCwUAA4IBMQB737uD1IwtlMPtjA34mprTcK9N\n"
"rdtWcM3DPBzO5S3ci3XdCn7q3mPjOr42Gvi+71vDdHGBzA7pnGSiJdShWhBvDUT/\n"
"cCng+pIGYZUyGlK90PJfOxUmJzt/V2WFZMz+cor238HaBu8rw9d7a5NO6qeCN1C0\n"
"mAWCKhtNhnvJuGb2hNCAOcJJ+x26Wdh3WBkXIdAQn1zYIxAbgC2RrlQE1YtiqWSh\n"
"orp5kYSqzLENP/XJFnJ5bTxxJIQvBAgn1ZG7Yewv+J5jiF6qePmapHQFebcmvTK+\n"
"AFFSBYl2QYhanBa9n7YW3IxxIA8ZApbO1XsViCYOUqvpBIbCAeelInKtcmapetfl\n"
"b++29Bug798FTrFL9b+KE1qfKRwyp7RRRyJXztuHY1ocio50W3TBrScbim9w\n"
"-----END CERTIFICATE-----\n",
"-----BEGIN CERTIFICATE-----\n"
"MIIDdTCCAi2gAwIBAgIBATANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
"MCIYDzIwMTQwMjE5MTQwMDU0WhgPOTk5OTEyMzEyMzU5NTlaMA8xDTALBgNVBAMT\n"
"BENBLTEwggFSMA0GCSqGSIb3DQEBAQUAA4IBPwAwggE6AoIBMQCwgr6+sQUmsLMy\n"
"YZFM02dZAKR2H/ApcVDN/q+LdWATtNs/WHnLtGGB+BGszc3cvOhf3t8kevPcGpD8\n"
"oeQLUCvHedZiWJDfIMg5IoI1nDg7AZ3LUckFp+bRuLzmadn+yLY1U0FMT6YcNNW8\n"
"pNeef6padDZzVmu/9VxXRElfsoTvRixz95xf4s7+zr7ear7lQrol633RPg52XK8C\n"
"eCZfSNl5oNTLZXMrXoNXMbMx4xqXROB0A5VRUX/gVF1ya/aPiItsRCW8iE8i3+4F\n"
"B0WaUfXDjT9Ivo5k9IKbkiLtNXdc4Pe87lXSgD7IBS84ZxFCxGItz8JwahwOCOyb\n"
"E12OKaRx0w9RxwojetAdHhyxClU8CvYdg1zn+zYzYQRg4FPeYkjC5HRrDB3xpiYH\n"
"/A9jj+mdAgMBAAGjeDB2MA8GA1UdEwEB/wQFMAMBAf8wEgYDVR0eAQH/BAgwBqEE\n"
"MAKCADAPBgNVHQ8BAf8EBQMDBwQAMB0GA1UdDgQWBBSfrXelzSZGKT+gf1S6ekw5\n"
"sxIapzAfBgNVHSMEGDAWgBSOYIy4FnV8n5ISmXZFTnQtp/qC8DANBgkqhkiG9w0B\n"
"AQsFAAOCATEAEAWy+NlgXYXtAJnbkfiYoKtiJ/fKL5zds2KBM0MIDpgMAupM6Y8E\n"
"ax8kgOJiZK7HVvJZDJXv/+40ATCwPWkRN4jzPCIhNize1CiDZQGaqYQb6hFVQZcQ\n"
"O7ICQQnDiSXh4bCGxHFJmgA3rk46iJsAKZT9FVuD5UdnVtgqJq4BziKGl91TA+g4\n"
"u61nQlCXe/FCFdAce1gqw8r5ipFSOxJ5zJwgAS4PejNv/kFnsnR6T0aVpwd+p67u\n"
"mErRoAZWx/3ehjQjfPqATK3fetv+3dvN1dHdM7oMndfdd23XrM24Cx51q83CBqr+\n"
"TglZMAXXTcv+G7Iyljc+1F3WAq2Y1v+zNRHpMRlCRhqAh0cFuo731E/1t1uViolb\n"
"okHo5ttt4r16eXhQQeIc7gQmFswojpx7FQ==\n"
"-----END CERTIFICATE-----\n",
"-----BEGIN CERTIFICATE-----\n"
"MIIDQDCCAfigAwIBAgIBADANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
"MCIYDzIwMTQwMjE5MTQwMDUzWhgPOTk5OTEyMzEyMzU5NTlaMA8xDTALBgNVBAMT\n"
"BENBLTAwggFSMA0GCSqGSIb3DQEBAQUAA4IBPwAwggE6AoIBMQCuejonioMDhNat\n"
"68k+yaeiAF0ohTGwSiHNbXpn0JzfCs8/5PysNDQb+P8wPtuOcYhOCGRXM5ooOzjF\n"
"4s8hT9uFlI8TOlRw8a22PU3ehTYL8PY98tvftErPw9wndojBZ48sdhA7PeouBjlc\n"
"J1FSvjw4vB9EImiqwPz9uLNYkWUOcVUXQYb4iNjddbdh5/0vX2X01+L7O/vVcgiD\n"
"/EnSYsEM/8I4hRUQa7amTTTWPjk4liVBq6TLnAUCumVU2AvkHMkVRMYtvnHPecii\n"
"IRazQeYSPbBHCvhTOkccM5Osb8OntYxuIoR8l2ZjPgEMRrB6lgHr3uIxZKfn6JRi\n"
"YVOvPxZPsnWOAm17JuReyTo7kGh15vRvhmhZ7TeWITTDy3CTWvQLSmiSfe12+9F4\n"
"E4c2v0KLAgMBAAGjQzBBMA8GA1UdEwEB/wQFMAMBAf8wDwYDVR0PAQH/BAUDAwcE\n"
"ADAdBgNVHQ4EFgQUjmCMuBZ1fJ+SEpl2RU50Laf6gvAwDQYJKoZIhvcNAQELBQAD\n"
"ggExAHS8glOsOKVNI/syYmSpWnaahS6f/Fio5B/VVkQSVNPt9UwF+e6FXIlODiZL\n"
"lndhuaNqY1OUNiILXBWXxbwuEf6uPdm+nWdu5tlaaJlmyf5cMYx/ChvqUO5RmK8P\n"
"Dt6oEJBVfkJcqNQQoRYVuTLZvHe+uR87YIHUwBIqAF/2HJAioxpx5BywnLFIeP2A\n"
"WdzfY+Bh0j6e+gwGpX0sOpOob69G2naqHkhlbTgTboxT4qT22wzS+pvFDfXMxsZN\n"
"FvBLMRoCrg36Gqnt8Gt9srHg0iVvtM3OYBCwk+eMPfu1sA6wHfVQTP4DqY9nkHSi\n"
"VCJXNpWIcxuJbJYUiQdobJQc3EnXoI93atVsizbFIymMDh6pV+xBo3IGV/HfaJoi\n"
"yhagfeusFmAjySf3RCuF8eUsCFA=\n"
"-----END CERTIFICATE-----\n",
NULL
};

static const char *nc_bad2[] = {
"-----BEGIN CERTIFICATE-----\n"
"MIIDKTCCAhGgAwIBAgIMUwSneRSvmvAA3BCjMA0GCSqGSIb3DQEBCwUAMA8xDTAL\n"
"BgNVBAMTBENBLTMwIhgPMjAxNDAyMTkxMjQ1NDVaGA85OTk5MTIzMTIzNTk1OVow\n"
"EzERMA8GA1UEAxMIc2VydmVyLTQwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEK\n"
"AoIBAQDgdtv7PLH1jsU7rhvPnIMEAvbC+7RWZcsI9qMVCP+UuiZ8tD/u8swrWGeW\n"
"3Dkbbhohw2dUJzUnCZ1uurb75qmzQvi23UC7j5d1NF0cei1u0P/yVgVauhBVBIy3\n"
"lJmr/sa7LYmuREln8XKOILnBdRGsVMhxzd0moyCxgPw5AkakEmUSSPFl5MuuoQM0\n"
"ESNIY2HX361nYSLrzbvH2XjpkXikBsd69OYXT2mQIVTlKJnGWbs1WIQnlxCXRF2x\n"
"dyrKC1yspa/V2zepAm6krwsYa350+askAtY8e3+cnawVcZ//c1Tv7Q/53WbMNdbi\n"
"k7XsiRekXJXEhoIygm2cL8TSzmr5AgMBAAGjfTB7MAwGA1UdEwEB/wQCMAAwGgYD\n"
"VR0RBBMwEYIPd3d3LmV4YW1wbGUuY29tMA8GA1UdDwEB/wQFAwMHoAAwHQYDVR0O\n"
"BBYEFFa85QCBsDo1OuhFL+X6CVVIeh3SMB8GA1UdIwQYMBaAFOfSQZTbK8EYUHmu\n"
"R372CIe1Bs5vMA0GCSqGSIb3DQEBCwUAA4IBAQABrWwW1PL1KpknTrUxHqXsSdsz\n"
"+Q8zOf/rl5m72DM/bp/Qk0bgU5qEK3OZorPLTN/r9pNqKj/boc9yadkmcsr0xvLZ\n"
"cIGwkwQ8dP9KazUUhBPboiTlXC7qGJU/HZ+9G/9MBHy/XVgkWX5OUH2mEIZS7I6E\n"
"aae2s+vAwYX4Y28L63WJHIyTMO6IavyBWBtFiKCZr5zNN39OhVYe/wRTMrhq7bxR\n"
"llaf9ys13p+ZPcQW2wzOLf42AhTr9A1p4SG2R0CRuaiOQvCWtGcW8JFcU+7LSlZQ\n"
"V4bP/+DffjugqGGQQptvaVsNRXv41f4Mdoy7wZYat/FqHBgtZrHXwjTl17yD\n"
"-----END CERTIFICATE-----\n",
"-----BEGIN CERTIFICATE-----\n"
"MIIDIjCCAgqgAwIBAgIBAzANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0y\n"
"MCIYDzIwMTQwMjE5MTI0NTQ0WhgPOTk5OTEyMzEyMzU5NTlaMA8xDTALBgNVBAMT\n"
"BENBLTMwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDdt0ClpQDZ55Jn\n"
"q1F6mEg2fXssdYDXeuTW2KbU6T4w8z5Ql0qRHPiKpJgnSLGOgGWQo1wcSJY7MjpG\n"
"VA8GvWSYaYR6pccFnNdll/VkTruy4CHFZgXU9sEMp+w6YPLu5KbkpOAI9EDO6yZr\n"
"aOiiCCVS1PKAFrpIU2Kq3cszmwamk3PgXnyfBLC09dBQ6ov3gSXtiuGkx0i0TeN4\n"
"YcWXhigDu20fvuWQxwvBVhvlDPqddu5qcWRRgyeXIT8R+ZO2+Et89RI6TqFPV+U8\n"
"8tzFEQkwPoGKAIupnFBmE68nR8DFdtDSXK4No9xZZtdiGHEOPQlGDOtodRDO7ZU8\n"
"AUg7KduvAgMBAAGjgYQwgYEwDwYDVR0TAQH/BAUwAwEB/zAdBgNVHR4BAf8EEzAR\n"
"oQ8wDYILZXhhbXBsZS5jb20wDwYDVR0PAQH/BAUDAwcEADAdBgNVHQ4EFgQU59JB\n"
"lNsrwRhQea5HfvYIh7UGzm8wHwYDVR0jBBgwFoAUQypdhNSFXwuHeMlFONBWC797\n"
"XwswDQYJKoZIhvcNAQELBQADggEBAIJwTFDAZ/ru9DbaAo++2RJU3sykynrdN4D7\n"
"NP7HqAZc/O+wXjG3rhPz/j4ADjMo/jjPZGvWvC5gtQMB0WTcWR8R7n7kPIb3l9Ac\n"
"OSkjKFZbRiTDFeTklraMc5SBEMwU8DCvoHVfWqIAeAsx9BGkXot6KbLQFYSaL2Mb\n"
"+3OuqEKcBPECimMp1rY3acgs+xYkFlNMTsq2PTzfRysDPAfpv6RAD2vuQ1nFQURB\n"
"vXMmw9be5svlBv5j0kIiEYCgB1kO+HzNSje9F3xxrTvc+IzhRYn23yHHYflX8pWp\n"
"lic0qllEYK+abjatj25c9n1w5keBdiVYlSxr6F9NVwpYZTmxM5g=\n"
"-----END CERTIFICATE-----\n",
"-----BEGIN CERTIFICATE-----\n"
"MIIDIjCCAgqgAwIBAgIBAjANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0x\n"
"MCIYDzIwMTQwMjE5MTI0NTQ0WhgPOTk5OTEyMzEyMzU5NTlaMA8xDTALBgNVBAMT\n"
"BENBLTIwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDGE+FahQcoLS3M\n"
"obarta3FoLhnWZIHQmbVMqK+a7DNYDchv6DZs6yPokidgDcL0LsWa0MFLRw9qEsQ\n"
"fK8eTbck3sXT9PUePnJfeccaRICf8HirRLSoug5w+pqWCQupkLbVNBTgf+Sj2cI6\n"
"jBprmbrPGeilIIRVkZlQJztkTy0FaXuXsIqWxhaPYZrVsAQpD2S/b/ED7riOFbst\n"
"rDfR+9ZKLwDAXM0YZwMe2mKiwMrEuGRAVtlxyrUON6+JQbWh7fLUZvminA0LGDsq\n"
"jWTpvJn5FCZdlIn9aYr8gzunR14vHrffskxqA2SZcKaIxpM5FJrWpY6aT7vZn85C\n"
"5GDSTqLFAgMBAAGjgYQwgYEwDwYDVR0TAQH/BAUwAwEB/zAdBgNVHR4BAf8EEzAR\n"
"oA8wDYILZXhhbXBsZS5jb20wDwYDVR0PAQH/BAUDAwcEADAdBgNVHQ4EFgQUQypd\n"
"hNSFXwuHeMlFONBWC797XwswHwYDVR0jBBgwFoAUZ+qKfar2spXn7Kf7vYBxQXTR\n"
"jvwwDQYJKoZIhvcNAQELBQADggEBAKFn26+aT+yius8A6UWKvX5ufkDUH3X6dUi5\n"
"gUc6tesLvzehwW5Q5kAp+3EA6j8yeLe6PIll9LdY2ry8JrUgPu+ApUCrJ/V0MFPi\n"
"Tg21FJXpP8+Z9wBhqItXDpRHrXY1ne6MTM3OoJkfaky4kp2gQjQ8CIo3xqk4htc7\n"
"1p0eSzBpu1a+mlMiwUsnFBCfqTiafmQwaDZOp8Cl8aTilugMCA998di35CC6D5eM\n"
"k4XITMbyob2+Piy1o/2GNSPtAmIncc/yZ0sTpdqQLzKZn9GojArReu3+dQtTP4XU\n"
"n3bhMgusV4mkBfIPZPdqkfEw4hCP9teEFd++xrR9iJA20ev+caE=\n"
"-----END CERTIFICATE-----\n",
"-----BEGIN CERTIFICATE-----\n"
"MIIDMTCCAhmgAwIBAgIBATANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
"MCIYDzIwMTQwMjE5MTI0NTQzWhgPOTk5OTEyMzEyMzU5NTlaMA8xDTALBgNVBAMT\n"
"BENBLTEwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDjC55IwBZ2SIxD\n"
"hNDjd/c/Ex6upabwLuocu3g5LW6fNhZP7fW4ZZ8eGripXUsQ0xkT4QkqST98vIzh\n"
"B0U3OOAJVyfHP2bQNkA2iewZRavo+O4YlTT7aDnSkGpx+2tEsKxvQAPB41TjjcaC\n"
"SlxTQj/Iij3xus9temIiJbbDuW1g694qtlQZmMElkfvLaRw9P4qqPtIn1S19tWnZ\n"
"TH2IJF6MB8iJghyIJsZ4mRXywIf3B3vXgmSAj7uBQBPnDaUXULC5HtZ+EC+Gi4Wx\n"
"JHr28fUUDvj+smSUydD8rvIJnH7V8b3BwX0HzvxjCY8bJeUcv9Px7P+98aNPZFXT\n"
"MRb0vjEnAgMBAAGjgZMwgZAwDwYDVR0TAQH/BAUwAwEB/zAsBgNVHR4BAf8EIjAg\n"
"oR4wDYILZXhhbXBsZS5uZXQwDYILZXhhbXBsZS5vcmcwDwYDVR0PAQH/BAUDAwcE\n"
"ADAdBgNVHQ4EFgQUZ+qKfar2spXn7Kf7vYBxQXTRjvwwHwYDVR0jBBgwFoAUugqZ\n"
"Ywh0FVP0vSv6j/AVcrNERRcwDQYJKoZIhvcNAQELBQADggEBAGKTyzBjwcnnYemJ\n"
"Hc/W6dvnPTeQgLhBSVHlVbC00OPLduPo+SJXdct6h9Qmh5k8beX4pPW8luC/MwJL\n"
"1gE34RorZdiEGai0VfB1gZEc0b3tIYmSzElBIJ8eCXu52s09Mrzf++D+uAZkOjdv\n"
"OHk/qbHNFmQ4dnM0xBMigz+OgeMuCd3GekHhXhI5IP8Qp7L+mfrt9f3xG0wb+kEl\n"
"echr2S2BCsNXJdlTnCM4yseBTezvBZ1fZzJrY7sL/SQKG/qBOwzNr0f8ZEBabCHp\n"
"MxvDCac6xF+MjLYjgL46J07778SS9pMbTsSkPUR24RVPefhvq2ZfatB677LEuLi/\n"
"O6Wgw5U=\n"
"-----END CERTIFICATE-----\n",
"-----BEGIN CERTIFICATE-----\n"
"MIIC4DCCAcigAwIBAgIBADANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
"MCIYDzIwMTQwMjE5MTI0NTQzWhgPOTk5OTEyMzEyMzU5NTlaMA8xDTALBgNVBAMT\n"
"BENBLTAwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCqxhEkeOnCDeFO\n"
"YVDGaKLLMld5jjcBTM5e5l76tD4JoZidOg6hz2h9T6eLTXEcNW+x9kCThjTHF39i\n"
"CE3w5yCiRJHDETD8RRIgKlt6AOSmpNhDLUb0C/0FxSQkCITYcoyrHYdDN+jcroCC\n"
"hDnOt0hPoaJYAZRAXI582GJ57UZl4o/Yk88333Z+4m8+jKNHIrmaMkxnI9u5pULR\n"
"S9M13YZNEEUKP0qmy/3E++0qXlHXtZfrkRsr1F3GtY1zabfJmUspwHLTrcfnuwKN\n"
"i2NCA+eHaEKbDS4cmWtSR8vpXWpUmJ/MSGGh1QFTSnTKc2/J66FxspXQauqon38x\n"
"1j0/6ktTAgMBAAGjQzBBMA8GA1UdEwEB/wQFMAMBAf8wDwYDVR0PAQH/BAUDAwcE\n"
"ADAdBgNVHQ4EFgQUugqZYwh0FVP0vSv6j/AVcrNERRcwDQYJKoZIhvcNAQELBQAD\n"
"ggEBAFRd7pezulfpZ0MjWxHa6ex7ZB/7azvJ9Guzl5rPoRw8dGXIX+61GZ59inj3\n"
"00dZy2i8EZzsceihBQeCAxa95hkZsW+LnnqjylIczrj94elkRZI7+rKtQXDmhvQM\n"
"VnSXgY0zLZ4e6ykZuW9JpltznlMZ980WcXlqEtAnVUCyVuW9gjO/nPnvAeZrGB51\n"
"bLTu6yisTIGTl0YcYi0GbYiFQwB0FsKiVONu3jcdrxs6J9ER6+l53q6OymgqmR29\n"
"UPRDV/0PawPjFKisFO06AEttjuAMBHp8oVgCmqKy5vAxPNXSBk7Cj2SgqBOcUDBQ\n"
"38485vMMQIMGetTHVODGUrx88+M=\n"
"-----END CERTIFICATE-----\n",
NULL
};

static const char *nc_good[] = {
"-----BEGIN CERTIFICATE-----\n"
"MIIDKTCCAhGgAwIBAgIMUwSn0zQhAl6M9tcWMA0GCSqGSIb3DQEBCwUAMA8xDTAL\n"
"BgNVBAMTBENBLTMwIhgPMjAxNDAyMTkxMjQ3MTVaGA85OTk5MTIzMTIzNTk1OVow\n"
"EzERMA8GA1UEAxMIc2VydmVyLTQwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEK\n"
"AoIBAQCuh+z1GAGt/hg+PaqXWPT+7uTzm3H3OoeEXqc6C5lVq3hxJ/FXhBmdeAuU\n"
"CL8TAeFWl0axBx/JW+5PUpAsi34fWg6ntCnu3Nf/vuK47Mikno9jkOZtSYtzl7eE\n"
"b/Wawj+GYigo4anUNmyOMhmNZtciOScxV3XozKsAGA9a9jRwmIoTVsRPx8XPiGRY\n"
"jqKf0FgjW2dQ6unE/l5NPMftk6rgI/AqgtEmwZDnJVjSSjssomSzj8fcqLexzxpX\n"
"panz3YW+1VK4FvHxc3d5wOxTGh7YvS1c60hME9VafF5PITNjQR2Xw7ckD3sNeX79\n"
"Fw9cMLdqHOH/OM7YKhG7Z0rCEOttAgMBAAGjfTB7MAwGA1UdEwEB/wQCMAAwGgYD\n"
"VR0RBBMwEYIPd3d3LmV4YW1wbGUuY29tMA8GA1UdDwEB/wQFAwMHoAAwHQYDVR0O\n"
"BBYEFN8ZiOy1DE22b9ntdV9FTfGv24P+MB8GA1UdIwQYMBaAFFvNPTPRA3lbnpz0\n"
"LNWkX6XWr6HDMA0GCSqGSIb3DQEBCwUAA4IBAQAn5jt+vV+/AaEqBuYq3kulLzkO\n"
"EdjMWFDoc/kOIWFtxek61gNo9UaddIwK3YCO5rF/9UwXnoz5FihUoTkUPwn5rcmD\n"
"IbGiS1KbqwwkGPBa+LevFuXUha1v/6pOo5tmHJnSBHesqw6WgAc/l12PSQOt4m69\n"
"z3C+td1ZZqDaJXo6ZCJdupLashJzPtN/swMHDA4tDwhMSacrIjuXqPuPouhM1h4+\n"
"6QN2E6W8S3UALFbQoCiUEtuXhzTT6WeyPr/vc0Ic+PIgAoYjFuTjBOhH7zixktLI\n"
"snqmVIkaL15fr0UNfc1BKbDhgMk11Ggg+HvASxIWsxTKMyay/wRaUBpQGoCj\n"
"-----END CERTIFICATE-----\n",
"-----BEGIN CERTIFICATE-----\n"
"MIIDIjCCAgqgAwIBAgIBAzANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0y\n"
"MCIYDzIwMTQwMjE5MTI0NzE1WhgPOTk5OTEyMzEyMzU5NTlaMA8xDTALBgNVBAMT\n"
"BENBLTMwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDwB+ra5HyuxFIq\n"
"C61gqNqmK5ooY7juclsAfhZ5AmdALJHibpVIU89lEhHSJByh+hgAYvG47uRoGZoe\n"
"R7OWWqKovSi6qPgRZJ+6esIwijnSk+YzBSkCsK6WWEpyMkQ5kSLy0+UioAScI1Kl\n"
"LoUWCKE/vvXb0FKH4929w13ufLTfgnSqfOmgR2c4zYqhSJBfKWETLc0lkpvGIlt3\n"
"84Wlr2xdpHeASQt0N7+LvziruzBmdSaHfgO91FxtSUGS2RE2s8xSDhFBmNjjygEz\n"
"62oUYHjyztwlIeGqKjuf2DD4/nQGMrJxLhrrHICGlmVq2yeSoriZpR6lvtQAg45E\n"
"qaZFyIgNAgMBAAGjgYQwgYEwDwYDVR0TAQH/BAUwAwEB/zAdBgNVHR4BAf8EEzAR\n"
"oA8wDYILZXhhbXBsZS5jb20wDwYDVR0PAQH/BAUDAwcEADAdBgNVHQ4EFgQUW809\n"
"M9EDeVuenPQs1aRfpdavocMwHwYDVR0jBBgwFoAUYa0SPy1Augq0D4z9KkxXSwGd\n"
"aLAwDQYJKoZIhvcNAQELBQADggEBAFrEJlcXOiV8/XdpYbX0v1d2VWzROTgDrHoG\n"
"CQ7QcQke5iM85J6kBcidHPE2Joa6N7a+K3rZy6NeODoCi19k5upBGYdPmqAy3RyM\n"
"4Loh4f8GGMDXWR2e9iNu1CoF0UEh6mYslWNELvbCKBNZMbC7i1fS5LnEVydsT3X6\n"
"CHpGSPvE/ouDyzNQZrb8CZFFn+t6Oa1tCLiCHHAjcHlbxJbMd9cmUAxranGGBfCZ\n"
"UuIMHmxXRzcMEmiIU+1GsE3aEUdMKCN158QhoKRSsdG0wGWAo+3KK3N5ID0ASufa\n"
"lj9Lx0zGnBoNWwSfzspyZ9c/e5jizSRnkETKERGfQNG9Zm9/CTQ=\n"
"-----END CERTIFICATE-----\n",
"-----BEGIN CERTIFICATE-----\n"
"MIIDATCCAemgAwIBAgIBAjANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0x\n"
"MCIYDzIwMTQwMjE5MTI0NzE1WhgPOTk5OTEyMzEyMzU5NTlaMA8xDTALBgNVBAMT\n"
"BENBLTIwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCyvjWWVcORUI1M\n"
"9IwACxbTQGzVLyWZunx9Ui+RgGjbSuQxbXopW+ElXWCGHYbegflU95zZf2MtsTam\n"
"6yttESx5rULpEMAvaSV6rjkJEZtc0GN8AHMmCZnDXEqJQenNZOWuNEhquBCGifeD\n"
"GV6jAGvpkuN7p2iPBLVn74/ta9Vi8I0ZcNlfLkJcjIwOOldJ8Uz7DtZuBWnkEqz3\n"
"Pu5pmVfP65gFJFFCMe5I0nVgwSJye2VF2PEW977Q9yBTbC8wHESQtyvAcYWm0b5S\n"
"IeRpN6rPNzS0OE1foBN+JqXhE6gEw7sgtfVixvsSXVlPUFADqJZ+TVXndAbdUOIw\n"
"ixFKNODDAgMBAAGjZDBiMA8GA1UdEwEB/wQFMAMBAf8wDwYDVR0PAQH/BAUDAwcE\n"
"ADAdBgNVHQ4EFgQUYa0SPy1Augq0D4z9KkxXSwGdaLAwHwYDVR0jBBgwFoAU7wb/\n"
"A8DjHcoff+5iq40XgqAnruUwDQYJKoZIhvcNAQELBQADggEBAFCmGqITk+aW16Ua\n"
"hrtHj2vHPVUyCjfVT3CCRSmUzRZdL/XdeVhL4Bdoj7ebx6JCjewEbm/zdHMl710X\n"
"HDTHU2ixkPF0uBcr8pop6joa+qe8c+QwI4hgXUSwjRXjJIVunHPy8zzCKJWj/BTf\n"
"bWuDAkFiIBlW1jDzRh1w3gS0e5Ewx/U9itICJf/OXeCCmOQw+eERFzgfHTXBCDZQ\n"
"XJ7lJ7md3Oow/DUMrKfO6EGJYDEpAqVsO+TCONQ0OIXRMS0WXYm0Q1qkAaEd3Rfo\n"
"nU0hoTQwOECbxELG3Acu5zURPpZY51MooupMw0uoWBhSew40Crw/THvWVtZzO08Q\n"
"Z1hjbl8=\n"
"-----END CERTIFICATE-----\n",
"-----BEGIN CERTIFICATE-----\n"
"MIIDATCCAemgAwIBAgIBATANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
"MCIYDzIwMTQwMjE5MTI0NzE0WhgPOTk5OTEyMzEyMzU5NTlaMA8xDTALBgNVBAMT\n"
"BENBLTEwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQC5fZ/i4x9RaEPc\n"
"NIY260dhNdaVZkBztAPSyDBZhTBXIRRiJwI3rkQvLayfVLMbovVcNl0evsfliI50\n"
"0EoQyOhaV2hhoUzCQhdfyW4Cg0AiUJ2nARCIyMDBybD4sVEaHTEcrJaYfR6X2+1R\n"
"Fk3pvN2ub0zBxdG8mvwMKiP8ZoApzZzZovu1kzgiBRXfTczBDlCyJrvfF8E6GMxk\n"
"lDRgy7M56sOyXjSPEcvL99WWho5zKgRoQ9z/PWtqpXzfDJ1947yKs2gGGAEKPeQ0\n"
"UnhtCvFGW29FINQGE6GVZ0m32babWCFkvZL9bXH0fH95ulsD6pZONPl1EoT0eSr9\n"
"bw6uDTf3AgMBAAGjZDBiMA8GA1UdEwEB/wQFMAMBAf8wDwYDVR0PAQH/BAUDAwcE\n"
"ADAdBgNVHQ4EFgQU7wb/A8DjHcoff+5iq40XgqAnruUwHwYDVR0jBBgwFoAUJSoB\n"
"Zx+Bq4giBE6WypFMma0bnkUwDQYJKoZIhvcNAQELBQADggEBAInI3xPeQ05Ls78i\n"
"kxytkUKaFZpUSPLPt2utI++eyl9CaoPaejU9JdIG9bWbvfrY6C0qrPSNbfQ3O0MK\n"
"KJREBBvsdlRY1O20k06nBGxY5hZ+yRRb4tYqoIamHmOn3Giq45K8u4Df6umgUARZ\n"
"IEBGDssqPoNS+MDIi+GBT9iro/yRfqBfKxj6dRa5+4G6yTy9tMHH1wp3SLeSAn/4\n"
"6F7nsW6MFbQVAetLgQDCo8hZYSXJuMd6hShtlduhKkAuIM/HpjMKMvw4+8iGpjYt\n"
"hBYRoD2xtbsZiTv9agtD8q0H2n0lCz9vK6t0cwFVy5K5zqT1h1ZDXFA0IFlgNKpP\n"
"WbO2kF4=\n"
"-----END CERTIFICATE-----\n",
"-----BEGIN CERTIFICATE-----\n"
"MIIC4DCCAcigAwIBAgIBADANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
"MCIYDzIwMTQwMjE5MTI0NzE0WhgPOTk5OTEyMzEyMzU5NTlaMA8xDTALBgNVBAMT\n"
"BENBLTAwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDz3ju9uSZcNTQM\n"
"dxAOa6UAY3O5ljCqx751te/+nz1Mx0/i9mjuTstq03987YPlDDAdDtW30TQje8J8\n"
"sMkVMnnMpmqxEjfi71wIS29hQtWH5j1oTs/0d8M0LtdZp9dU+BEZlpYIZSj7vvCt\n"
"TeBJjpBC/AR8LbM8+gC2BxxT0wXgEIP1It3O5QoaKhRhs7SlD6RNapaQjWrNDE0q\n"
"loORFMZawncaVS84j+qiD+2dBE4XnM2o7ZLjOwVr5Xxy8yGzW1vucGlcq7ecLRsV\n"
"qZqOZ3rBwj6gNnGtVdUoapCKMMB+KOOF+RM0Iaybj1CBT1ORwwIhXxvpJJNxbLqs\n"
"ee2nmV39AgMBAAGjQzBBMA8GA1UdEwEB/wQFMAMBAf8wDwYDVR0PAQH/BAUDAwcE\n"
"ADAdBgNVHQ4EFgQUJSoBZx+Bq4giBE6WypFMma0bnkUwDQYJKoZIhvcNAQELBQAD\n"
"ggEBAKIRanr7UScoQblASm+btwX3oXuXdYe/xT0z6rnKiU7K8jIeX4EcLTyfhp/0\n"
"v+DvMuVxddq71qXpuFxyTYXyjg6L8Ce0MnwnLz1Wd4uzBqolBq5IvS2PUoO4Ptwg\n"
"XHny7BIDGK4omMz+TatItmvS4r2Ww9UBbi/0MS6d/om0K8Used9s+fUGPjuMfyC0\n"
"y3i4Ba3q0G7x6351i9cSRY0cR9JTpVdyD6Jqd4fZq+0CA6/DgZQ6sInkUmSwDdSn\n"
"RaxXSFegk7++yOHkxODEn+Yf34KRI8RRr7ZsgfolN6b6MpwMymmhoZX3rteH00oX\n"
"NITdnUmuGlGTxXhiC02UeeZedTg=\n"
"-----END CERTIFICATE-----\n",
NULL
};

static const char *v1_intermed_check[] = {
  "-----BEGIN CERTIFICATE-----\n"
  "MIIDGzCCAgOgAwIBAgIIUvuL4ymDgpEwDQYJKoZIhvcNAQELBQAwDjEMMAoGA1UE\n"
  "AxMDQ0ExMCIYDzIwMTQwMjEyMTQ1NzQwWhgPOTk5OTEyMzEyMzU5NTlaMBExDzAN\n"
  "BgNVBAMTBnNlcnZlcjCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBALkh\n"
  "OTKvbV/OQcP9fn02UGzWNLGwS06248rOo+fHqCngf3nl/IefzktuI4Al5Qq9pq42\n"
  "X2oLn/zr8kaO3L0rDcc54nVRuipfsw5nxAmwmjpfwnDgyla4Y88n57dhia/tmFlL\n"
  "rpspg4YJ8Jt4/tGaNVS0OZ57LEoW7/OrqoGM7U3Xxa2QbzaNYMGcSt2ePvccCg13\n"
  "+CJcXxOQcr/cUxyuk9neATJoulFtO8ycpmkLFUdi0WoThBjNCCJ8s7ZuvnGpF4vD\n"
  "3fuvyM2ftiS08B2c5cv6FH9+4I7Elrb++TdVf43F0Awc2pLhm8L1fAuTtMjtbd9w\n"
  "pxs0yaWR8IvQYbWM/XUCAwEAAaN2MHQwDAYDVR0TAQH/BAIwADATBgNVHSUEDDAK\n"
  "BggrBgEFBQcDATAPBgNVHQ8BAf8EBQMDB6AAMB0GA1UdDgQWBBTlqmq9/mceVMMP\n"
  "Z0HtoWZvHDc/tTAfBgNVHSMEGDAWgBRq1Eg1exmWS45j+lFklVwTQe5NsjANBgkq\n"
  "hkiG9w0BAQsFAAOCAQEAAWX2f1XE/bR+DYCMaNAFpTjOFZ54eCBdazUqfVamPRoP\n"
  "/8qyEYpMaA+IpHkJ5tXsx/rdKLgg1kNv/6bXyCwVgVcNBxpt05WUxqFG9xxLLz1K\n"
  "UkbOZUA0/P9GqjRt9HeodP0Hqog1c1d4jgU3Ng7FIn5JXmLNVfl5qXfXCJ2S9WKu\n"
  "pHw0M9TLOKUD2DD2T5K/iZAU0AXGRVUH39e+xsFIoawPWflfN5eNz8uikMQc+kxt\n"
  "DdEMfPZidecToAcMolle53F7zZvqQswfla/3esb/bnndFAqIsnXRpi6Oj06ajzwE\n"
  "TKP745KouHnNPZ3Hz1mPeusn4EJkLfTb5aBT3nJ+Kw==\n"
  "-----END CERTIFICATE-----\n",
  "-----BEGIN CERTIFICATE-----\n"
  "MIICmzCCAYMCCFL7i7wg78R3MA0GCSqGSIb3DQEBCwUAMA4xDDAKBgNVBAMTA0NB\n"
  "MjAiGA8yMDE0MDIxMjE0NTcwMloYDzk5OTkxMjMxMjM1OTU5WjAOMQwwCgYDVQQD\n"
  "EwNDQTEwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDDS0OoKnGy/0Ju\n"
  "U/Q+T3b5RR53oe24OoISkKE4Sia50aHeQZbYitIsQUEvMq06kdjaSA2p8qT29p9n\n"
  "feDYsrhgZkWm4stp2EMkbbkB6k0hBobo4YQiQRa4ttYIsOWLMk/xR0z5ST+UoUv6\n"
  "L/5IOxxJzdpUEbMJpM0Zp8AUZRKnXTc88a/zpPbYiO+LicdhlIKiUvIlnVTlvHOz\n"
  "yN9y5o0Dry9i3IlDSTK8Ls54Nm6v7Z+1F1UwIXkYJCX0WxJ6w/4jHmbiRSitbH9s\n"
  "UqSUm9sHCUakBJA3Y9/9K2SVWNJrG/G4LmZ+Zwr8NdZN3RrxQnWnudL4bTtM0WgY\n"
  "QynV12XDAgMBAAEwDQYJKoZIhvcNAQELBQADggEBAGJbRVVxERtx5Li15n1bdAzW\n"
  "HaDNKt/TkUcZvnfdtggvZLkKhPiNWksQ+9jk1RS71dSZHT9Kb9bIVhaYzaHdI+hG\n"
  "7coftkY66wjD9xLv2DyqnwvuR0S8Uhj9jas5xf/P9S79ZDk61Afg7yX8aLBJpJIH\n"
  "citi1c65C8rYwB8zsF1Zbbh2/6Enty+eFhS1JOuEgUFP1oO2Nj2vh4IqR3yEGdGt\n"
  "Tr57CD/C97fcaeRE4LlHJIMQ9toeZ5Fc9avnOzNIxJd7BPqWWvOnu3TWufj7uaq+\n"
  "CcHTlq9h0NKf9dI1GsxbscJbO3+I+hzOwYfFcNrQ+8BFGbcwx9ZcS2xO3Rx9dbc=\n"
  "-----END CERTIFICATE-----\n",
  "-----BEGIN CERTIFICATE-----\n"
  "MIIC3jCCAcagAwIBAgIBATANBgkqhkiG9w0BAQsFADAOMQwwCgYDVQQDEwNDQTIw\n"
  "IhgPMjAxNDAyMTIxNDU0NTJaGA85OTk5MTIzMTIzNTk1OVowDjEMMAoGA1UEAxMD\n"
  "Q0EyMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAwZPztwmz136OBSTT\n"
  "t4pXys9dTfaOBajrO2s4JcPa0C+7D2wfWD941q1q01TV5+QgLDTF7OO5VSXt2W3p\n"
  "cKlXS/Ll2N8sxTaULcVCodFrHOdfHV1V65VlWhJOnPdpboBtM2V8Iory+d2mNXZT\n"
  "wkcNJ/Z8YBUZCTeR1zaLjq0GAITyJoMiI4+x9Djc+iBDGJarRW7A/JyDN4EFjDzw\n"
  "svdWpHg710I+qtKnlMO/whEmw9r3L486JTSlrrrruUSVGY9UWJpv62az1jbu63d8\n"
  "6/PBp0xbBpiv1xA0qSSquN/THurTZ0Y0MS0vbpnAYkws8YxnFAV1TU4B7AZ0IQId\n"
  "Zjo6HQIDAQABo0MwQTAPBgNVHRMBAf8EBTADAQH/MA8GA1UdDwEB/wQFAwMHBAAw\n"
  "HQYDVR0OBBYEFNVrkTf8WoQKImozc6dBqx8J8tvhMA0GCSqGSIb3DQEBCwUAA4IB\n"
  "AQBpX+j6Fd5aLnMs55qqbFBTWU9bH4/fGh6OVcJztZyvRTOQMLoha9Lsa2C1c1u0\n"
  "kjj3coRWIq8YH8FbOhu0x3pij5dcnn1FQCKcwEmjdDf6ltxplkZXpR86yW2ZyR2W\n"
  "WmIPUrMPJNFkBbgVKFyYoj+9QUyoWHAWNSLJhqBI5v5CRNYIIat1Nt5SuTDm3ggw\n"
  "GUfMH/snytxVq23tj+02pBCdahTqN1w83W1yFX39URChPpl9RZ6HcIg3DFrXhXte\n"
  "lA+/t8l+o7w7POJ4xMyRtbTuGpGHQac+VJBWKFkduY3sbXN2GdQPL6/VvKH115Tr\n"
  "Bos85afmGYPR/gUP0hVSlFzj\n"
  "-----END CERTIFICATE-----\n",
  NULL
};

static const char *v1_root_check[] = {
  "-----BEGIN CERTIFICATE-----\n"
  "MIIDGzCCAgOgAwIBAgIIUvuL4ymDgpEwDQYJKoZIhvcNAQELBQAwDjEMMAoGA1UE\n"
  "AxMDQ0ExMCIYDzIwMTQwMjEyMTQ1NzQwWhgPOTk5OTEyMzEyMzU5NTlaMBExDzAN\n"
  "BgNVBAMTBnNlcnZlcjCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBALkh\n"
  "OTKvbV/OQcP9fn02UGzWNLGwS06248rOo+fHqCngf3nl/IefzktuI4Al5Qq9pq42\n"
  "X2oLn/zr8kaO3L0rDcc54nVRuipfsw5nxAmwmjpfwnDgyla4Y88n57dhia/tmFlL\n"
  "rpspg4YJ8Jt4/tGaNVS0OZ57LEoW7/OrqoGM7U3Xxa2QbzaNYMGcSt2ePvccCg13\n"
  "+CJcXxOQcr/cUxyuk9neATJoulFtO8ycpmkLFUdi0WoThBjNCCJ8s7ZuvnGpF4vD\n"
  "3fuvyM2ftiS08B2c5cv6FH9+4I7Elrb++TdVf43F0Awc2pLhm8L1fAuTtMjtbd9w\n"
  "pxs0yaWR8IvQYbWM/XUCAwEAAaN2MHQwDAYDVR0TAQH/BAIwADATBgNVHSUEDDAK\n"
  "BggrBgEFBQcDATAPBgNVHQ8BAf8EBQMDB6AAMB0GA1UdDgQWBBTlqmq9/mceVMMP\n"
  "Z0HtoWZvHDc/tTAfBgNVHSMEGDAWgBRq1Eg1exmWS45j+lFklVwTQe5NsjANBgkq\n"
  "hkiG9w0BAQsFAAOCAQEAAWX2f1XE/bR+DYCMaNAFpTjOFZ54eCBdazUqfVamPRoP\n"
  "/8qyEYpMaA+IpHkJ5tXsx/rdKLgg1kNv/6bXyCwVgVcNBxpt05WUxqFG9xxLLz1K\n"
  "UkbOZUA0/P9GqjRt9HeodP0Hqog1c1d4jgU3Ng7FIn5JXmLNVfl5qXfXCJ2S9WKu\n"
  "pHw0M9TLOKUD2DD2T5K/iZAU0AXGRVUH39e+xsFIoawPWflfN5eNz8uikMQc+kxt\n"
  "DdEMfPZidecToAcMolle53F7zZvqQswfla/3esb/bnndFAqIsnXRpi6Oj06ajzwE\n"
  "TKP745KouHnNPZ3Hz1mPeusn4EJkLfTb5aBT3nJ+Kw==\n"
  "-----END CERTIFICATE-----\n",
  "-----BEGIN CERTIFICATE-----\n"
  "MIICmzCCAYMCCFL7i7wg78R3MA0GCSqGSIb3DQEBCwUAMA4xDDAKBgNVBAMTA0NB\n"
  "MjAiGA8yMDE0MDIxMjE0NTcwMloYDzk5OTkxMjMxMjM1OTU5WjAOMQwwCgYDVQQD\n"
  "EwNDQTEwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDDS0OoKnGy/0Ju\n"
  "U/Q+T3b5RR53oe24OoISkKE4Sia50aHeQZbYitIsQUEvMq06kdjaSA2p8qT29p9n\n"
  "feDYsrhgZkWm4stp2EMkbbkB6k0hBobo4YQiQRa4ttYIsOWLMk/xR0z5ST+UoUv6\n"
  "L/5IOxxJzdpUEbMJpM0Zp8AUZRKnXTc88a/zpPbYiO+LicdhlIKiUvIlnVTlvHOz\n"
  "yN9y5o0Dry9i3IlDSTK8Ls54Nm6v7Z+1F1UwIXkYJCX0WxJ6w/4jHmbiRSitbH9s\n"
  "UqSUm9sHCUakBJA3Y9/9K2SVWNJrG/G4LmZ+Zwr8NdZN3RrxQnWnudL4bTtM0WgY\n"
  "QynV12XDAgMBAAEwDQYJKoZIhvcNAQELBQADggEBAGJbRVVxERtx5Li15n1bdAzW\n"
  "HaDNKt/TkUcZvnfdtggvZLkKhPiNWksQ+9jk1RS71dSZHT9Kb9bIVhaYzaHdI+hG\n"
  "7coftkY66wjD9xLv2DyqnwvuR0S8Uhj9jas5xf/P9S79ZDk61Afg7yX8aLBJpJIH\n"
  "citi1c65C8rYwB8zsF1Zbbh2/6Enty+eFhS1JOuEgUFP1oO2Nj2vh4IqR3yEGdGt\n"
  "Tr57CD/C97fcaeRE4LlHJIMQ9toeZ5Fc9avnOzNIxJd7BPqWWvOnu3TWufj7uaq+\n"
  "CcHTlq9h0NKf9dI1GsxbscJbO3+I+hzOwYfFcNrQ+8BFGbcwx9ZcS2xO3Rx9dbc=\n"
  "-----END CERTIFICATE-----\n",
  NULL
};

static const char *pathlen_check[] = {
  "-----BEGIN CERTIFICATE-----\n"
  "MIIDGzCCAgOgAwIBAgIIUvuL4ymDgpEwDQYJKoZIhvcNAQELBQAwDjEMMAoGA1UE\n"
  "AxMDQ0ExMCIYDzIwMTQwMjEyMTQ1NzQwWhgPOTk5OTEyMzEyMzU5NTlaMBExDzAN\n"
  "BgNVBAMTBnNlcnZlcjCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBALkh\n"
  "OTKvbV/OQcP9fn02UGzWNLGwS06248rOo+fHqCngf3nl/IefzktuI4Al5Qq9pq42\n"
  "X2oLn/zr8kaO3L0rDcc54nVRuipfsw5nxAmwmjpfwnDgyla4Y88n57dhia/tmFlL\n"
  "rpspg4YJ8Jt4/tGaNVS0OZ57LEoW7/OrqoGM7U3Xxa2QbzaNYMGcSt2ePvccCg13\n"
  "+CJcXxOQcr/cUxyuk9neATJoulFtO8ycpmkLFUdi0WoThBjNCCJ8s7ZuvnGpF4vD\n"
  "3fuvyM2ftiS08B2c5cv6FH9+4I7Elrb++TdVf43F0Awc2pLhm8L1fAuTtMjtbd9w\n"
  "pxs0yaWR8IvQYbWM/XUCAwEAAaN2MHQwDAYDVR0TAQH/BAIwADATBgNVHSUEDDAK\n"
  "BggrBgEFBQcDATAPBgNVHQ8BAf8EBQMDB6AAMB0GA1UdDgQWBBTlqmq9/mceVMMP\n"
  "Z0HtoWZvHDc/tTAfBgNVHSMEGDAWgBRq1Eg1exmWS45j+lFklVwTQe5NsjANBgkq\n"
  "hkiG9w0BAQsFAAOCAQEAAWX2f1XE/bR+DYCMaNAFpTjOFZ54eCBdazUqfVamPRoP\n"
  "/8qyEYpMaA+IpHkJ5tXsx/rdKLgg1kNv/6bXyCwVgVcNBxpt05WUxqFG9xxLLz1K\n"
  "UkbOZUA0/P9GqjRt9HeodP0Hqog1c1d4jgU3Ng7FIn5JXmLNVfl5qXfXCJ2S9WKu\n"
  "pHw0M9TLOKUD2DD2T5K/iZAU0AXGRVUH39e+xsFIoawPWflfN5eNz8uikMQc+kxt\n"
  "DdEMfPZidecToAcMolle53F7zZvqQswfla/3esb/bnndFAqIsnXRpi6Oj06ajzwE\n"
  "TKP745KouHnNPZ3Hz1mPeusn4EJkLfTb5aBT3nJ+Kw==\n"
  "-----END CERTIFICATE-----\n",
  "-----BEGIN CERTIFICATE-----\n"
  "MIICmzCCAYMCCFL7i7wg78R3MA0GCSqGSIb3DQEBCwUAMA4xDDAKBgNVBAMTA0NB\n"
  "MjAiGA8yMDE0MDIxMjE0NTcwMloYDzk5OTkxMjMxMjM1OTU5WjAOMQwwCgYDVQQD\n"
  "EwNDQTEwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDDS0OoKnGy/0Ju\n"
  "U/Q+T3b5RR53oe24OoISkKE4Sia50aHeQZbYitIsQUEvMq06kdjaSA2p8qT29p9n\n"
  "feDYsrhgZkWm4stp2EMkbbkB6k0hBobo4YQiQRa4ttYIsOWLMk/xR0z5ST+UoUv6\n"
  "L/5IOxxJzdpUEbMJpM0Zp8AUZRKnXTc88a/zpPbYiO+LicdhlIKiUvIlnVTlvHOz\n"
  "yN9y5o0Dry9i3IlDSTK8Ls54Nm6v7Z+1F1UwIXkYJCX0WxJ6w/4jHmbiRSitbH9s\n"
  "UqSUm9sHCUakBJA3Y9/9K2SVWNJrG/G4LmZ+Zwr8NdZN3RrxQnWnudL4bTtM0WgY\n"
  "QynV12XDAgMBAAEwDQYJKoZIhvcNAQELBQADggEBAGJbRVVxERtx5Li15n1bdAzW\n"
  "HaDNKt/TkUcZvnfdtggvZLkKhPiNWksQ+9jk1RS71dSZHT9Kb9bIVhaYzaHdI+hG\n"
  "7coftkY66wjD9xLv2DyqnwvuR0S8Uhj9jas5xf/P9S79ZDk61Afg7yX8aLBJpJIH\n"
  "citi1c65C8rYwB8zsF1Zbbh2/6Enty+eFhS1JOuEgUFP1oO2Nj2vh4IqR3yEGdGt\n"
  "Tr57CD/C97fcaeRE4LlHJIMQ9toeZ5Fc9avnOzNIxJd7BPqWWvOnu3TWufj7uaq+\n"
  "CcHTlq9h0NKf9dI1GsxbscJbO3+I+hzOwYfFcNrQ+8BFGbcwx9ZcS2xO3Rx9dbc=\n"
  "-----END CERTIFICATE-----\n",
  "-----BEGIN CERTIFICATE-----\n"
  "MIIC6DCCAdCgAwIBAgIIUvuTdCOiZ3IwDQYJKoZIhvcNAQELBQAwDjEMMAoGA1UE\n"
  "AxMDQ0EyMCIYDzIwMTQwMjEyMTUyOTU3WhgPOTk5OTEyMzEyMzU5NTlaMA4xDDAK\n"
  "BgNVBAMTA0NBMjCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAMGT87cJ\n"
  "s9d+jgUk07eKV8rPXU32jgWo6ztrOCXD2tAvuw9sH1g/eNatatNU1efkICw0xezj\n"
  "uVUl7dlt6XCpV0vy5djfLMU2lC3FQqHRaxznXx1dVeuVZVoSTpz3aW6AbTNlfCKK\n"
  "8vndpjV2U8JHDSf2fGAVGQk3kdc2i46tBgCE8iaDIiOPsfQ43PogQxiWq0VuwPyc\n"
  "gzeBBYw88LL3VqR4O9dCPqrSp5TDv8IRJsPa9y+POiU0pa6667lElRmPVFiab+tm\n"
  "s9Y27ut3fOvzwadMWwaYr9cQNKkkqrjf0x7q02dGNDEtL26ZwGJMLPGMZxQFdU1O\n"
  "AewGdCECHWY6Oh0CAwEAAaNGMEQwEgYDVR0TAQH/BAgwBgEB/wIBADAPBgNVHQ8B\n"
  "Af8EBQMDBwQAMB0GA1UdDgQWBBTVa5E3/FqECiJqM3OnQasfCfLb4TANBgkqhkiG\n"
  "9w0BAQsFAAOCAQEAtQudk32tQ30ldwLy5QyNzwpxTq1izycXGMkh3LvNUQrxmwzl\n"
  "8EPi1d4bxdAi3ghwppImJPZ1aWOrSl9cxl7kH4clq/QdG6bKhhr/40ImZctV35fA\n"
  "Kd1/aDlUUNQIv7cD/T8fb8rMmZ7RPoLsgLcVfodKwafY+X/y4ZacA2uF2L2dX37T\n"
  "etQprA+hjeKu6rej9eb+ERZqYChDvp7FNbJ5fOnIZ9iG1Z714fUeuRDzvosJl6n8\n"
  "aVIRHXdZbhCgKdJTR4bvFPGVFL86xLMkV7jhCImNBN9rmd59wD6g79nTUUoPDM3r\n"
  "rpNkoLGmlBhUorRWbx0YAz9UojNdd4GWMefwZw==\n"
  "-----END CERTIFICATE-----\n",
  NULL
};

static const char *cve_2014_0092_check[] = {
  "-----BEGIN CERTIFICATE-----\n"
  "MIIDtDCCAmygAwIBAgIETeC0yjANBgkqhkiG9w0BAQsFADAZMRcwFQYDVQQDEw5H\n"
  "bnVUTFMgVGVzdCBDQTAeFw0xMTA1MjgwODM5MzlaFw0zODEwMTIwODM5NDBaMC8x\n"
  "LTArBgNVBAMTJEdudUFBQSBUZXN0IFNlcnZlciAoUlNBIGNlcnRpZmljYXRlKTCC\n"
  "AVIwDQYJKoZIhvcNAQEBBQADggE/ADCCAToCggExALRrJ5glr8H/HsqwfvTYvO1D\n"
  "hmdUXdq0HsKQX4M8AhH8E3KFsoikZUELdl8jvoqf/nlLczsux0s8vxbJl1U1F/Oh\n"
  "ckswwuAnlBLzVgDmzoJLEV2kHpv6+rkbKk0Ytbql5gzHqKihbaqIhNyWDrJsHDWq\n"
  "58eUPfnVx8KiDUuzbnr3CF/FCc0Vkxr3mN8qTGaJJO0f0BZjgWWlWDuhzSVim5mB\n"
  "VAgXGOx8LwiiOyhXMp0XRwqG+2KxQZnm+96o6iB+8xvuuuqaIWQpkvKtc+UZBZ03\n"
  "U+IRnxhfIrriiw0AjJ4vp4c9QL5KoqWSCAwuYcBYfJqZ4dasgzklzz4b7eujbZ3L\n"
  "xTjewcdumzQUvjA+gpAeuUqaduTvMwxGojFy9sNhC/iqZ4n0peV2N6Epn4B5qnUC\n"
  "AwEAAaOBjTCBijAMBgNVHRMBAf8EAjAAMBQGA1UdEQQNMAuCCWxvY2FsaG9zdDAT\n"
  "BgNVHSUEDDAKBggrBgEFBQcDATAPBgNVHQ8BAf8EBQMDB6AAMB0GA1UdDgQWBBR2\n"
  "B1hM6rUp9S2ABoyDSoINCeyT3jAfBgNVHSMEGDAWgBRNVrdqAFjxZ5L0pnVVG45T\n"
  "AQPvzzANBgkqhkiG9w0BAQsFAAOCATEBdNWmTsh5uIfngyhOWwm7pK2+vgUMY8nH\n"
  "gMoMFHt0yuxuImcUMXu3LRS1dZSoCJACBpTFGi/Dg2U0qvOHQcEmc3OwNqHB90R3\n"
  "LG5jUSCtq/bYW7h/6Gd9KeWCgZczaHbQ9IPTjLH1dLswVPt+fXKB6Eh0ggSrGATE\n"
  "/wRZT/XgDCW8t4C+2+TmJ8ZEzvU87KAPQ9rUBS1+p3EUAR/FfMApApsEig1IZ+ZD\n"
  "5joaGBW7zh1H0B9mEKidRvD7yuRJyzAcvD25nT15NLW0QR3dEeXosLc720xxJl1h\n"
  "h8NJ7YOvn323mOjR9er4i4D6iJlXmJ8tvN9vakCankWvBzb7plFn2sfMQqICFpRc\n"
  "w075D8hdQxfpGffL2tEeKSgjyNHXS7x3dFhUpN3IQjUi2x4f2e/ZXg==\n"
  "-----END CERTIFICATE-----\n",
  "-----BEGIN CERTIFICATE-----\n"
  "MIIDUDCCAgigAwIBAgIBADANBgkqhkiG9w0BAQsFADAZMRcwFQYDVQQDEw5HbnVU\n"
  "TFMgVGVzdCBDQTAeFw0xMTA1MjgwODM2MzBaFw0zODEwMTIwODM2MzNaMBkxFzAV\n"
  "BgNVBAMTDkdudVRMUyBUZXN0IENBMIIBUjANBgkqhkiG9w0BAQEFAAOCAT8AMIIB\n"
  "OgKCATEAnORCsX1unl//fy2d1054XduIg/3CqVBaT3Hca65SEoDwh0KiPtQoOgZL\n"
  "dKY2cobGs/ojYtOjcs0KnlPYdmtjEh6WEhuJU95v4TQdC4OLMiE56eIGq252hZAb\n"
  "HoTL84Q14DxQWGuzQK830iml7fbw2WcIcRQ8vFGs8SzfXw63+MI6Fq6iMAQIqP08\n"
  "WzGmRRzL5wvCiPhCVkrPmwbXoABub6AAsYwWPJB91M9/lx5gFH5k9/iPfi3s2Kg3\n"
  "F8MOcppqFYjxDSnsfiz6eMh1+bYVIAo367vGVYHigXMEZC2FezlwIHaZzpEoFlY3\n"
  "a7LFJ00yrjQ910r8UE+CEMTYzE40D0olCMo7FA9RCjeO3bUIoYaIdVTUGWEGHWSe\n"
  "oxGei9Gkm6u+ASj8f+i0jxdD2qXsewIDAQABo0MwQTAPBgNVHRMBAf8EBTADAQH/\n"
  "MA8GA1UdDwEB/wQFAwMHBgAwHQYDVR0OBBYEFE1Wt2oAWPFnkvSmdVUbjlMBA+/P\n"
  "MA0GCSqGSIb3DQEBCwUAA4IBMQAesOgjGFi1zOYpA/N3gkUVRcBHDxmN7g2yOcqH\n"
  "VfhFc+e4zhOehR11WCt2RgzNlnYVmV5zBmQBdTAt8Po/MVhLCDW1BULHlLvL0DFc\n"
  "4sB1RlcGeQcCKQa4b+Q9VWf4f6TfuEWZQC5j5stiXjVgOqrOMrzKZ2eKWA4JsL9s\n"
  "V+7ANSZE+hOt1X1mA8moyqe95U2Ecih+nFJSWSBd1WFiEzVnXv4FVWPXbH9HERDK\n"
  "VbasjofWWmQO1YlQPishLgm1IbwqOkOk4sDgoLuUZ4GgP0DDeN6EmRDOzByrv+9u\n"
  "f45Bl9IQf4IJNPLU9lEqjyMOydqT6kBi7fjV5ICuQZ4EeVJsOGuX7PqNyoDzJHLv\n"
  "ferRfNLr6eQSHSxBhS0cVyDjb5gCawK6u7xTU+b7xikEie9k\n"
  "-----END CERTIFICATE-----\n",
  NULL
};

/* Triggers incorrect verification success on older versions */
static const char *cve_2008_4989_chain[] = {
  /* chain[0] */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIB6zCCAVQCCQCgwnB/k0WZrDANBgkqhkiG9w0BAQUFADA9MQswCQYDVQQGEwJE\n"
  "RTEXMBUGA1UEChMOR05VIFRMUyBBdHRhY2sxFTATBgNVBAMTDGludGVybWVkaWF0\n"
  "ZTAeFw0wODExMDMxMjA1MDRaFw0wODEyMDMxMjA1MDRaMDcxCzAJBgNVBAYTAkRF\n"
  "MRcwFQYDVQQKEw5HTlUgVExTIEF0dGFjazEPMA0GA1UEAxMGc2VydmVyMIGfMA0G\n"
  "CSqGSIb3DQEBAQUAA4GNADCBiQKBgQDKdL9g5ErMLOLRCjiomZlNLhy0moWGaKIW\n"
  "aX6vyUIfh8d6FcArHoKoqhmX7ckvod50sOYPojQesDpl7gVaQNA6Ntr1VCcuNPef\n"
  "UKWtEwL0Qu9JbPnUoIYd7mAaqVQgFp6W6yzV/dp63LH4XSdzBMhpZ/EU6vZoE8Sv\n"
  "VLdqj5r6jwIDAQABMA0GCSqGSIb3DQEBBQUAA4GBAH4QRR7sZEbjW00tXYk/3O/Z\n"
  "96AxJNg0F78W5B68gaJrLJ7DTE2RTglscuEq1+2Jyb4AIziwXpYqxgwcP91QpH97\n"
  "XfwdXIcyjYvVLHiKmkQj2zJTY7MeyiEQQ2it8VstZG2fYmi2EiMZIEnyJ2JJ7bA7\n"
  "bF7pG7Cg3oEHUM0H5KUU\n"
  "-----END CERTIFICATE-----\n",
  /* chain[1] (not signed by next cert) */
  "-----BEGIN CERTIFICATE-----\n"
  "MIICADCCAWmgAwIBAgIJAIZ4nkHQAqTFMA0GCSqGSIb3DQEBBQUAMDUxCzAJBgNV\n"
  "BAYTAkRFMRcwFQYDVQQKEw5HTlUgVExTIEF0dGFjazENMAsGA1UEAxMEcm9vdDAe\n"
  "Fw0wODExMDMxMjA0NDVaFw0wODEyMDMxMjA0NDVaMD0xCzAJBgNVBAYTAkRFMRcw\n"
  "FQYDVQQKEw5HTlUgVExTIEF0dGFjazEVMBMGA1UEAxMMaW50ZXJtZWRpYXRlMIGf\n"
  "MA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQDvBpW8sAhIuUmNvcBE6wv/q7MtM1Z9\n"
  "2I1SDL8eJ8I2nPg6BlCX+OIqNruynj8J7uPEQ04ZLwLxNXoyZa8057YFyrKLOvoj\n"
  "5IfBtidsLWYv6PO3qqHJXVvwGdS7PKMuUlsjucCRyXVgQ07ODF7piqoVFi9KD99w\n"
  "AU5+9plGrZNP/wIDAQABoxAwDjAMBgNVHRMEBTADAQH/MA0GCSqGSIb3DQEBBQUA\n"
  "A4GBAGPg+M+8MsB6zHN2o+jAtyqovrTTwmzVWEgfEH/aHC9+imGZRQ5lFNc2vdny\n"
  "AgaJ9/izO5S6Ibb5zUowN2WhoUJOVipuQa2m9AviOgheoU7tmANC9ylm/pRkKy/0\n"
  "n5UVzlKxDhRp/xBb7MWOw3KEQjiAf2Z3wCLcCPUqcJUdJC4v\n"
  "-----END CERTIFICATE-----\n",
  /* chain[2] (trusted CA cert) */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIEIDCCAwigAwIBAgIQNE7VVyDV7exJ9C/ON9srbTANBgkqhkiG9w0BAQUF\n"
  "ADCBqTELMAkGA1UEBhMCVVMxFTATBgNVBAoTDHRoYXd0ZSwgSW5jLjEoMCYG\n"
  "A1UECxMfQ2VydGlmaWNhdGlvbiBTZXJ2aWNlcyBEaXZpc2lvbjE4MDYGA1UE\n"
  "CxMvKGMpIDIwMDYgdGhhd3RlLCBJbmMuIC0gRm9yIGF1dGhvcml6ZWQgdXNl\n"
  "IG9ubHkxHzAdBgNVBAMTFnRoYXd0ZSBQcmltYXJ5IFJvb3QgQ0EwHhcNMDYx\n"
  "MTE3MDAwMDAwWhcNMzYwNzE2MjM1OTU5WjCBqTELMAkGA1UEBhMCVVMxFTAT\n"
  "BgNVBAoTDHRoYXd0ZSwgSW5jLjEoMCYGA1UECxMfQ2VydGlmaWNhdGlvbiBT\n"
  "ZXJ2aWNlcyBEaXZpc2lvbjE4MDYGA1UECxMvKGMpIDIwMDYgdGhhd3RlLCBJ\n"
  "bmMuIC0gRm9yIGF1dGhvcml6ZWQgdXNlIG9ubHkxHzAdBgNVBAMTFnRoYXd0\n"
  "ZSBQcmltYXJ5IFJvb3QgQ0EwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEK\n"
  "AoIBAQCsoPD7gFnUnMekz52hWXMJEEUMDSxuaPFsW0hoSVk3/AszGcJ3f8wQ\n"
  "LZU0HObrTQmnHNK4yZc2AreJ1CRfBsDMRJSUjQJib+ta3RGNKJpchJAQeg29\n"
  "dGYvajig4tVUROsdB58Hum/u6f1OCyn1PoSgAfGcq/gcfomk6KHYcWUNo1F7\n"
  "7rzSImANuVud37r8UVsLr5iy6S7pBOhih94ryNdOwUxkHt3Ph1i6Sk/KaAcd\n"
  "HJ1KxtUvkcx8cXIcxcBn6zL9yZJclNqFwJu/U30rCfSMnZEfl2pSy94JNqR3\n"
  "2HuHUETVPm4pafs5SSYeCaWAe0At6+gnhcn+Yf1+5nyXHdWdAgMBAAGjQjBA\n"
  "MA8GA1UdEwEB/wQFMAMBAf8wDgYDVR0PAQH/BAQDAgEGMB0GA1UdDgQWBBR7\n"
  "W0XPr87Lev0xkhpqtvNG61dIUDANBgkqhkiG9w0BAQUFAAOCAQEAeRHAS7OR\n"
  "tvzw6WfUDW5FvlXok9LOAz/t2iWwHVfLHjp2oEzsUHboZHIMpKnxuIvW1oeE\n"
  "uzLlQRHAd9mzYJ3rG9XRbkREqaYB7FViHXe4XI5ISXycO1cRrK1zN44veFyQ\n"
  "aEfZYGDm/Ac9IiAXxPcW6cTYcvnIc3zfFi8VqT79aie2oetaupgf1eNNZAqd\n"
  "E8hhuvU5HIe6uL17In/2/qxAeeWsEG89jxt5dovEN7MhGITlNgDrYyCZuen+\n"
  "MwS7QcjBAvlEYyCegc5C09Y/LHbTY5xZ3Y+m4Q6gLkH3LpVHz7z9M/P2C2F+\n"
  "fpErgUfCJzDupxBdN49cOSvkBPB7jVaMaA==\n"
  "-----END CERTIFICATE-----\n",
  NULL
};

/* Chain length 3 ends with trusted v1 RSA-MD2 chain */
static const char *verisign_com_chain[] = {
  /* chain[0] */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIGCDCCBPCgAwIBAgIQakrDGzEQ5utI8PxRo5oXHzANBgkqhkiG9w0BAQUFADCB\n"
  "vjELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQL\n"
  "ExZWZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTswOQYDVQQLEzJUZXJtcyBvZiB1c2Ug\n"
  "YXQgaHR0cHM6Ly93d3cudmVyaXNpZ24uY29tL3JwYSAoYykwNjE4MDYGA1UEAxMv\n"
  "VmVyaVNpZ24gQ2xhc3MgMyBFeHRlbmRlZCBWYWxpZGF0aW9uIFNTTCBTR0MgQ0Ew\n"
  "HhcNMDcwNTA5MDAwMDAwWhcNMDkwNTA4MjM1OTU5WjCCAUAxEDAOBgNVBAUTBzI0\n"
  "OTc4ODYxEzARBgsrBgEEAYI3PAIBAxMCVVMxGTAXBgsrBgEEAYI3PAIBAhMIRGVs\n"
  "YXdhcmUxCzAJBgNVBAYTAlVTMQ4wDAYDVQQRFAU5NDA0MzETMBEGA1UECBMKQ2Fs\n"
  "aWZvcm5pYTEWMBQGA1UEBxQNTW91bnRhaW4gVmlldzEiMCAGA1UECRQZNDg3IEVh\n"
  "c3QgTWlkZGxlZmllbGQgUm9hZDEXMBUGA1UEChQOVmVyaVNpZ24sIEluYy4xJTAj\n"
  "BgNVBAsUHFByb2R1Y3Rpb24gU2VjdXJpdHkgU2VydmljZXMxMzAxBgNVBAsUKlRl\n"
  "cm1zIG9mIHVzZSBhdCB3d3cudmVyaXNpZ24uY29tL3JwYSAoYykwNjEZMBcGA1UE\n"
  "AxQQd3d3LnZlcmlzaWduLmNvbTCBnzANBgkqhkiG9w0BAQEFAAOBjQAwgYkCgYEA\n"
  "xxA35ev879drgQCpENGRQ3ARaCPz/WneT9dtMe3qGNvzXQJs6cjm1Bx8XegyW1gB\n"
  "jJX5Zl4WWbr9wpAWZ1YyJ0bEyShIGmkU8fPfbcXYwSyWoWwvE5NRaUB2ztmfAVdv\n"
  "OaGMUKxny2Dnj3tAdaQ+FOeRDJJYg6K1hzczq/otOfsCAwEAAaOCAf8wggH7MAkG\n"
  "A1UdEwQCMAAwHQYDVR0OBBYEFPFaiZNVR0u6UfVO4MsWVfTXzDhnMAsGA1UdDwQE\n"
  "AwIFoDA+BgNVHR8ENzA1MDOgMaAvhi1odHRwOi8vRVZJbnRsLWNybC52ZXJpc2ln\n"
  "bi5jb20vRVZJbnRsMjAwNi5jcmwwRAYDVR0gBD0wOzA5BgtghkgBhvhFAQcXBjAq\n"
  "MCgGCCsGAQUFBwIBFhxodHRwczovL3d3dy52ZXJpc2lnbi5jb20vcnBhMDQGA1Ud\n"
  "JQQtMCsGCCsGAQUFBwMBBggrBgEFBQcDAgYJYIZIAYb4QgQBBgorBgEEAYI3CgMD\n"
  "MB8GA1UdIwQYMBaAFE5DyB127zdTek/yWG+U8zji1b3fMHYGCCsGAQUFBwEBBGow\n"
  "aDArBggrBgEFBQcwAYYfaHR0cDovL0VWSW50bC1vY3NwLnZlcmlzaWduLmNvbTA5\n"
  "BggrBgEFBQcwAoYtaHR0cDovL0VWSW50bC1haWEudmVyaXNpZ24uY29tL0VWSW50\n"
  "bDIwMDYuY2VyMG0GCCsGAQUFBwEMBGEwX6FdoFswWTBXMFUWCWltYWdlL2dpZjAh\n"
  "MB8wBwYFKw4DAhoEFI/l0xqGrI2Oa8PPgGrUSBgsexkuMCUWI2h0dHA6Ly9sb2dv\n"
  "LnZlcmlzaWduLmNvbS92c2xvZ28uZ2lmMA0GCSqGSIb3DQEBBQUAA4IBAQBEueAg\n"
  "xZJrjGPKAZk1NT8VtTn0yi87i9XUnSOnkFkAuI3THDd+cWbNSUzc5uFJg42GhMK7\n"
  "S1Rojm8FHxESovLvimH/w111BKF9wNU2XSOb9KohfYq3GRiQG8O7v9JwIjjLepkc\n"
  "iyITx7sYiJ+kwZlrNBwN6TwVHrONg6NzyzSnxCg+XgKRbJu2PqEQb6uQVkYhb+Oq\n"
  "Vi9d4by9YqpnuXImSffQ0OZ/6s3Rl6vY08zIPqa6OVfjGs/H45ETblzezcUKpX0L\n"
  "cqnOwUB9dVuPhtlX3X/hgz/ROxz96NBwwzha58HUgfEfkVtm+piI6TTI7XxS/7Av\n"
  "nKMfhbyFQYPQ6J9g\n"
  "-----END CERTIFICATE-----\n",
  /* chain[1] */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIGCjCCBPKgAwIBAgIQESoAbTflEG/WynzD77rMGDANBgkqhkiG9w0BAQUFADCB\n"
  "yjELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQL\n"
  "ExZWZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTowOAYDVQQLEzEoYykgMjAwNiBWZXJp\n"
  "U2lnbiwgSW5jLiAtIEZvciBhdXRob3JpemVkIHVzZSBvbmx5MUUwQwYDVQQDEzxW\n"
  "ZXJpU2lnbiBDbGFzcyAzIFB1YmxpYyBQcmltYXJ5IENlcnRpZmljYXRpb24gQXV0\n"
  "aG9yaXR5IC0gRzUwHhcNMDYxMTA4MDAwMDAwWhcNMTYxMTA3MjM1OTU5WjCBvjEL\n"
  "MAkGA1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQLExZW\n"
  "ZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTswOQYDVQQLEzJUZXJtcyBvZiB1c2UgYXQg\n"
  "aHR0cHM6Ly93d3cudmVyaXNpZ24uY29tL3JwYSAoYykwNjE4MDYGA1UEAxMvVmVy\n"
  "aVNpZ24gQ2xhc3MgMyBFeHRlbmRlZCBWYWxpZGF0aW9uIFNTTCBTR0MgQ0EwggEi\n"
  "MA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQC9Voi6iDRkZM/NyrDu5xlzxXLZ\n"
  "u0W8taj/g74cA9vtibcuEBolvFXKQaGfC88ZXnC5XjlLnjEcX4euKqqoK6IbOxAj\n"
  "XxOx3QiMThTag4HjtYzjaO0kZ85Wtqybc5ZE24qMs9bwcZOO23FUSutzWWqPcFEs\n"
  "A5+X0cwRerxiDZUqyRx1V+n1x+q6hDXLx4VafuRN4RGXfQ4gNEXb8aIJ6+s9nriW\n"
  "Q140SwglHkMaotm3igE0PcP45a9PjP/NZfAjTsWXs1zakByChQ0GDcEitnsopAPD\n"
  "TFPRWLxyvAg5/KB2qKjpS26IPeOzMSWMcylIDjJ5Bu09Q/T25On8fb6OCNUfAgMB\n"
  "AAGjggH0MIIB8DAdBgNVHQ4EFgQUTkPIHXbvN1N6T/JYb5TzOOLVvd8wEgYDVR0T\n"
  "AQH/BAgwBgEB/wIBADA9BgNVHSAENjA0MDIGBFUdIAAwKjAoBggrBgEFBQcCARYc\n"
  "aHR0cHM6Ly93d3cudmVyaXNpZ24uY29tL2NwczA9BgNVHR8ENjA0MDKgMKAuhixo\n"
  "dHRwOi8vRVZTZWN1cmUtY3JsLnZlcmlzaWduLmNvbS9wY2EzLWc1LmNybDAgBgNV\n"
  "HSUEGTAXBglghkgBhvhCBAEGCmCGSAGG+EUBCAEwDgYDVR0PAQH/BAQDAgEGMBEG\n"
  "CWCGSAGG+EIBAQQEAwIBBjBtBggrBgEFBQcBDARhMF+hXaBbMFkwVzBVFglpbWFn\n"
  "ZS9naWYwITAfMAcGBSsOAwIaBBSP5dMahqyNjmvDz4Bq1EgYLHsZLjAlFiNodHRw\n"
  "Oi8vbG9nby52ZXJpc2lnbi5jb20vdnNsb2dvLmdpZjApBgNVHREEIjAgpB4wHDEa\n"
  "MBgGA1UEAxMRQ2xhc3MzQ0EyMDQ4LTEtNDgwPQYIKwYBBQUHAQEEMTAvMC0GCCsG\n"
  "AQUFBzABhiFodHRwOi8vRVZTZWN1cmUtb2NzcC52ZXJpc2lnbi5jb20wHwYDVR0j\n"
  "BBgwFoAUf9Nlp8Ld7LvwMAnzQzn6Aq8zMTMwDQYJKoZIhvcNAQEFBQADggEBAFqi\n"
  "sb/rjdQ4qIBywtw4Lqyncfkro7tHu21pbxA2mIzHVi67vKtKm3rW8oKT4BT+is6D\n"
  "t4Pbk4errGV5Sf1XqbHOCR+6EBXECQ5i4/kKJdVkmPDyqA92Mn6R5hjuvOfa0E6N\n"
  "eLvincBZK8DOlQ0kDHLKNF5wIokrSrDxaIfz7kSNKEB3OW5IckUxXWs5DoYC6maZ\n"
  "kzEP32fepp+MnUzOcW86Ifa5ND/5btia9z7a84Ffelxtj3z2mXS3/+QXXe1hXqtI\n"
  "u5aNZkU5tBIK9nDpnHYiS2DpKhs0Sfei1GfAsSatE7rZhAHBq+GObXAWO3eskZq7\n"
  "Gh/aWKfkT8Fhrryi/ks=\n"
  "-----END CERTIFICATE-----\n",
  /* chain[2] */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIE/zCCBGigAwIBAgIQY5Jrio9Agv2swDvTeCmmwDANBgkqhkiG9w0BAQUFADBf\n"
  "MQswCQYDVQQGEwJVUzEXMBUGA1UEChMOVmVyaVNpZ24sIEluYy4xNzA1BgNVBAsT\n"
  "LkNsYXNzIDMgUHVibGljIFByaW1hcnkgQ2VydGlmaWNhdGlvbiBBdXRob3JpdHkw\n"
  "HhcNMDYxMTA4MDAwMDAwWhcNMjExMTA3MjM1OTU5WjCByjELMAkGA1UEBhMCVVMx\n"
  "FzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQLExZWZXJpU2lnbiBUcnVz\n"
  "dCBOZXR3b3JrMTowOAYDVQQLEzEoYykgMjAwNiBWZXJpU2lnbiwgSW5jLiAtIEZv\n"
  "ciBhdXRob3JpemVkIHVzZSBvbmx5MUUwQwYDVQQDEzxWZXJpU2lnbiBDbGFzcyAz\n"
  "IFB1YmxpYyBQcmltYXJ5IENlcnRpZmljYXRpb24gQXV0aG9yaXR5IC0gRzUwggEi\n"
  "MA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCvJAgIKXo1nmAMqudLO07cfLw8\n"
  "RRy7K+D+KQL5VwijZIUVJ/XxrcgxiV0i6CqqpkKzj/i5Vbext0uz/o9+B1fs70Pb\n"
  "ZmIVYc9gDaTY3vjgw2IIPVQT60nKWVSFJuUrjxuf6/WhkcIzSdhDY2pSS9KP6HBR\n"
  "TdGJaXvHcPaz3BJ023tdS1bTlr8Vd6Gw9KIl8q8ckmcY5fQGBO+QueQA5N06tRn/\n"
  "Arr0PO7gi+s3i+z016zy9vA9r911kTMZHRxAy3QkGSGT2RT+rCpSx4/VBEnkjWNH\n"
  "iDxpg8v+R70rfk/Fla4OndTRQ8Bnc+MUCH7lP59zuDMKz10/NIeWiu5T6CUVAgMB\n"
  "AAGjggHKMIIBxjAPBgNVHRMBAf8EBTADAQH/MDEGA1UdHwQqMCgwJqAkoCKGIGh0\n"
  "dHA6Ly9jcmwudmVyaXNpZ24uY29tL3BjYTMuY3JsMA4GA1UdDwEB/wQEAwIBBjBt\n"
  "BggrBgEFBQcBDARhMF+hXaBbMFkwVzBVFglpbWFnZS9naWYwITAfMAcGBSsOAwIa\n"
  "BBSP5dMahqyNjmvDz4Bq1EgYLHsZLjAlFiNodHRwOi8vbG9nby52ZXJpc2lnbi5j\n"
  "b20vdnNsb2dvLmdpZjA9BgNVHSAENjA0MDIGBFUdIAAwKjAoBggrBgEFBQcCARYc\n"
  "aHR0cHM6Ly93d3cudmVyaXNpZ24uY29tL2NwczAdBgNVHQ4EFgQUf9Nlp8Ld7Lvw\n"
  "MAnzQzn6Aq8zMTMwgYAGA1UdIwR5MHehY6RhMF8xCzAJBgNVBAYTAlVTMRcwFQYD\n"
  "VQQKEw5WZXJpU2lnbiwgSW5jLjE3MDUGA1UECxMuQ2xhc3MgMyBQdWJsaWMgUHJp\n"
  "bWFyeSBDZXJ0aWZpY2F0aW9uIEF1dGhvcml0eYIQcLrkHRDZKTS2OMp7A8y6vzAg\n"
  "BgNVHSUEGTAXBglghkgBhvhCBAEGCmCGSAGG+EUBCAEwDQYJKoZIhvcNAQEFBQAD\n"
  "gYEAUNfnArcMK6xK11/59ADJdeNqKOck4skH3qw6WCAYQxfrcn4eobTInOn5G3Gu\n"
  "39g6DapSHmBex2UtZSxvKnJVlWYQgE4P4wGoXdzV69YdCNssXNVVc59DYhDH05dZ\n"
  "P4sJH99fucYDkJjUgRUYw35ww0OFwKgUp3CxiizbXxCqEQc=\n"
  "-----END CERTIFICATE-----\n",
  /* chain[3] (CA) */
  "-----BEGIN CERTIFICATE-----\n"
  "MIICPDCCAaUCEHC65B0Q2Sk0tjjKewPMur8wDQYJKoZIhvcNAQECBQAwXzELMAkG\n"
  "A1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMTcwNQYDVQQLEy5DbGFz\n"
  "cyAzIFB1YmxpYyBQcmltYXJ5IENlcnRpZmljYXRpb24gQXV0aG9yaXR5MB4XDTk2\n"
  "MDEyOTAwMDAwMFoXDTI4MDgwMTIzNTk1OVowXzELMAkGA1UEBhMCVVMxFzAVBgNV\n"
  "BAoTDlZlcmlTaWduLCBJbmMuMTcwNQYDVQQLEy5DbGFzcyAzIFB1YmxpYyBQcmlt\n"
  "YXJ5IENlcnRpZmljYXRpb24gQXV0aG9yaXR5MIGfMA0GCSqGSIb3DQEBAQUAA4GN\n"
  "ADCBiQKBgQDJXFme8huKARS0EN8EQNvjV69qRUCPhAwL0TPZ2RHP7gJYHyX3KqhE\n"
  "BarsAx94f56TuZoAqiN91qyFomNFx3InzPRMxnVx0jnvT0Lwdd8KkMaOIG+YD/is\n"
  "I19wKTakyYbnsZogy1Olhec9vn2a/iRFM9x2Fe0PonFkTGUugWhFpwIDAQABMA0G\n"
  "CSqGSIb3DQEBAgUAA4GBALtMEivPLCYATxQT3ab7/AoRhIzzKBxnki98tsX63/Do\n"
  "lbwdj2wsqFHMc9ikwFPwTtYmwHYBV4GSXiHx0bH/59AhWM1pF+NEHJwZRDmJXNyc\n"
  "AA9WjQKZ7aKQRUzkuxCkPfAyAw7xzvjoyVGM5mKf5p/AfbdynMk2OmufTqj/ZA1k\n"
  "-----END CERTIFICATE-----\n",
  NULL
};

/* Chain length 2 ends with trusted v1 RSA-MD2 cert */
static const char *citibank_com_chain[] = {
  /* chain[0] */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIENDCCA52gAwIBAgIQauOJMlH5Ob2tFZ6rJMBdjjANBgkqhkiG9w0BAQUFADCB\n"
  "ujEfMB0GA1UEChMWVmVyaVNpZ24gVHJ1c3QgTmV0d29yazEXMBUGA1UECxMOVmVy\n"
  "aVNpZ24sIEluYy4xMzAxBgNVBAsTKlZlcmlTaWduIEludGVybmF0aW9uYWwgU2Vy\n"
  "dmVyIENBIC0gQ2xhc3MgMzFJMEcGA1UECxNAd3d3LnZlcmlzaWduLmNvbS9DUFMg\n"
  "SW5jb3JwLmJ5IFJlZi4gTElBQklMSVRZIExURC4oYyk5NyBWZXJpU2lnbjAeFw0w\n"
  "ODA4MjkwMDAwMDBaFw0xMDA4MjkyMzU5NTlaMHgxCzAJBgNVBAYTAlVTMRMwEQYD\n"
  "VQQIEwpOZXcgSmVyc2V5MRIwEAYDVQQHFAlXZWVoYXdrZW4xEjAQBgNVBAoUCUNp\n"
  "dGlncm91cDERMA8GA1UECxQId2hnLW9hazYxGTAXBgNVBAMUEHd3dy5jaXRpYmFu\n"
  "ay5jb20wgZ8wDQYJKoZIhvcNAQEBBQADgY0AMIGJAoGBALQJbSYtbndsIlslFveP\n"
  "IlVNE38HnUD56BHcwfvcb8rQItXeHzYmgOf/RgHPTKG3LEZOxKqM0QpcZtEJ6xwV\n"
  "cTG7Wjw/FrMisN8aO4JWaxe8dFGajstlEMxz43G5zlprb9jzjnbIvvcnz0ILikOQ\n"
  "qmcThopBTs1+d4j7w/yEJo1zAgMBAAGjggF6MIIBdjAJBgNVHRMEAjAAMAsGA1Ud\n"
  "DwQEAwIFoDBGBgNVHR8EPzA9MDugOaA3hjVodHRwOi8vY3JsLnZlcmlzaWduLmNv\n"
  "bS9DbGFzczNJbnRlcm5hdGlvbmFsU2VydmVyLmNybDBEBgNVHSAEPTA7MDkGC2CG\n"
  "SAGG+EUBBxcDMCowKAYIKwYBBQUHAgEWHGh0dHBzOi8vd3d3LnZlcmlzaWduLmNv\n"
  "bS9ycGEwKAYDVR0lBCEwHwYJYIZIAYb4QgQBBggrBgEFBQcDAQYIKwYBBQUHAwIw\n"
  "NAYIKwYBBQUHAQEEKDAmMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC52ZXJpc2ln\n"
  "bi5jb20wbgYIKwYBBQUHAQwEYjBgoV6gXDBaMFgwVhYJaW1hZ2UvZ2lmMCEwHzAH\n"
  "BgUrDgMCGgQUS2u5KJYGDLvQUjibKaxLB4shBRgwJhYkaHR0cDovL2xvZ28udmVy\n"
  "aXNpZ24uY29tL3ZzbG9nbzEuZ2lmMA0GCSqGSIb3DQEBBQUAA4GBAFDXKsxtWkoo\n"
  "HBkNjcCvcnjNAo3Pe+eOtLHb39e5qhkNQLPGA/1/7AofY9KmEtSV2LVGeuuJI4Pi\n"
  "Lg7fPl9Q0OE/oHJpj5JkObBP9Wo1vbrDR2nGWUlCRWm20rH81dTn7OcDxarwGWsR\n"
  "ewTCNmpKYaMx8Q1dyMYunHJApu+fbrHu\n"
  "-----END CERTIFICATE-----\n",
  /* chain[1] */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIDgzCCAuygAwIBAgIQJUuKhThCzONY+MXdriJupDANBgkqhkiG9w0BAQUFADBf\n"
  "MQswCQYDVQQGEwJVUzEXMBUGA1UEChMOVmVyaVNpZ24sIEluYy4xNzA1BgNVBAsT\n"
  "LkNsYXNzIDMgUHVibGljIFByaW1hcnkgQ2VydGlmaWNhdGlvbiBBdXRob3JpdHkw\n"
  "HhcNOTcwNDE3MDAwMDAwWhcNMTExMDI0MjM1OTU5WjCBujEfMB0GA1UEChMWVmVy\n"
  "aVNpZ24gVHJ1c3QgTmV0d29yazEXMBUGA1UECxMOVmVyaVNpZ24sIEluYy4xMzAx\n"
  "BgNVBAsTKlZlcmlTaWduIEludGVybmF0aW9uYWwgU2VydmVyIENBIC0gQ2xhc3Mg\n"
  "MzFJMEcGA1UECxNAd3d3LnZlcmlzaWduLmNvbS9DUFMgSW5jb3JwLmJ5IFJlZi4g\n"
  "TElBQklMSVRZIExURC4oYyk5NyBWZXJpU2lnbjCBnzANBgkqhkiG9w0BAQEFAAOB\n"
  "jQAwgYkCgYEA2IKA6NYZAn0fhRg5JaJlK+G/1AXTvOY2O6rwTGxbtueqPHNFVbLx\n"
  "veqXQu2aNAoV1Klc9UAl3dkHwTKydWzEyruj/lYncUOqY/UwPpMo5frxCTvzt01O\n"
  "OfdcSVq4wR3Tsor+cDCVQsv+K1GLWjw6+SJPkLICp1OcTzTnqwSye28CAwEAAaOB\n"
  "4zCB4DAPBgNVHRMECDAGAQH/AgEAMEQGA1UdIAQ9MDswOQYLYIZIAYb4RQEHAQEw\n"
  "KjAoBggrBgEFBQcCARYcaHR0cHM6Ly93d3cudmVyaXNpZ24uY29tL0NQUzA0BgNV\n"
  "HSUELTArBggrBgEFBQcDAQYIKwYBBQUHAwIGCWCGSAGG+EIEAQYKYIZIAYb4RQEI\n"
  "ATALBgNVHQ8EBAMCAQYwEQYJYIZIAYb4QgEBBAQDAgEGMDEGA1UdHwQqMCgwJqAk\n"
  "oCKGIGh0dHA6Ly9jcmwudmVyaXNpZ24uY29tL3BjYTMuY3JsMA0GCSqGSIb3DQEB\n"
  "BQUAA4GBAAgB7ORolANC8XPxI6I63unx2sZUxCM+hurPajozq+qcBBQHNgYL+Yhv\n"
  "1RPuKSvD5HKNRO3RrCAJLeH24RkFOLA9D59/+J4C3IYChmFOJl9en5IeDCSk9dBw\n"
  "E88mw0M9SR2egi5SX7w+xmYpAY5Okiy8RnUDgqxz6dl+C2fvVFIa\n"
  "-----END CERTIFICATE-----\n",
  /* chain[2] (CA) */
  "-----BEGIN CERTIFICATE-----\n"
  "MIICPDCCAaUCEHC65B0Q2Sk0tjjKewPMur8wDQYJKoZIhvcNAQECBQAwXzELMAkG\n"
  "A1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMTcwNQYDVQQLEy5DbGFz\n"
  "cyAzIFB1YmxpYyBQcmltYXJ5IENlcnRpZmljYXRpb24gQXV0aG9yaXR5MB4XDTk2\n"
  "MDEyOTAwMDAwMFoXDTI4MDgwMTIzNTk1OVowXzELMAkGA1UEBhMCVVMxFzAVBgNV\n"
  "BAoTDlZlcmlTaWduLCBJbmMuMTcwNQYDVQQLEy5DbGFzcyAzIFB1YmxpYyBQcmlt\n"
  "YXJ5IENlcnRpZmljYXRpb24gQXV0aG9yaXR5MIGfMA0GCSqGSIb3DQEBAQUAA4GN\n"
  "ADCBiQKBgQDJXFme8huKARS0EN8EQNvjV69qRUCPhAwL0TPZ2RHP7gJYHyX3KqhE\n"
  "BarsAx94f56TuZoAqiN91qyFomNFx3InzPRMxnVx0jnvT0Lwdd8KkMaOIG+YD/is\n"
  "I19wKTakyYbnsZogy1Olhec9vn2a/iRFM9x2Fe0PonFkTGUugWhFpwIDAQABMA0G\n"
  "CSqGSIb3DQEBAgUAA4GBALtMEivPLCYATxQT3ab7/AoRhIzzKBxnki98tsX63/Do\n"
  "lbwdj2wsqFHMc9ikwFPwTtYmwHYBV4GSXiHx0bH/59AhWM1pF+NEHJwZRDmJXNyc\n"
  "AA9WjQKZ7aKQRUzkuxCkPfAyAw7xzvjoyVGM5mKf5p/AfbdynMk2OmufTqj/ZA1k\n"
  "-----END CERTIFICATE-----\n",
  NULL
};

/* Self-signed certificate */
static const char *pem_self_cert[] = {
  "-----BEGIN CERTIFICATE-----\n"
    "MIIDgjCCAmygAwIBAgIBADALBgkqhkiG9w0BAQUwSzELMAkGA1UEBhMCQlIxFDAS\n"
    "BgNVBAoTC01pbmFzIExpdnJlMSYwJAYDVQQDEx1UaGFkZXUgTGltYSBkZSBTb3V6\n"
    "YSBDYXNjYXJkbzAeFw0wODA1MzAxOTUzNDNaFw0wODExMjYxOTUzNDNaMEsxCzAJ\n"
    "BgNVBAYTAkJSMRQwEgYDVQQKEwtNaW5hcyBMaXZyZTEmMCQGA1UEAxMdVGhhZGV1\n"
    "IExpbWEgZGUgU291emEgQ2FzY2FyZG8wggEfMAsGCSqGSIb3DQEBAQOCAQ4AMIIB\n"
    "CQKCAQC4D934O6wrXJbMyu1w8gu6nN0aNUDGqrX9UgaB/4xVuYhPlhjH0z9Dqic9\n"
    "0pEZmyNCjQmzDSg/hnlY3fBG0i9Iel2oYn1UB4SdcJ2qGkLS87y2ZbMTS1oyMR7/\n"
    "y9l3WGEWqwgjIvOjGstcZo0rCIF8Qr21QGX22KWg2HXlMaZyA9bGtJ+L+x6f2hoo\n"
    "yIPCA30VMvIgHjOSPQJF3iJFE4Uxq1PQ65W91NyI6/bRKFOmFdCUJW8tqqvntYP8\n"
    "hEE08wGlKimFNv7CqZuRI8QuOnhZ7pBXkyvQpW8yHrORlOHxSjkNQKjddt92TCJb\n"
    "1q6eKv2CtCuDLgCuIy0Onr4U9n+hAgMBAAGjeDB2MA8GA1UdEwEB/wQFMAMBAf8w\n"
    "HgYDVR0RBBcwFYITbWFpbC5taW5hc2xpdnJlLm9yZzATBgNVHSUEDDAKBggrBgEF\n"
    "BQcDATAPBgNVHQ8BAf8EBQMDB6QAMB0GA1UdDgQWBBQ/5v42y0jBHUKEfqpPmr5a\n"
    "WsjCGjALBgkqhkiG9w0BAQUDggEBAC/WfO2yK3vM9bG0qFEj8sd0cWiapMhf5PtH\n"
    "jigcPb/OKqSFQVXpAdNiUclPRP79Ih3CuWiXfZ/CW0+k2Z8tyy6AnEQItWvoVh/b\n"
    "8lS7Ph/f9JUYHp2DtgsQWcNQbrUZOPFBu8J4MD6cDWG5Uxwl3YASg30ZdmMDNT8B\n"
    "HshYz0HUOAhYwVSI3J/f7LFhD5OpjSroHgE7wA9UJrerAp9f7e3e9D7kNQ8DlvLP\n"
    "kz6Jh+5M/xD3JO1yl+evaCp3LA+z4M2xiNvtzkAEgj3t6RaJ81Sh5XGiooDYZ14R\n"
    "DgEBYLTUfBYBPzoaahPEdG/f0kUjUBJ34fkBUSjJKURPTHJfDfA=\n"
    "-----END CERTIFICATE-----\n",
  NULL
};

/* Chain length 2, CA constraint FALSE in v3 CA cert)*/
static const char *thea_chain[] = {
  /* chain[0] */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIC7DCCAlWgAwIBAgIBATANBgkqhkiG9w0BAQUFADBnMQswCQYDVQQGEwJERTEM\n"
  "MAoGA1UECBMDUkxQMSAwHgYDVQQKExdUZWNobmlzY2hlIFVuaXZlcnNpdGFldDEb\n"
  "MBkGA1UECxMSRmFjaGJlcmVpY2ggUGh5c2lrMQswCQYDVQQDEwJDQTAeFw0wODA5\n"
  "MTExMDUyMDdaFw0xODA5MDkxMDUyMDdaMIGTMQswCQYDVQQGEwJERTEMMAoGA1UE\n"
  "CBMDUkxQMRcwFQYDVQQHEw5LYWlzZXJzbGF1dGVybjEgMB4GA1UEChMXVGVjaG5p\n"
  "c2NoZSBVbml2ZXJzaXRhZXQxGzAZBgNVBAsTEkZhY2hiZXJlaWNoIFBoeXNpazEe\n"
  "MBwGA1UEAxMVdGhlYS5waHlzaWsudW5pLWtsLmRlMIGfMA0GCSqGSIb3DQEBAQUA\n"
  "A4GNADCBiQKBgQC/gTUrXSeNvuRH+ibdR7zvlCGs+66C6tDaq14SpEDiY/FEw/S4\n"
  "mkhsHohiQkmqpcPJ0FONok7bvJryKZwwhGFHeESvvWjFVNIdxFgf6Jx2McKsRzBD\n"
  "nbgVNeK6bywh2L5WgOeckRm0vUxCwX+jWtETorNHSYnZI9smmBtJ1FIPkQIDAQAB\n"
  "o3sweTAJBgNVHRMEAjAAMCwGCWCGSAGG+EIBDQQfFh1PcGVuU1NMIEdlbmVyYXRl\n"
  "ZCBDZXJ0aWZpY2F0ZTAdBgNVHQ4EFgQUS0IiRshnnlH2bneYeCn6OkY9nZAwHwYD\n"
  "VR0jBBgwFoAU+rCwSUUzK53X9W5otZG4okyY/rswDQYJKoZIhvcNAQEFBQADgYEA\n"
  "g0f6XFxpUL2hncpQcnKorNYdOkZkZHiKqu2SINtla+IbLZFO4nVVO+LKt+RCo2o7\n"
  "tZIMLEU3aCeH5dgSEKQeyL5MPMg3MbA6ezjOBTkT/YgngzM4CMLOKcvAMLncfH/z\n"
  "GYBW1DXijIy1r/SxO0k9zy8OEtKeOOUO0GqQTWuTOOg=\n"
  "-----END CERTIFICATE-----\n",
  /* chain[1] (CA) */
  "-----BEGIN CERTIFICATE-----\n"
  "MIICvzCCAiigAwIBAgIBADANBgkqhkiG9w0BAQUFADBnMQswCQYDVQQGEwJERTEM\n"
  "MAoGA1UECBMDUkxQMSAwHgYDVQQKExdUZWNobmlzY2hlIFVuaXZlcnNpdGFldDEb\n"
  "MBkGA1UECxMSRmFjaGJlcmVpY2ggUGh5c2lrMQswCQYDVQQDEwJDQTAeFw0wODA5\n"
  "MTExMDQ3NDRaFw0xODA5MDkxMDQ3NDRaMGcxCzAJBgNVBAYTAkRFMQwwCgYDVQQI\n"
  "EwNSTFAxIDAeBgNVBAoTF1RlY2huaXNjaGUgVW5pdmVyc2l0YWV0MRswGQYDVQQL\n"
  "ExJGYWNoYmVyZWljaCBQaHlzaWsxCzAJBgNVBAMTAkNBMIGfMA0GCSqGSIb3DQEB\n"
  "AQUAA4GNADCBiQKBgQC76RbqsB5J+VvU1KbBCrkIL3lgY8BxgFvYF3HiHgxtCdqq\n"
  "BmRpAaDBcVAuEb1ihhP68181sYQ1UPMY+zwBwXVNSVvjeBba1JjGmagwPnJXOCay\n"
  "7Cw5orY8KB7U33neEOGrlz1EKQGVaPsr993wGD/7AmntuVuxrRVpzoDP5s0PIwID\n"
  "AQABo3sweTAJBgNVHRMEAjAAMCwGCWCGSAGG+EIBDQQfFh1PcGVuU1NMIEdlbmVy\n"
  "YXRlZCBDZXJ0aWZpY2F0ZTAdBgNVHQ4EFgQU+rCwSUUzK53X9W5otZG4okyY/rsw\n"
  "HwYDVR0jBBgwFoAU+rCwSUUzK53X9W5otZG4okyY/rswDQYJKoZIhvcNAQEFBQAD\n"
  "gYEAUT+LmosiDHGuLAZmY40obam0eexJzn/g++mDy3FMh3WmMBKSsfwFsFsQ4k7N\n"
  "lv1SCfTYeh2hpw/DQzkiYZUkcQI4mBR4hG5Zv56AfYQLGeLtN4VOOCMxguftvzv0\n"
  "kziQa2QW+VzVJqV1gpRCRT30Jaa9s4u6ipO9DT5N03F4CcI=\n"
  "-----END CERTIFICATE-----\n",
  NULL
};

/* Chain length 3 ends with trusted v1 RSA-MD2 cert, similar to
   verisign_com_chain above */
static const char *hbci_chain[] = {
  /* chain[0] */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIEczCCA9ygAwIBAgIQeODCPg2RbK2r7/1KoWjWZzANBgkqhkiG9w0BAQUFADCB\n"
  "ujEfMB0GA1UEChMWVmVyaVNpZ24gVHJ1c3QgTmV0d29yazEXMBUGA1UECxMOVmVy\n"
  "aVNpZ24sIEluYy4xMzAxBgNVBAsTKlZlcmlTaWduIEludGVybmF0aW9uYWwgU2Vy\n"
  "dmVyIENBIC0gQ2xhc3MgMzFJMEcGA1UECxNAd3d3LnZlcmlzaWduLmNvbS9DUFMg\n"
  "SW5jb3JwLmJ5IFJlZi4gTElBQklMSVRZIExURC4oYyk5NyBWZXJpU2lnbjAeFw0w\n"
  "ODA2MTAwMDAwMDBaFw0wOTA3MzAyMzU5NTlaMIG2MQswCQYDVQQGEwJERTEPMA0G\n"
  "A1UECBMGSGVzc2VuMRowGAYDVQQHFBFGcmFua2Z1cnQgYW0gTWFpbjEsMCoGA1UE\n"
  "ChQjU3Bhcmthc3NlbiBJbmZvcm1hdGlrIEdtYkggJiBDby4gS0cxKTAnBgNVBAsU\n"
  "IFRlcm1zIG9mIHVzZSBhdCB3d3cudmVyaXNpZ24uY29tMSEwHwYDVQQDFBhoYmNp\n"
  "LXBpbnRhbi1ycC5zLWhiY2kuZGUwgZ8wDQYJKoZIhvcNAQEBBQADgY0AMIGJAoGB\n"
  "AK1CdQ9lqmChZWaRAInimuK7I36VImTuAVU0N6BIS4a2BbblkiekbVf15GVHGb6e\n"
  "QV06ANN6Nd8XIdfoxi3LoAs8sa+Ku7eoEsRFi/XIU96GgtFlxf3EsVA9RbGdtfer\n"
  "9iJGIBae2mJTlk+5LVg2EQr50PJlBuTgiYFc41xs9O2RAgMBAAGjggF6MIIBdjAJ\n"
  "BgNVHRMEAjAAMAsGA1UdDwQEAwIFoDBGBgNVHR8EPzA9MDugOaA3hjVodHRwOi8v\n"
  "Y3JsLnZlcmlzaWduLmNvbS9DbGFzczNJbnRlcm5hdGlvbmFsU2VydmVyLmNybDBE\n"
  "BgNVHSAEPTA7MDkGC2CGSAGG+EUBBxcDMCowKAYIKwYBBQUHAgEWHGh0dHBzOi8v\n"
  "d3d3LnZlcmlzaWduLmNvbS9ycGEwKAYDVR0lBCEwHwYJYIZIAYb4QgQBBggrBgEF\n"
  "BQcDAQYIKwYBBQUHAwIwNAYIKwYBBQUHAQEEKDAmMCQGCCsGAQUFBzABhhhodHRw\n"
  "Oi8vb2NzcC52ZXJpc2lnbi5jb20wbgYIKwYBBQUHAQwEYjBgoV6gXDBaMFgwVhYJ\n"
  "aW1hZ2UvZ2lmMCEwHzAHBgUrDgMCGgQUS2u5KJYGDLvQUjibKaxLB4shBRgwJhYk\n"
  "aHR0cDovL2xvZ28udmVyaXNpZ24uY29tL3ZzbG9nbzEuZ2lmMA0GCSqGSIb3DQEB\n"
  "BQUAA4GBAJ03R0YAjYzlWm54gMSn6MqJi0mHdLCO2lk3CARwjbg7TEYAZvDsKqTd\n"
  "cRuhNk079BqrQ3QapffeN55SAVrc3mzHO54Nla4n5y6x3XIQXVvRjbJGwmWXsdvr\n"
  "W899F/pBEN30Tgdbmn7JR/iZlGhIJpY9Us1i7rwQhKYir9ZQBdj3\n"
  "-----END CERTIFICATE-----\n",
  /* chain[1] */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIDgzCCAuygAwIBAgIQJUuKhThCzONY+MXdriJupDANBgkqhkiG9w0BAQUFADBf\n"
  "MQswCQYDVQQGEwJVUzEXMBUGA1UEChMOVmVyaVNpZ24sIEluYy4xNzA1BgNVBAsT\n"
  "LkNsYXNzIDMgUHVibGljIFByaW1hcnkgQ2VydGlmaWNhdGlvbiBBdXRob3JpdHkw\n"
  "HhcNOTcwNDE3MDAwMDAwWhcNMTExMDI0MjM1OTU5WjCBujEfMB0GA1UEChMWVmVy\n"
  "aVNpZ24gVHJ1c3QgTmV0d29yazEXMBUGA1UECxMOVmVyaVNpZ24sIEluYy4xMzAx\n"
  "BgNVBAsTKlZlcmlTaWduIEludGVybmF0aW9uYWwgU2VydmVyIENBIC0gQ2xhc3Mg\n"
  "MzFJMEcGA1UECxNAd3d3LnZlcmlzaWduLmNvbS9DUFMgSW5jb3JwLmJ5IFJlZi4g\n"
  "TElBQklMSVRZIExURC4oYyk5NyBWZXJpU2lnbjCBnzANBgkqhkiG9w0BAQEFAAOB\n"
  "jQAwgYkCgYEA2IKA6NYZAn0fhRg5JaJlK+G/1AXTvOY2O6rwTGxbtueqPHNFVbLx\n"
  "veqXQu2aNAoV1Klc9UAl3dkHwTKydWzEyruj/lYncUOqY/UwPpMo5frxCTvzt01O\n"
  "OfdcSVq4wR3Tsor+cDCVQsv+K1GLWjw6+SJPkLICp1OcTzTnqwSye28CAwEAAaOB\n"
  "4zCB4DAPBgNVHRMECDAGAQH/AgEAMEQGA1UdIAQ9MDswOQYLYIZIAYb4RQEHAQEw\n"
  "KjAoBggrBgEFBQcCARYcaHR0cHM6Ly93d3cudmVyaXNpZ24uY29tL0NQUzA0BgNV\n"
  "HSUELTArBggrBgEFBQcDAQYIKwYBBQUHAwIGCWCGSAGG+EIEAQYKYIZIAYb4RQEI\n"
  "ATALBgNVHQ8EBAMCAQYwEQYJYIZIAYb4QgEBBAQDAgEGMDEGA1UdHwQqMCgwJqAk\n"
  "oCKGIGh0dHA6Ly9jcmwudmVyaXNpZ24uY29tL3BjYTMuY3JsMA0GCSqGSIb3DQEB\n"
  "BQUAA4GBAAgB7ORolANC8XPxI6I63unx2sZUxCM+hurPajozq+qcBBQHNgYL+Yhv\n"
  "1RPuKSvD5HKNRO3RrCAJLeH24RkFOLA9D59/+J4C3IYChmFOJl9en5IeDCSk9dBw\n"
  "E88mw0M9SR2egi5SX7w+xmYpAY5Okiy8RnUDgqxz6dl+C2fvVFIa\n"
  "-----END CERTIFICATE-----\n",
  /* chain[2] */
  "-----BEGIN CERTIFICATE-----\n"
  "MIICPDCCAaUCEHC65B0Q2Sk0tjjKewPMur8wDQYJKoZIhvcNAQECBQAwXzELMAkG\n"
  "A1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMTcwNQYDVQQLEy5DbGFz\n"
  "cyAzIFB1YmxpYyBQcmltYXJ5IENlcnRpZmljYXRpb24gQXV0aG9yaXR5MB4XDTk2\n"
  "MDEyOTAwMDAwMFoXDTI4MDgwMTIzNTk1OVowXzELMAkGA1UEBhMCVVMxFzAVBgNV\n"
  "BAoTDlZlcmlTaWduLCBJbmMuMTcwNQYDVQQLEy5DbGFzcyAzIFB1YmxpYyBQcmlt\n"
  "YXJ5IENlcnRpZmljYXRpb24gQXV0aG9yaXR5MIGfMA0GCSqGSIb3DQEBAQUAA4GN\n"
  "ADCBiQKBgQDJXFme8huKARS0EN8EQNvjV69qRUCPhAwL0TPZ2RHP7gJYHyX3KqhE\n"
  "BarsAx94f56TuZoAqiN91qyFomNFx3InzPRMxnVx0jnvT0Lwdd8KkMaOIG+YD/is\n"
  "I19wKTakyYbnsZogy1Olhec9vn2a/iRFM9x2Fe0PonFkTGUugWhFpwIDAQABMA0G\n"
  "CSqGSIb3DQEBAgUAA4GBALtMEivPLCYATxQT3ab7/AoRhIzzKBxnki98tsX63/Do\n"
  "lbwdj2wsqFHMc9ikwFPwTtYmwHYBV4GSXiHx0bH/59AhWM1pF+NEHJwZRDmJXNyc\n"
  "AA9WjQKZ7aKQRUzkuxCkPfAyAw7xzvjoyVGM5mKf5p/AfbdynMk2OmufTqj/ZA1k\n"
  "-----END CERTIFICATE-----\n",
  NULL
};

/* End-entity cert signed using RSA-MD5. */
static const char *mayfirst_chain[] = {
  /* chain[0] */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIDVTCCAr6gAwIBAgIDCHp1MA0GCSqGSIb3DQEBBAUAMFoxCzAJBgNVBAYTAlVT\n"
  "MRwwGgYDVQQKExNFcXVpZmF4IFNlY3VyZSBJbmMuMS0wKwYDVQQDEyRFcXVpZmF4\n"
  "IFNlY3VyZSBHbG9iYWwgZUJ1c2luZXNzIENBLTEwHhcNMDgwNTE5MDUyOTE5WhcN\n"
  "MDkxMDE5MDUyOTE5WjCBxDELMAkGA1UEBhMCVVMxHTAbBgNVBAoTFHN1cHBvcnQu\n"
  "bWF5Zmlyc3Qub3JnMRMwEQYDVQQLEwpHVDY5MDc5ODgwMTEwLwYDVQQLEyhTZWUg\n"
  "d3d3LnJhcGlkc3NsLmNvbS9yZXNvdXJjZXMvY3BzIChjKTA3MS8wLQYDVQQLEyZE\n"
  "b21haW4gQ29udHJvbCBWYWxpZGF0ZWQgLSBSYXBpZFNTTChSKTEdMBsGA1UEAxMU\n"
  "c3VwcG9ydC5tYXlmaXJzdC5vcmcwgZ8wDQYJKoZIhvcNAQEBBQADgY0AMIGJAoGB\n"
  "AN0TWIZwJ/hIfMHc08/bBMlzZ5WucJqEvxU/ZnxPo/H6V/m4v1iLpM2hip2c5cg0\n"
  "BcEMc/TBHQ1UEV8sb0Lh91kWfiMB1Sp+L2Fpz/wnhsivXC5j6jq9IcPqmOZOXBYX\n"
  "k04W1B6FKTvk9KrZJ0at2J44hp4SsAfWQI0eCKuas+R1AgMBAAGjgb0wgbowDgYD\n"
  "VR0PAQH/BAQDAgTwMB0GA1UdDgQWBBS0D4iuCxp35TLADTkINq2AhgTYVTA7BgNV\n"
  "HR8ENDAyMDCgLqAshipodHRwOi8vY3JsLmdlb3RydXN0LmNvbS9jcmxzL2dsb2Jh\n"
  "bGNhMS5jcmwwHwYDVR0jBBgwFoAUvqigdHJQa0S3ySPY+6j/s1draGwwHQYDVR0l\n"
  "BBYwFAYIKwYBBQUHAwEGCCsGAQUFBwMCMAwGA1UdEwEB/wQCMAAwDQYJKoZIhvcN\n"
  "AQEEBQADgYEAXNWYnrO1mZgBSCwPlWhVa2aOKGCFmehLIcAPEBN+8xhXuOeigYBm\n"
  "ic5ShCO583ttgHNCV3Y5dW9sNhv1US4vSb6soKjgUlG11fJKUqU8mwFKvbs7TUSq\n"
  "j6h+1uvlfFI34WzODjJloY4QSM7FmbnW+HCiFKYyvra3iUqjcl9AeR4=\n"
  "-----END CERTIFICATE-----\n",
  /* chain[1] (CA) */
  "-----BEGIN CERTIFICATE-----\n"
  "MIICkDCCAfmgAwIBAgIBATANBgkqhkiG9w0BAQQFADBaMQswCQYDVQQGEwJV\n"
  "UzEcMBoGA1UEChMTRXF1aWZheCBTZWN1cmUgSW5jLjEtMCsGA1UEAxMkRXF1\n"
  "aWZheCBTZWN1cmUgR2xvYmFsIGVCdXNpbmVzcyBDQS0xMB4XDTk5MDYyMTA0\n"
  "MDAwMFoXDTIwMDYyMTA0MDAwMFowWjELMAkGA1UEBhMCVVMxHDAaBgNVBAoT\n"
  "E0VxdWlmYXggU2VjdXJlIEluYy4xLTArBgNVBAMTJEVxdWlmYXggU2VjdXJl\n"
  "IEdsb2JhbCBlQnVzaW5lc3MgQ0EtMTCBnzANBgkqhkiG9w0BAQEFAAOBjQAw\n"
  "gYkCgYEAuucXkAJlsTRVPEnCUdXfp9E3j9HngXNBUmCbnaEXJnitx7HoJpQy\n"
  "td4zjTov2/KaelpzmKNc6fuKcxtc58O/gGzNqfTWK8D3+ZmqY6KxRwIP1ORR\n"
  "OhI8bIpaVIRw28HFkM9yRcuoWcDNM50/o5brhTMhHD4ePmBudpxnhcXIw2EC\n"
  "AwEAAaNmMGQwEQYJYIZIAYb4QgEBBAQDAgAHMA8GA1UdEwEB/wQFMAMBAf8w\n"
  "HwYDVR0jBBgwFoAUvqigdHJQa0S3ySPY+6j/s1draGwwHQYDVR0OBBYEFL6o\n"
  "oHRyUGtEt8kj2Puo/7NXa2hsMA0GCSqGSIb3DQEBBAUAA4GBADDiAVGqx+pf\n"
  "2rnQZQ8w1j7aDRRJbpGTJxQx78T3LUX47Me/okENI7SS+RkAZ70Br83gcfxa\n"
  "z2TE4JaY0KNA4gGK7ycH8WUBikQtBmV1UsCGECAhX2xrD2yuCRyv8qIYNMR1\n"
  "pHMc8Y3c7635s3a0kr/clRAevsvIO1qEYBlWlKlV\n"
  "-----END CERTIFICATE-----\n",
  NULL
};

/* Test V1 CA without basicConstraint. */
static const char *v1ca[] = {
  /* chain[0] */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIE/zCCA+egAwIBAgIQBSsgZODO6vk6ayagofBQJDANBgkqhkiG9w0BAQUFADCB\n"
  "sDELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQL\n"
  "ExZWZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTswOQYDVQQLEzJUZXJtcyBvZiB1c2Ug\n"
  "YXQgaHR0cHM6Ly93d3cudmVyaXNpZ24uY29tL3JwYSAoYykwNTEqMCgGA1UEAxMh\n"
  "VmVyaVNpZ24gQ2xhc3MgMyBTZWN1cmUgU2VydmVyIENBMB4XDTA4MDUwNTAwMDAw\n"
  "MFoXDTA5MDUyMjIzNTk1OVowczELMAkGA1UEBhMCVVMxETAPBgNVBAgTCElsbGlu\n"
  "b2lzMRAwDgYDVQQHFAdEdSBQYWdlMSQwIgYDVQQKFBtBcmdvbm5lIE5hdGlvbmFs\n"
  "IExhYm9yYXRvcnkxGTAXBgNVBAMUEGF1dGgyLml0LmFubC5nb3YwgZ8wDQYJKoZI\n"
  "hvcNAQEBBQADgY0AMIGJAoGBAMg6YPOXsPQedkLUug3RoMjv/OB+SfuDgGXxtef5\n"
  "iE0SjCcsKT5v+bfxt2+ccs7IN7kWn1luJ5NTb0ZrdE6LQoYp9oLsaX/ukOnxKUMY\n"
  "YhJJyHgutPtwyPvfZTZPpATWycJnZGIehY1S6thwxeofUyE3ykec2lalULzwXgel\n"
  "iC97AgMBAAGjggHTMIIBzzAJBgNVHRMEAjAAMAsGA1UdDwQEAwIFoDBEBgNVHR8E\n"
  "PTA7MDmgN6A1hjNodHRwOi8vU1ZSU2VjdXJlLWNybC52ZXJpc2lnbi5jb20vU1ZS\n"
  "U2VjdXJlMjAwNS5jcmwwRAYDVR0gBD0wOzA5BgtghkgBhvhFAQcXAzAqMCgGCCsG\n"
  "AQUFBwIBFhxodHRwczovL3d3dy52ZXJpc2lnbi5jb20vcnBhMB0GA1UdJQQWMBQG\n"
  "CCsGAQUFBwMBBggrBgEFBQcDAjAfBgNVHSMEGDAWgBRv7K+g3Yqk7/UqEGctP1WC\n"
  "vNfvJTB5BggrBgEFBQcBAQRtMGswJAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLnZl\n"
  "cmlzaWduLmNvbTBDBggrBgEFBQcwAoY3aHR0cDovL1NWUlNlY3VyZS1haWEudmVy\n"
  "aXNpZ24uY29tL1NWUlNlY3VyZTIwMDUtYWlhLmNlcjBuBggrBgEFBQcBDARiMGCh\n"
  "XqBcMFowWDBWFglpbWFnZS9naWYwITAfMAcGBSsOAwIaBBRLa7kolgYMu9BSOJsp\n"
  "rEsHiyEFGDAmFiRodHRwOi8vbG9nby52ZXJpc2lnbi5jb20vdnNsb2dvMS5naWYw\n"
  "DQYJKoZIhvcNAQEFBQADggEBAEAKzE4gXeyjRDBSgAgWIaCozbWK+b1ct4aZhWZt\n"
  "ihAyASxsNgDjDJzkInINjXoO5FWuJHDuoyyWHBQBb7t96+KgFu/4ye90VUDNTuin\n"
  "mmqdOKeLSHVnlhfvGLCdrhWSWg/jZmAjYrXYRwkvxehl9IcHmOgNrHV3INdrSTdZ\n"
  "ZCVLL74tuMqhMMm/NJ0tdEmWgpJe+/0dky2F2gAB+mFXlyzFvCLoyS2Vl0PW/BxM\n"
  "Ly5t+scmAbgni9gzmFTNhbKHd0s2UE395z4ra6fUdZ0BClFgMDvUnb6kJ/uyKRSa\n"
  "h7uQbWFJbA8aNgGLvfTf6o9n+GwbZkcgtBgIVENt8wzqg2I=\n"
  "-----END CERTIFICATE-----\n",
  /* chain[1] */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIEnDCCBAWgAwIBAgIQdTN9mrDhIzuuLX3kRpFi1DANBgkqhkiG9w0BAQUFADBf\n"
  "MQswCQYDVQQGEwJVUzEXMBUGA1UEChMOVmVyaVNpZ24sIEluYy4xNzA1BgNVBAsT\n"
  "LkNsYXNzIDMgUHVibGljIFByaW1hcnkgQ2VydGlmaWNhdGlvbiBBdXRob3JpdHkw\n"
  "HhcNMDUwMTE5MDAwMDAwWhcNMTUwMTE4MjM1OTU5WjCBsDELMAkGA1UEBhMCVVMx\n"
  "FzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQLExZWZXJpU2lnbiBUcnVz\n"
  "dCBOZXR3b3JrMTswOQYDVQQLEzJUZXJtcyBvZiB1c2UgYXQgaHR0cHM6Ly93d3cu\n"
  "dmVyaXNpZ24uY29tL3JwYSAoYykwNTEqMCgGA1UEAxMhVmVyaVNpZ24gQ2xhc3Mg\n"
  "MyBTZWN1cmUgU2VydmVyIENBMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKC\n"
  "AQEAlcMhEo5AxQ0BX3ZeZpTZcyxYGSK4yfx6OZAqd3J8HT732FXjr0LLhzAC3Fus\n"
  "cOa4RLQrNeuT0hcFfstG1lxToDJRnXRkWPkMmgDqXkRJZHL0zRDihQr5NO6ziGap\n"
  "paRa0A6Yf1gNK1K7hql+LvqySHyN2y1fAXWijQY7i7RhB8m+Ipn4G9G1V2YETTX0\n"
  "kXGWtZkIJZuXyDrzILHdnpgMSmO3ps6wAc74k2rzDG6fsemEe4GYQeaB3D0s57Rr\n"
  "4578CBbXs9W5ZhKZfG1xyE2+xw/j+zet1XWHIWuG0EQUWlR5OZZpVsm5Mc2JYVjh\n"
  "2XYFBa33uQKvp/1HkaIiNFox0QIDAQABo4IBgTCCAX0wEgYDVR0TAQH/BAgwBgEB\n"
  "/wIBADBEBgNVHSAEPTA7MDkGC2CGSAGG+EUBBxcDMCowKAYIKwYBBQUHAgEWHGh0\n"
  "dHBzOi8vd3d3LnZlcmlzaWduLmNvbS9ycGEwMQYDVR0fBCowKDAmoCSgIoYgaHR0\n"
  "cDovL2NybC52ZXJpc2lnbi5jb20vcGNhMy5jcmwwDgYDVR0PAQH/BAQDAgEGMBEG\n"
  "CWCGSAGG+EIBAQQEAwIBBjApBgNVHREEIjAgpB4wHDEaMBgGA1UEAxMRQ2xhc3Mz\n"
  "Q0EyMDQ4LTEtNDUwHQYDVR0OBBYEFG/sr6DdiqTv9SoQZy0/VYK81+8lMIGABgNV\n"
  "HSMEeTB3oWOkYTBfMQswCQYDVQQGEwJVUzEXMBUGA1UEChMOVmVyaVNpZ24sIElu\n"
  "Yy4xNzA1BgNVBAsTLkNsYXNzIDMgUHVibGljIFByaW1hcnkgQ2VydGlmaWNhdGlv\n"
  "biBBdXRob3JpdHmCEHC65B0Q2Sk0tjjKewPMur8wDQYJKoZIhvcNAQEFBQADgYEA\n"
  "w34IRl2RNs9n3Nenr6+4IsOLBHTTsWC85v63RBKBWzFzFGNWxnIu0RoDQ1w4ClBK\n"
  "Tc3athmo9JkNr+P32PF1KGX2av6b9L1S2T/L2hbLpZ4ujmZSeD0m+v6UNohKlV4q\n"
  "TBnvbvqCPy0D79YoszcYz0KyNCFkR9MgazpM3OYDkAw=\n"
  "-----END CERTIFICATE-----\n",
  /* chain[2] (CA) */
  "-----BEGIN CERTIFICATE-----\n"
  "MIICPDCCAaUCEHC65B0Q2Sk0tjjKewPMur8wDQYJKoZIhvcNAQECBQAwXzEL\n"
  "MAkGA1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMTcwNQYDVQQL\n"
  "Ey5DbGFzcyAzIFB1YmxpYyBQcmltYXJ5IENlcnRpZmljYXRpb24gQXV0aG9y\n"
  "aXR5MB4XDTk2MDEyOTAwMDAwMFoXDTI4MDgwMTIzNTk1OVowXzELMAkGA1UE\n"
  "BhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMTcwNQYDVQQLEy5DbGFz\n"
  "cyAzIFB1YmxpYyBQcmltYXJ5IENlcnRpZmljYXRpb24gQXV0aG9yaXR5MIGf\n"
  "MA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQDJXFme8huKARS0EN8EQNvjV69q\n"
  "RUCPhAwL0TPZ2RHP7gJYHyX3KqhEBarsAx94f56TuZoAqiN91qyFomNFx3In\n"
  "zPRMxnVx0jnvT0Lwdd8KkMaOIG+YD/isI19wKTakyYbnsZogy1Olhec9vn2a\n"
  "/iRFM9x2Fe0PonFkTGUugWhFpwIDAQABMA0GCSqGSIb3DQEBAgUAA4GBALtM\n"
  "EivPLCYATxQT3ab7/AoRhIzzKBxnki98tsX63/Dolbwdj2wsqFHMc9ikwFPw\n"
  "TtYmwHYBV4GSXiHx0bH/59AhWM1pF+NEHJwZRDmJXNycAA9WjQKZ7aKQRUzk\n"
  "uxCkPfAyAw7xzvjoyVGM5mKf5p/AfbdynMk2OmufTqj/ZA1k\n"
  "-----END CERTIFICATE-----\n",
  NULL
};

/* Test CACert chain with RSA-MD5 signature. */
static const char *cacertrsamd5[] = {
  /* chain[0] (EE cert) */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIE3zCCAsegAwIBAgICbmgwDQYJKoZIhvcNAQEFBQAwVDEUMBIGA1UEChMLQ0Fj\n"
  "ZXJ0IEluYy4xHjAcBgNVBAsTFWh0dHA6Ly93d3cuQ0FjZXJ0Lm9yZzEcMBoGA1UE\n"
  "AxMTQ0FjZXJ0IENsYXNzIDMgUm9vdDAeFw0wOTAxMTYyMjI5NDdaFw0xMTAxMTYy\n"
  "MjI5NDdaMBsxGTAXBgNVBAMTEGZyeS5zZXJ2ZXJhbWEuZGUwgZ8wDQYJKoZIhvcN\n"
  "AQEBBQADgY0AMIGJAoGBAMh3WSR8Dxw/zDAZpHMjA1To0HJIaoyR9TqzQfLgn7Yr\n"
  "sQFrRMefVMWYHiEFAVJTRcO5GuUtkw3IxALLlyNMl7xJbZESzRKw3Tz3NtM3DopB\n"
  "8L7rI8ANy7Hh6P5QRMWJ9OJyiLhSpAi0TuJeGr+kKovHRj64V2/NtoPgDsytHMt9\n"
  "AgMBAAGjggF2MIIBcjAMBgNVHRMBAf8EAjAAMDQGA1UdJQQtMCsGCCsGAQUFBwMC\n"
  "BggrBgEFBQcDAQYJYIZIAYb4QgQBBgorBgEEAYI3CgMDMAsGA1UdDwQEAwIFoDAz\n"
  "BggrBgEFBQcBAQQnMCUwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLmNhY2VydC5v\n"
  "cmcvMIHpBgNVHREEgeEwgd6CEGZyeS5zZXJ2ZXJhbWEuZGWgHgYIKwYBBQUHCAWg\n"
  "EgwQZnJ5LnNlcnZlcmFtYS5kZYIQZnJ5LnNlcnZlcmFtYS5kZaAeBggrBgEFBQcI\n"
  "BaASDBBmcnkuc2VydmVyYW1hLmRlghIqLmZyeS5zZXJ2ZXJhbWEuZGWgIAYIKwYB\n"
  "BQUHCAWgFAwSKi5mcnkuc2VydmVyYW1hLmRlggd6cHViLmRloBUGCCsGAQUFBwgF\n"
  "oAkMB3pwdWIuZGWCCSouenB1Yi5kZaAXBggrBgEFBQcIBaALDAkqLnpwdWIuZGUw\n"
  "DQYJKoZIhvcNAQEFBQADggIBAEWSsOlLbjdRjijMmOnDc2RcLQ5PQC9pjUW+bzGR\n"
  "KTJbf8Hf/wSdmHAam+UsIM6HzdQVi058dGyb8/NJQJD+9Dgv1m57x1prLerkt6xq\n"
  "UQCYmOpMxCJOykLqzEUnou9WtL5FaD+wBlOuqWFy0Cy2O3LHXkSkaMR+gdxC4pkI\n"
  "wSkI2SDdC0juvnoVI7iBaaIhYI/1FwV56hc6lxsAslf0NbtiiwhneVbHm5XRK1d4\n"
  "tabVKwOHnEuDyAnZd1yM1EqXKz+NwBlhoKWhC0fVUByID5A2WGEejBJcW/lVrYft\n"
  "4+sJpnwS+/VDS5yrDXMqMdYGE8TVMy7RsaoUdaeFQYv4Go48BBGDJB5uEkBJiSq8\n"
  "ViZA4iEKujBa5zKJ+CZXy3D/eHLBKUL+ayc9dLeeFTPZU0jYb83kE1wtlnWwF4J1\n"
  "8lUQI10nLFg+ALoZoAmFZej19XgbyG6im+ZRFuwrpV6F3HJRP+AMNInsLoQTuD9I\n"
  "l2gftVaIU1MqUmVMBcUeeNXG1BZ9vRonKzAC4Otfk1B6aW4Lz0E+sZ+HfCMicD3j\n"
  "N01KAeNZ64j8emgnLffurb7qUWbanTpMEzxrelBRufxJkXcn6BcFcxPBVgFnsMgF\n"
  "tP7e7N/cm55pI8Et+Gjp+ORJetSio118yu9bf7etSAJWOS6tQ2Ac7JeKP+a8jsvq\n"
  "Uyx7\n"
  "-----END CERTIFICATE-----\n",
  /* chain[1] (Class 3 CA) */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIGCDCCA/CgAwIBAgIBATANBgkqhkiG9w0BAQQFADB5MRAwDgYDVQQKEwdSb290\n"
  "IENBMR4wHAYDVQQLExVodHRwOi8vd3d3LmNhY2VydC5vcmcxIjAgBgNVBAMTGUNB\n"
  "IENlcnQgU2lnbmluZyBBdXRob3JpdHkxITAfBgkqhkiG9w0BCQEWEnN1cHBvcnRA\n"
  "Y2FjZXJ0Lm9yZzAeFw0wNTEwMTQwNzM2NTVaFw0zMzAzMjgwNzM2NTVaMFQxFDAS\n"
  "BgNVBAoTC0NBY2VydCBJbmMuMR4wHAYDVQQLExVodHRwOi8vd3d3LkNBY2VydC5v\n"
  "cmcxHDAaBgNVBAMTE0NBY2VydCBDbGFzcyAzIFJvb3QwggIiMA0GCSqGSIb3DQEB\n"
  "AQUAA4ICDwAwggIKAoICAQCrSTURSHzSJn5TlM9Dqd0o10Iqi/OHeBlYfA+e2ol9\n"
  "4fvrcpANdKGWZKufoCSZc9riVXbHF3v1BKxGuMO+f2SNEGwk82GcwPKQ+lHm9WkB\n"
  "Y8MPVuJKQs/iRIwlKKjFeQl9RrmK8+nzNCkIReQcn8uUBByBqBSzmGXEQ+xOgo0J\n"
  "0b2qW42S0OzekMV/CsLj6+YxWl50PpczWejDAz1gM7/30W9HxM3uYoNSbi4ImqTZ\n"
  "FRiRpoWSR7CuSOtttyHshRpocjWr//AQXcD0lKdq1TuSfkyQBX6TwSyLpI5idBVx\n"
  "bgtxA+qvFTia1NIFcm+M+SvrWnIl+TlG43IbPgTDZCciECqKT1inA62+tC4T7V2q\n"
  "SNfVfdQqe1z6RgRQ5MwOQluM7dvyz/yWk+DbETZUYjQ4jwxgmzuXVjit89Jbi6Bb\n"
  "6k6WuHzX1aCGcEDTkSm3ojyt9Yy7zxqSiuQ0e8DYbF/pCsLDpyCaWt8sXVJcukfV\n"
  "m+8kKHA4IC/VfynAskEDaJLM4JzMl0tF7zoQCqtwOpiVcK01seqFK6QcgCExqa5g\n"
  "eoAmSAC4AcCTY1UikTxW56/bOiXzjzFU6iaLgVn5odFTEcV7nQP2dBHgbbEsPyyG\n"
  "kZlxmqZ3izRg0RS0LKydr4wQ05/EavhvE/xzWfdmQnQeiuP43NJvmJzLR5iVQAX7\n"
  "6QIDAQABo4G/MIG8MA8GA1UdEwEB/wQFMAMBAf8wXQYIKwYBBQUHAQEEUTBPMCMG\n"
  "CCsGAQUFBzABhhdodHRwOi8vb2NzcC5DQWNlcnQub3JnLzAoBggrBgEFBQcwAoYc\n"
  "aHR0cDovL3d3dy5DQWNlcnQub3JnL2NhLmNydDBKBgNVHSAEQzBBMD8GCCsGAQQB\n"
  "gZBKMDMwMQYIKwYBBQUHAgEWJWh0dHA6Ly93d3cuQ0FjZXJ0Lm9yZy9pbmRleC5w\n"
  "aHA/aWQ9MTAwDQYJKoZIhvcNAQEEBQADggIBAH8IiKHaGlBJ2on7oQhy84r3HsQ6\n"
  "tHlbIDCxRd7CXdNlafHCXVRUPIVfuXtCkcKZ/RtRm6tGpaEQU55tiKxzbiwzpvD0\n"
  "nuB1wT6IRanhZkP+VlrRekF490DaSjrxC1uluxYG5sLnk7mFTZdPsR44Q4Dvmw2M\n"
  "77inYACHV30eRBzLI++bPJmdr7UpHEV5FpZNJ23xHGzDwlVks7wU4vOkHx4y/CcV\n"
  "Bc/dLq4+gmF78CEQGPZE6lM5+dzQmiDgxrvgu1pPxJnIB721vaLbLmINQjRBvP+L\n"
  "ivVRIqqIMADisNS8vmW61QNXeZvo3MhN+FDtkaVSKKKs+zZYPumUK5FQhxvWXtaM\n"
  "zPcPEAxSTtAWYeXlCmy/F8dyRlecmPVsYGN6b165Ti/Iubm7aoW8mA3t+T6XhDSU\n"
  "rgCvoeXnkm5OvfPi2RSLXNLrAWygF6UtEOucekq9ve7O/e0iQKtwOIj1CodqwqsF\n"
  "YMlIBdpTwd5Ed2qz8zw87YC8pjhKKSRf/lk7myV6VmMAZLldpGJ9VzZPrYPvH5JT\n"
  "oI53V93lYRE9IwCQTDz6o2CTBKOvNfYOao9PSmCnhQVsRqGP9Md246FZV/dxssRu\n"
  "FFxtbUFm3xuTsdQAw+7Lzzw9IYCpX2Nl/N3gX6T0K/CFcUHUZyX7GrGXrtaZghNB\n"
  "0m6lG5kngOcLqagA\n"
  "-----END CERTIFICATE-----\n",
  /* chain[2] (Root CA) */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIHPTCCBSWgAwIBAgIBADANBgkqhkiG9w0BAQQFADB5MRAwDgYDVQQKEwdSb290\n"
  "IENBMR4wHAYDVQQLExVodHRwOi8vd3d3LmNhY2VydC5vcmcxIjAgBgNVBAMTGUNB\n"
  "IENlcnQgU2lnbmluZyBBdXRob3JpdHkxITAfBgkqhkiG9w0BCQEWEnN1cHBvcnRA\n"
  "Y2FjZXJ0Lm9yZzAeFw0wMzAzMzAxMjI5NDlaFw0zMzAzMjkxMjI5NDlaMHkxEDAO\n"
  "BgNVBAoTB1Jvb3QgQ0ExHjAcBgNVBAsTFWh0dHA6Ly93d3cuY2FjZXJ0Lm9yZzEi\n"
  "MCAGA1UEAxMZQ0EgQ2VydCBTaWduaW5nIEF1dGhvcml0eTEhMB8GCSqGSIb3DQEJ\n"
  "ARYSc3VwcG9ydEBjYWNlcnQub3JnMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIIC\n"
  "CgKCAgEAziLA4kZ97DYoB1CW8qAzQIxL8TtmPzHlawI229Z89vGIj053NgVBlfkJ\n"
  "8BLPRoZzYLdufujAWGSuzbCtRRcMY/pnCujW0r8+55jE8Ez64AO7NV1sId6eINm6\n"
  "zWYyN3L69wj1x81YyY7nDl7qPv4coRQKFWyGhFtkZip6qUtTefWIonvuLwphK42y\n"
  "fk1WpRPs6tqSnqxEQR5YYGUFZvjARL3LlPdCfgv3ZWiYUQXw8wWRBB0bF4LsyFe7\n"
  "w2t6iPGwcswlWyCR7BYCEo8y6RcYSNDHBS4CMEK4JZwFaz+qOqfrU0j36NK2B5jc\n"
  "G8Y0f3/JHIJ6BVgrCFvzOKKrF11myZjXnhCLotLddJr3cQxyYN/Nb5gznZY0dj4k\n"
  "epKwDpUeb+agRThHqtdB7Uq3EvbXG4OKDy7YCbZZ16oE/9KTfWgu3YtLq1i6L43q\n"
  "laegw1SJpfvbi1EinbLDvhG+LJGGi5Z4rSDTii8aP8bQUWWHIbEZAWV/RRyH9XzQ\n"
  "QUxPKZgh/TMfdQwEUfoZd9vUFBzugcMd9Zi3aQaRIt0AUMyBMawSB3s42mhb5ivU\n"
  "fslfrejrckzzAeVLIL+aplfKkQABi6F1ITe1Yw1nPkZPcCBnzsXWWdsC4PDSy826\n"
  "YreQQejdIOQpvGQpQsgi3Hia/0PsmBsJUUtaWsJx8cTLc6nloQsCAwEAAaOCAc4w\n"
  "ggHKMB0GA1UdDgQWBBQWtTIb1Mfz4OaO873SsDrusjkY0TCBowYDVR0jBIGbMIGY\n"
  "gBQWtTIb1Mfz4OaO873SsDrusjkY0aF9pHsweTEQMA4GA1UEChMHUm9vdCBDQTEe\n"
  "MBwGA1UECxMVaHR0cDovL3d3dy5jYWNlcnQub3JnMSIwIAYDVQQDExlDQSBDZXJ0\n"
  "IFNpZ25pbmcgQXV0aG9yaXR5MSEwHwYJKoZIhvcNAQkBFhJzdXBwb3J0QGNhY2Vy\n"
  "dC5vcmeCAQAwDwYDVR0TAQH/BAUwAwEB/zAyBgNVHR8EKzApMCegJaAjhiFodHRw\n"
  "czovL3d3dy5jYWNlcnQub3JnL3Jldm9rZS5jcmwwMAYJYIZIAYb4QgEEBCMWIWh0\n"
  "dHBzOi8vd3d3LmNhY2VydC5vcmcvcmV2b2tlLmNybDA0BglghkgBhvhCAQgEJxYl\n"
  "aHR0cDovL3d3dy5jYWNlcnQub3JnL2luZGV4LnBocD9pZD0xMDBWBglghkgBhvhC\n"
  "AQ0ESRZHVG8gZ2V0IHlvdXIgb3duIGNlcnRpZmljYXRlIGZvciBGUkVFIGhlYWQg\n"
  "b3ZlciB0byBodHRwOi8vd3d3LmNhY2VydC5vcmcwDQYJKoZIhvcNAQEEBQADggIB\n"
  "ACjH7pyCArpcgBLKNQodgW+JapnM8mgPf6fhjViVPr3yBsOQWqy1YPaZQwGjiHCc\n"
  "nWKdpIevZ1gNMDY75q1I08t0AoZxPuIrA2jxNGJARjtT6ij0rPtmlVOKTV39O9lg\n"
  "18p5aTuxZZKmxoGCXJzN600BiqXfEVWqFcofN8CCmHBh22p8lqOOLlQ+TyGpkO/c\n"
  "gr/c6EWtTZBzCDyUZbAEmXZ/4rzCahWqlwQ3JNgelE5tDlG+1sSPypZt90Pf6DBl\n"
  "Jzt7u0NDY8RD97LsaMzhGY4i+5jhe1o+ATc7iwiwovOVThrLm82asduycPAtStvY\n"
  "sONvRUgzEv/+PDIqVPfE94rwiCPCR/5kenHA0R6mY7AHfqQv0wGP3J8rtsYIqQ+T\n"
  "SCX8Ev2fQtzzxD72V7DX3WnRBnc0CkvSyqD/HMaMyRa+xMwyN2hzXwj7UfdJUzYF\n"
  "CpUCTPJ5GhD22Dp1nPMd8aINcGeGG7MW9S/lpOt5hvk9C8JzC6WZrG/8Z7jlLwum\n"
  "GCSNe9FINSkYQKyTYOGWhlC0elnYjyELn8+CkcY7v2vcB5G5l1YjqrZslMZIBjzk\n"
  "zk6q5PYvCdxTby78dOs6Y5nCpqyJvKeyRKANihDjbPIky/qbn3BHLt4Ui9SyIAmW\n"
  "omTxJBzcoTWcFbLUvFUufQb1nA5V9FrWk9p2rSVzTMVD\n"
  "-----END CERTIFICATE-----\n",
  NULL
};

/* Test Certicom cert with ECC-SHA256 signature. */
static const char *ecc_cert[] = {
  /* chain[0] (ECC cert) */
"-----BEGIN CERTIFICATE-----\n"
"MIICbzCCAhSgAwIBAgIIZLkW6EZO5PQwCgYIKoZIzj0EAwIwgZsxFDASBgNVBAsT\n"
"C1NBTVBMRSBPTkxZMRcwFQYDVQQKEw5DZXJ0aWNvbSBDb3JwLjEQMA4GA1UEBxMH\n"
"VG9yb250bzEQMA4GA1UEBBMHT250YXJpbzE5MDcGA1UEAxMwdGxzLnNlY2cub3Jn\n"
"IEVDQyBzZWNwMjU2cjEgQ2VydGlmaWNhdGUgQXV0aG9yaXR5MQswCQYDVQQGEwJD\n"
"QTAeFw0wOTA1MDcwMDAwMDBaFw0xNTA1MDEwMDAwMDBaMIGYMRQwEgYDVQQLEwtT\n"
"QU1QTEUgT05MWTEXMBUGA1UEChMOQ2VydGljb20gQ29ycC4xEDAOBgNVBAcTB1Rv\n"
"cm9udG8xEDAOBgNVBAgTB09udGFyaW8xNjA0BgNVBAMTLXRscy5zZWNnLm9yZyBF\n"
"Q0Mgc2VjcDI1NnIxIFNlcnZlciBDZXJ0aWZpY2F0ZTELMAkGA1UEBhMCQ0EwWTAT\n"
"BgcqhkjOPQIBBggqhkjOPQMBBwNCAATf63kPhr3D6a2scrHWVr0oOXQMnBDT6Jv/\n"
"ifqzt4/xTbXsZNEyD96nyh82sk0tM+FVfBlsIwGc7vqBfyq0mC/Io0MwQTAOBgNV\n"
"HQ8BAf8EBAMCA4gwFgYDVR0lAQH/BAwwCgYIKwYBBQUHAwEwFwYDVR0RBBAwDoIM\n"
"dGxzLnNlY2cub3JnMAoGCCqGSM49BAMCA0kAMEYCIQDfacZHsdsj6SXQ2hyJS4Do\n"
"SMclqGLo2Sop7hfAeEJA+wIhAOMo7eLya44SIcuzrLBpg29g5ZzYOeuEzRcg9mch\n"
"AB1w\n"
"-----END CERTIFICATE-----\n",
"-----BEGIN CERTIFICATE-----\n"
"MIICTjCCAfagAwIBAgIICvq6Bj3Av6EwCQYHKoZIzj0EATCBmzEUMBIGA1UECxML\n"
"U0FNUExFIE9OTFkxFzAVBgNVBAoTDkNlcnRpY29tIENvcnAuMRAwDgYDVQQHEwdU\n"
"b3JvbnRvMRAwDgYDVQQEEwdPbnRhcmlvMTkwNwYDVQQDEzB0bHMuc2VjZy5vcmcg\n"
"RUNDIHNlY3AyNTZyMSBDZXJ0aWZpY2F0ZSBBdXRob3JpdHkxCzAJBgNVBAYTAkNB\n"
"MB4XDTA2MDUwMTAwMDAwMFoXDTE1MDUwMTAwMDAwMFowgZsxFDASBgNVBAsTC1NB\n"
"TVBMRSBPTkxZMRcwFQYDVQQKEw5DZXJ0aWNvbSBDb3JwLjEQMA4GA1UEBxMHVG9y\n"
"b250bzEQMA4GA1UEBBMHT250YXJpbzE5MDcGA1UEAxMwdGxzLnNlY2cub3JnIEVD\n"
"QyBzZWNwMjU2cjEgQ2VydGlmaWNhdGUgQXV0aG9yaXR5MQswCQYDVQQGEwJDQTBZ\n"
"MBMGByqGSM49AgEGCCqGSM49AwEHA0IABB2oofFVa6akTK6hpaJLs+6skdhn0sQp\n"
"uJwVwG99T0VZY8v7q6NMIWrpYQFmOxQyVVNlxWWyr2cLYJTyqx/zuDejIzAhMA4G\n"
"A1UdDwEB/wQEAwIBhjAPBgNVHRMBAf8EBTADAQH/MAkGByqGSM49BAEDRwAwRAIg\n"
"W2KAhfAKWFoh47A7muk8K10cGqOKRtb9lCgdOltj19oCIG+ZJQv5m+RnL4X2Ti0y\n"
"ZJzOOuzBQVGiUFwZdn1dLv4X\n"
"-----END CERTIFICATE-----\n",
  NULL
};

static struct
{
  const char *name;
  const char **chain;
  const char **ca;
  unsigned int verify_flags;
  unsigned int expected_verify_result;
} chains[] =
{
  { "CVE-2014-0092", cve_2014_0092_check, &cve_2014_0092_check[1],
    0, GNUTLS_CERT_SIGNER_NOT_CA | GNUTLS_CERT_INVALID },
  { "CVE-2008-4989", cve_2008_4989_chain, &cve_2008_4989_chain[2],
    0, GNUTLS_CERT_SIGNER_NOT_FOUND | GNUTLS_CERT_INVALID },
  { "verisign.com v1 fail", verisign_com_chain, &verisign_com_chain[3],
    0,
    GNUTLS_CERT_EXPIRED | GNUTLS_CERT_INVALID },
  { "verisign.com v1 ok", verisign_com_chain, &verisign_com_chain[3],
    GNUTLS_VERIFY_DISABLE_TIME_CHECKS | GNUTLS_PROFILE_TO_VFLAGS(GNUTLS_PROFILE_LOW),
    0 },
  { "verisign.com v1 not ok due to profile", verisign_com_chain, &verisign_com_chain[3],
    GNUTLS_VERIFY_DISABLE_TIME_CHECKS | GNUTLS_PROFILE_TO_VFLAGS(GNUTLS_PROFILE_LEGACY),
    GNUTLS_CERT_INSECURE_ALGORITHM | GNUTLS_CERT_INVALID },
  { "verisign.com v1 not ok due to profile", verisign_com_chain, &verisign_com_chain[3],
    GNUTLS_VERIFY_DISABLE_TIME_CHECKS | GNUTLS_PROFILE_TO_VFLAGS(GNUTLS_PROFILE_HIGH),
    GNUTLS_CERT_INSECURE_ALGORITHM | GNUTLS_CERT_INVALID },
  { "citibank.com v1 fail", citibank_com_chain, &citibank_com_chain[2],
    GNUTLS_VERIFY_DO_NOT_ALLOW_X509_V1_CA_CRT, GNUTLS_CERT_SIGNER_NOT_CA | GNUTLS_CERT_INVALID },
  { "expired self signed", pem_self_cert, &pem_self_cert[0],
    0, GNUTLS_CERT_EXPIRED | GNUTLS_CERT_INVALID },
  { "self signed", pem_self_cert, &pem_self_cert[0],
    GNUTLS_VERIFY_DISABLE_TIME_CHECKS, 0 },
  { "ca=false", thea_chain, &thea_chain[1],
    0,
    GNUTLS_CERT_SIGNER_NOT_CA | GNUTLS_CERT_INVALID },
  { "ca=false2", thea_chain, &thea_chain[1],
    0, GNUTLS_CERT_SIGNER_NOT_CA | GNUTLS_CERT_INVALID },
  { "hbci v1 fail", hbci_chain, &hbci_chain[2],
    GNUTLS_VERIFY_DO_NOT_ALLOW_X509_V1_CA_CRT, GNUTLS_CERT_SIGNER_NOT_CA | GNUTLS_CERT_INVALID},
  { "hbci v1 ok expired", hbci_chain, &hbci_chain[2],
    0,
    GNUTLS_CERT_EXPIRED | GNUTLS_CERT_INVALID },
  { "hbci v1 ok", hbci_chain, &hbci_chain[2],
    GNUTLS_VERIFY_DISABLE_TIME_CHECKS,
    0 },
  { "rsa-md5 fail", mayfirst_chain, &mayfirst_chain[1],
    0, GNUTLS_CERT_INSECURE_ALGORITHM | GNUTLS_CERT_INVALID },
  { "rsa-md5 not ok", mayfirst_chain, &mayfirst_chain[1],
    GNUTLS_VERIFY_ALLOW_SIGN_RSA_MD2,
    GNUTLS_CERT_INSECURE_ALGORITHM | GNUTLS_CERT_INVALID },
  { "rsa-md5 not ok2", mayfirst_chain, &mayfirst_chain[1],
    GNUTLS_VERIFY_ALLOW_SIGN_RSA_MD5,
    GNUTLS_CERT_EXPIRED | GNUTLS_CERT_INVALID },
  { "rsa-md5 ok", mayfirst_chain, &mayfirst_chain[1],
    GNUTLS_VERIFY_DISABLE_TIME_CHECKS | GNUTLS_VERIFY_ALLOW_SIGN_RSA_MD5, 0 },
  { "v1ca fail", v1ca, &v1ca[2],
    GNUTLS_VERIFY_DO_NOT_ALLOW_X509_V1_CA_CRT, GNUTLS_CERT_SIGNER_NOT_CA | GNUTLS_CERT_INVALID },

  { "pathlen fail", pathlen_check, &pathlen_check[2],
    GNUTLS_VERIFY_ALLOW_ANY_X509_V1_CA_CRT | GNUTLS_VERIFY_DISABLE_TIME_CHECKS, GNUTLS_CERT_INVALID | GNUTLS_CERT_SIGNER_CONSTRAINTS_FAILURE},

  /* Test whether a V1 root certificate is correctly accepted */
  { "v1root fail", v1_root_check, &v1_root_check[1],
    GNUTLS_VERIFY_DO_NOT_ALLOW_X509_V1_CA_CRT | GNUTLS_VERIFY_DISABLE_TIME_CHECKS, GNUTLS_CERT_SIGNER_NOT_CA | GNUTLS_CERT_INVALID },
  { "v1root ok", v1_root_check, &v1_root_check[1],
    GNUTLS_VERIFY_DISABLE_TIME_CHECKS, 0 },

  /* test whether a v1 intermediate certificate is rejected */
  { "v1invalid fail", v1_intermed_check, &v1_intermed_check[2],
    GNUTLS_VERIFY_DISABLE_TIME_CHECKS, GNUTLS_CERT_SIGNER_NOT_CA | GNUTLS_CERT_INVALID },
  { "v1 leaf ok", &v1_intermed_check[1], &v1_intermed_check[2],
    GNUTLS_VERIFY_DISABLE_TIME_CHECKS, 0 },

  { "v1ca expired", v1ca, &v1ca[2],
    0,
    GNUTLS_CERT_EXPIRED | GNUTLS_CERT_INVALID  },
  { "v1ca ok", v1ca, &v1ca[2],
    GNUTLS_VERIFY_DISABLE_TIME_CHECKS,
    0 },
  { "v1ca2 expired", v1ca, &v1ca[2],
    GNUTLS_VERIFY_ALLOW_ANY_X509_V1_CA_CRT,
    GNUTLS_CERT_EXPIRED | GNUTLS_CERT_INVALID },
  { "v1ca2 ok", v1ca, &v1ca[2],
    GNUTLS_VERIFY_DISABLE_TIME_CHECKS | GNUTLS_VERIFY_ALLOW_ANY_X509_V1_CA_CRT,
    0 },
  { "cacertrsamd5 fail", cacertrsamd5, &cacertrsamd5[2],
    0, GNUTLS_CERT_INSECURE_ALGORITHM | GNUTLS_CERT_INVALID },
  { "cacertrsamd5 ok", cacertrsamd5, &cacertrsamd5[2],
    GNUTLS_VERIFY_ALLOW_SIGN_RSA_MD5, 0 },
  { "cacertrsamd5 short-cut not ok", cacertrsamd5, &cacertrsamd5[0],
    GNUTLS_VERIFY_DO_NOT_ALLOW_SAME,
    GNUTLS_CERT_SIGNER_NOT_FOUND | GNUTLS_CERT_INVALID },
  { "cacertrsamd5 short-cut ok", cacertrsamd5, &cacertrsamd5[1],
    0, 0 },
  { "ecc cert ok", ecc_cert, &ecc_cert[1], GNUTLS_PROFILE_TO_VFLAGS(GNUTLS_PROFILE_HIGH), 0 },
  { "ecc cert ok", ecc_cert, &ecc_cert[1], GNUTLS_PROFILE_TO_VFLAGS(GNUTLS_PROFILE_SUITEB128), 0 },
  { "ecc cert not ok (due to profile)", ecc_cert, &ecc_cert[1], GNUTLS_PROFILE_TO_VFLAGS(GNUTLS_PROFILE_ULTRA), 
  	GNUTLS_CERT_INSECURE_ALGORITHM | GNUTLS_CERT_INVALID },
  { "ecc cert not ok (due to profile)", ecc_cert, &ecc_cert[1], GNUTLS_PROFILE_TO_VFLAGS(GNUTLS_PROFILE_SUITEB192), 
  	GNUTLS_CERT_INSECURE_ALGORITHM | GNUTLS_CERT_INVALID },
  { "name constraints chain ok", nc_good, &nc_good[4], GNUTLS_VERIFY_DISABLE_TIME_CHECKS, 0 },
  { "name constraints chain bad1", nc_bad1, &nc_bad1[2], GNUTLS_VERIFY_DISABLE_TIME_CHECKS, GNUTLS_CERT_INVALID | GNUTLS_CERT_SIGNER_CONSTRAINTS_FAILURE},
  { "name constraints chain bad2", nc_bad2, &nc_bad2[4], GNUTLS_VERIFY_DISABLE_TIME_CHECKS, GNUTLS_CERT_INVALID | GNUTLS_CERT_SIGNER_CONSTRAINTS_FAILURE},
  { NULL, NULL, NULL, 0, 0}
};
/* *INDENT-ON* */
