/*
 * gaia - opensource 3D interface to the planet
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

/*
Simple program to fetch single image from Google Earth server.
*/

#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <gefetch.h>
#include <zlib.h>

int main(int argc, char **argv) {
	int f;
	gefetch_t gef;

	if ((gef = gefetch_init()) == 0) {
		fprintf(stderr, "Unable to init libgefetch\n");
		return 1;
	}

	if (gefetch_fetch_uri(gef, "/flatfile?q2-0") != GEFETCH_OK) {
		fprintf(stderr, "Unable to fetch file\n");
		gefetch_cleanup(gef);
		return 2;
	}

	unsigned char *data = gefetch_get_data_ptr(gef);
	size_t size = gefetch_get_data_size(gef);

	if (size < 8) {
		fprintf(stderr, "File too small\n");
		gefetch_cleanup(gef);
		return 2;
	}

	void *ddata;
	uLongf dsize = ((uLongf)data[4]) |
		((uLongf)data[5] << 8) |
		((uLongf)data[6] << 16) |
		((uLongf)data[7] << 24);

	if ((ddata = malloc(dsize)) == 0) {
		fprintf(stderr, "Uname to allocate buffer for decompressed data\n");
		return 2;
	}

	uLongf realdsize = dsize;
	if (uncompress(ddata, &realdsize, data+8, size-8) != Z_OK) {
		fprintf(stderr, "Error decompressing data\n");
		free(ddata);
		return 2;
	}

	if (realdsize != dsize)
		fprintf(stderr, "Warning: real and expected decompressed data size do not match\n");

	if ((f = open("meta.root", O_CREAT | O_TRUNC | O_WRONLY, 0644)) < 0) {
		fprintf(stderr, "Unable to open meta.root for writing\n");
		free(ddata);
		gefetch_cleanup(gef);
		return 3;
	}

	if (write(f, ddata, dsize) != dsize) {
		fprintf(stderr, "Unable write %d bytes\n", (int)dsize);
		free(ddata);
		gefetch_cleanup(gef);
		close(f);
		return 4;
	}

	fprintf(stderr, "meta.root fetched successfully\n");

	free(ddata);
	close(f);
	gefetch_cleanup(gef);

	return 0;
}
