unit Wizard;

{
  Inno Setup
  Copyright (C) 1998-99 Jordan Russell
  For conditions of distribution and use, see LICENSE.TXT.

  Wizard form
}

interface

uses
  WinTypes, WinProcs, SysUtils, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, NewBevel, NewFCtrl, ExtCtrls,
  ChildFrm, RichEditViewer;

type
  TWizardPage = (wpWelcome, wpPassword, wpLicense, wpInfoBefore, wpSelectDir,
    wpSelectProgramGroup, wpReady, wpInfoAfter, wpFinished);

  TWizardForm = class(TSetupChildForm)
    CancelButton: TButton;
    NextButton: TButton;
    BackButton: TButton;
    Notebook1: TNotebook;
    PicturePaintBox: TPaintBox;
    Bevel: TBevel;
    ClickLabel: TLabel;
    Notebook2: TNotebook;
    WelcomeLabel: TLabel;
    SelectFolderLabel: TLabel;
    DiskSpaceLabel: TLabel;
    DirEdit: TEdit;
    DirList: TNewDirectoryListBox;
    DriveList: TNewDriveComboBox;
    IconsLabel: TLabel;
    GroupEdit: TEdit;
    NoIconsCheck: TCheckBox;
    GroupList: TListBox;
    LicenseLabel1: TLabel;
    LicenseMemo: TRichEditViewer;
    LicenseLabel2: TLabel;
    LicenseIcon: TPaintBox;
    ReadmeBeforeIcon: TPaintBox;
    InfoBeforeLabel: TLabel;
    InfoBeforeMemo: TRichEditViewer;
    ReadmeAfterIcon: TPaintBox;
    InfoAfterLabel: TLabel;
    InfoAfterMemo: TRichEditViewer;
    FinishedLabel: TLabel;
    ShowReadmeCheck: TCheckBox;
    InfoBeforeClickLabel: TLabel;
    InfoAfterClickLabel: TLabel;
    YesRadio: TRadioButton;
    NoRadio: TRadioButton;
    ReadyLabel: TLabel;
    PasswordLabel: TLabel;
    PasswordEdit: TEdit;
    PasswordEditLabel: TLabel;
    procedure NextButtonClick(Sender: TObject);
    procedure BackButtonClick(Sender: TObject);
    procedure CancelButtonClick(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure DirListChange(Sender: TObject);
    procedure GroupListClick(Sender: TObject);
    procedure NoIconsCheckClick(Sender: TObject);
    procedure PicturePaintBoxPaint(Sender: TObject);
    procedure GroupEditChange(Sender: TObject);
    procedure SetupIconPaint(Sender: TObject);
  private
    { Private declarations }
    DisableCloseQuery: Boolean;
    procedure ReadGroupList;
    procedure WMSysCommand (var Message: TWMSysCommand); message WM_SYSCOMMAND;
  public
    { Public declarations }
    CurPage: TWizardPage;
    OrigFinishedLabelWidth: Integer;
    PreviousAppDir: String;
    constructor Create (AOwner: TComponent); override;
    procedure ChangeFinishedLabel (const S: String);
    procedure CurPageChanged;
    function SkipCurPage: Boolean;
  end;

var
  WizardForm: TWizardForm;

function FixLabel (const S: String): String;

implementation

uses
  ShlObj,
  Msgs, MsgIDs, Main, Install, Struct,
  CmnFunc, CmnFunc2,
  DDEInt, zlib;

{$R *.DFM}

const
  NotebookPages: array[TWizardPage, 0..1] of Integer =
    ((0, 0), (0, 1), (1, -1), (2, -1), (0, 2), (0, 3), (0, 4), (3, -1), (0, 5));
  PageCaptions: array[TWizardPage] of TSetupMessageID =
    (msgWizardWelcome, MsgWizardPassword, msgWizardLicense, msgWizardInfoBefore,
     msgWizardSelectDir, msgWizardSelectProgramGroup, msgWizardReady,
     msgWizardInfoAfter, msgWizardFinished);

var
  MinimumSpaceStr: String;

function IntToMBStr (const I: Longint): String;
var
  MB, MBFrac, MBDec: Longint;
begin
  MB := I div 1048576;
  MBFrac := (I mod 1048576) * 10;
  MBDec := MBFrac div 1048576;
  if MBFrac mod 1048576 <> 0 then  { round up }
    Inc (MBDec);
  if MBDec = 10 then begin
    Inc (MB);
    MBDec := 0;
  end;
  Result := Format('%d%s%d', [MB, DecimalSeparator, MBDec]);
end;

function FixLabel (const S: String): String;
begin
  Result := S;
  {don't localize these}
  StringChange (Result, '[name]', SetupHeader.AppName);
  StringChange (Result, '[name/ver]', SetupHeader.AppVerName);
  if MinimumSpaceStr = '' then
    MinimumSpaceStr := IntToMBStr(MinimumSpace);
  StringChange (Result, '[mb]', MinimumSpaceStr);
end;


{ TWizardForm }

constructor TWizardForm.Create (AOwner: TComponent);
{ Do all initialization of the wizard form. We're overriding Create instead of
  using the FormCreate event, because if an exception is raised in FormCreate
  it's not propogated out. }

  function TrySetDirListDirectory (const S: String): Boolean;
  begin
    Result := True;
    try
      DirList.Directory := S;
    except
      Result := False;
    end;
  end;

  procedure FindPreviousData (var AAppDir, AGroup: String);
  const
    RootKeys: array[0..1] of HKEY = (HKEY_CURRENT_USER, HKEY_LOCAL_MACHINE);
  var
    I: Integer;
    H: HKEY;
  begin
    AAppDir := '';
    AGroup := '';
    { First look in HKEY_CURRENT_USER. That's where Inno Setup creates
      the uninstall key for non-administrators. If the key doesn't exist
      under HKEY_CURRENT_USER, check HKEY_LOCAL_MACHINE. }
    for I := 0 to 1 do
      if RegOpenKeyEx(RootKeys[I],
         PChar(Format('%s\%s_is1', [NEWREGSTR_PATH_UNINSTALL, UninstallRegKeyBaseName])),
         0, KEY_QUERY_VALUE, H) = ERROR_SUCCESS then begin
        try
          { do not localize or change the following strings }
          RegQueryStringValue (H, 'Inno Setup: App Path', AAppDir);
          RegQueryStringValue (H, 'Inno Setup: Icon Group', AGroup);
        finally
          RegCloseKey (H);
        end;
        Break;
      end;
  end;

var
  OldTextHeight, NewTextHeight, X, W1, W2, W3: Integer;
  S: String;
  SystemMenu: HMENU;
  PrevAppDir, PrevGroup, P, P2, P3: String;
const
  idDirectories: array[Boolean] of TSetupMessageID =
    (msgDirectoryOld, msgDirectoryNew);
  idProgramManagers: array[Boolean] of TSetupMessageID =
    (msgProgramManagerOld, msgProgramManagerNew);
  WizardButtonWidth = 75;
begin
  inherited;

  SetFormFont (Self, OldTextHeight, NewTextHeight);

  { Give it a minimize button if main window isn't visible }
  if not(shWindowVisible in SetupHeader.Options) then begin
    BorderIcons := BorderIcons + [biMinimize];
    BorderStyle := bsSingle;
  end;

  { Position the buttons, and scale their size }
  W1 := MulDiv(WizardButtonWidth, NewTextHeight, OldTextHeight);  { width of each button }
  W2 := MulDiv(8, NewTextHeight, OldTextHeight);   { space between edge of form and Cancel button }
  W3 := MulDiv(10, NewTextHeight, OldTextHeight);  { space between Next and Cancel button }
  BackButton.Width := W1;
  NextButton.Width := W1;
  CancelButton.Width := W1;
  X := ClientWidth - W2 - W1;
  CancelButton.Left := X;
  Dec (X, W3);
  Dec (X, W1);
  NextButton.Left := X;
  Dec (X, W1);
  BackButton.Left := X;

  { Assign captions }
  WelcomeLabel.Caption := FixLabel(SetupMessages[msgWelcomeLabel]) +
    SNewLine2 + SetupMessages[msgClickNext];
  ClickLabel.Caption := SetupMessages[msgClickNext];
  PasswordLabel.Caption := SetupMessages[msgPasswordLabel];
  PasswordEditLabel.Caption := SetupMessages[msgPasswordEditLabel];
  LicenseLabel1.Caption := FixLabel(SetupMessages[msgLicenseLabel1]);
  LicenseLabel2.Caption := FixLabel(SetupMessages[msgLicenseLabel2]);
  InfoBeforeLabel.Caption := SetupMessages[msgInfoBeforeLabel];
  InfoBeforeClickLabel.Caption := SetupMessages[msgInfoBeforeClickLabel];
  DiskSpaceLabel.Caption := FixLabel(SetupMessages[msgDiskSpaceMBLabel]);
  NoIconsCheck.Caption := SetupMessages[msgNoIconsCheck];
  InfoAfterLabel.Caption := SetupMessages[msgInfoAfterLabel];
  InfoAfterClickLabel.Caption := SetupMessages[msgInfoAfterClickLabel];
  S := FixLabel(SetupMessages[msgReadyLabel1]) + SNewLine2;
  if not(shDisableDirPage in SetupHeader.Options) or
     (HasIcons and not(shDisableProgramGroupPage in SetupHeader.Options)) then
    S := S + SetupMessages[msgReadyLabel2a]
  else
    S := S + SetupMessages[msgReadyLabel2b];
  ReadyLabel.Caption := S;
  if HasIcons then
    S := FixLabel(SetupMessages[msgFinishedLabel])
  else
    S := FixLabel(SetupMessages[msgFinishedLabelNoIcons]);
  OrigFinishedLabelWidth := FinishedLabel.Width;
  ChangeFinishedLabel (S + SNewLine2 + SetupMessages[msgClickFinish]);
  ShowReadmeCheck.Caption := SetupMessages[msgShowReadmeCheck];
  YesRadio.Caption := SetupMessages[msgYesRadio];
  NoRadio.Caption := SetupMessages[msgNoRadio];
  SelectFolderLabel.Caption := FixLabel(FmtSetupMessage1(msgSelectFolderLabel,
    SetupMessages[idDirectories[NewGUI]]));
  IconsLabel.Caption := FmtSetupMessage1(msgIconsLabel,
    SetupMessages[idProgramManagers[NewGUI]]);

  { Don't set UseRichEdit to True on the TRichEditViewers unless they are going
    to be used. There's no need to load riched*.dll unnecessarily. }
  if SetupHeader.LicenseText <> '' then begin
    LicenseMemo.UseRichEdit := True;
    LicenseMemo.RTFText := SetupHeader.LicenseText;
  end;
  if SetupHeader.InfoBeforeText <> '' then begin
    InfoBeforeMemo.UseRichEdit := True;
    InfoBeforeMemo.RTFText := SetupHeader.InfoBeforeText;
  end;
  { Note: InfoAfterMemo is set up in the Main unit }

  { Center the form inside the MainForm's client area }
  MainForm.CenterForm (Self);

  { Append an 'About Setup' item to the wizard form also }
  SystemMenu := GetSystemMenu(Handle, False);
  AppendMenu (SystemMenu, MF_SEPARATOR, 0, nil);
  AppendMenu (SystemMenu, MF_STRING, 9999, PChar(SetupMessages[msgAboutSetupMenuItem]));

  { Read settings from a previous install if available }
  FindPreviousData (PrevAppDir, PrevGroup);

  { Assign default directory name }
  if shCreateAppDir in SetupHeader.Options then begin
    { Try different values and see what works. The 'except end;' throws away any
      exceptions that might occur. }
    TrySetDirListDirectory ('\');
    TrySetDirListDirectory (SystemDrive + '\');
    P := '';
    if shUsePreviousAppDir in SetupHeader.Options then begin
      P := PrevAppDir;
      PreviousAppDir := PrevAppDir;
    end;
    if P = '' then
      P := ChangeDirConst(SetupHeader.DefaultDirName);
    if not(shDisableAppendDir in SetupHeader.Options) or
       not TrySetDirListDirectory(P) then begin
      P2 := P;
      while True do begin
        P3 := RemoveBackslashUnlessRoot(ExtractFilePath(P2));
        if Length(P3) < 3 then Break;
        if TrySetDirListDirectory(P3) or (P3 = P2) then
          Break;
        P2 := P3;
      end;
    end;
    DirEdit.Text := P;

    if (InitDir <> '') and not(shDisableDirPage in SetupHeader.Options) then begin
      { ^ InitDir currently isn't supported for installations with
        shDisableDirPage set. If the wizard page isn't displayed, it doesn't
        get a chance to validate the path specified. }
      try
        DirList.Directory := ExtractFilePath(InitDir);
        DirList.Directory := InitDir;
      except
      end;
      DirEdit.Text := InitDir;
    end;
  end
  else
    DirEdit.Text := WinDir;

  { Assign default group name }
  if (InitProgramGroup <> '') and not(shDisableProgramGroupPage in SetupHeader.Options) then
    { ^ InitProgramGroup currently isn't supported for installations with
      shDisableProgramGroupPage set. If the wizard page isn't displayed, it
      doesn't get a chance to validate the program group name specified. }
    GroupEdit.Text := InitProgramGroup
  else begin
    if (PrevGroup = '') or not(shUsePreviousGroup in SetupHeader.Options) then
      GroupEdit.Text := SetupHeader.DefaultGroupName
    else
      GroupEdit.Text := PrevGroup;
  end;

  NoIconsCheck.Visible := shAllowNoIcons in SetupHeader.Options;

  CurPage := Low(TWizardPage);
  CurPageChanged;

  ReadGroupList;
end;

procedure TWizardForm.ChangeFinishedLabel (const S: String);
var
  Y: Integer;
begin
  FinishedLabel.AutoSize := False;
  FinishedLabel.Width := OrigFinishedLabelWidth;
  FinishedLabel.Caption := S + SNewLine;
  FinishedLabel.AutoSize := True;
  Y := FinishedLabel.Top + FinishedLabel.Height;
  ShowReadmeCheck.Top := Y;
  YesRadio.Top := Y;
  NoRadio.Top := Y + MulDiv(24, Screen.PixelsPerInch, 96);
end;

procedure TWizardForm.CurPageChanged;
{ Call this whenever the current page is changed }
var
  V: Boolean;
label 1;
begin
  Notebook1.PageIndex := NotebookPages[CurPage, 0];
  if NotebookPages[CurPage, 1] <> -1 then
    Notebook2.PageIndex := NotebookPages[CurPage, 1];

  { Set button visibility and captions }
  V := not(CurPage in [Low(TWizardPage), wpInfoAfter, wpFinished]);
  if (SetupHeader.InfoAfterText <> '') and (CurPage = wpFinished) then
    V := True;
  BackButton.Visible := V;
  CancelButton.Visible := CurPage < wpInfoAfter;
  BackButton.Caption := SetupMessages[msgButtonBack];
  case CurPage of
    wpLicense: begin
        NextButton.Caption := SetupMessages[msgButtonYes];
        CancelButton.Caption := SetupMessages[msgButtonNo];
      end;
    wpReady: begin
        NextButton.Caption := SetupMessages[msgButtonInstall];
        CancelButton.Caption := SetupMessages[msgButtonCancel];
      end;
    wpFinished: begin
        NextButton.Caption := SetupMessages[msgButtonFinish];
        CancelButton.Caption := SetupMessages[msgButtonCancel];
      end;
  else
  1:NextButton.Caption := SetupMessages[msgButtonNext];
    CancelButton.Caption := SetupMessages[msgButtonCancel];
  end;

  { Set the Caption to match the current page's title }
  Caption := SetupMessages[PageCaptions[CurPage]];

  { Hide ClickLabel if it's on the Welcome, Ready to Install, or Setup
    Completed wizard pages }
  ClickLabel.Visible := not(CurPage in [wpWelcome, wpReady, wpFinished]);

  { Don't make the Next button "default" (i.e. selectable with the Enter key)
    when on the License page }
  NextButton.Default := CurPage <> wpLicense;
  { Adjust focus }
  case CurPage of
    wpWelcome: ActiveControl := NextButton;
    wpPassword: ActiveControl := PasswordEdit;
    wpLicense: ActiveControl := LicenseMemo;
    wpInfoBefore: ActiveControl := InfoBeforeMemo;
    wpSelectDir: ActiveControl := DirEdit;
    wpSelectProgramGroup: begin
        if not NoIconsCheck.Checked then
          ActiveControl := GroupEdit
        else
          ActiveControl := NoIconsCheck;
      end;
    wpReady: ActiveControl := NextButton;
    wpInfoAfter: ActiveControl := InfoAfterMemo;
    wpFinished:
      if ShowReadmeCheck.Visible then
        ActiveControl := ShowReadmeCheck
      else
      if YesRadio.Visible and YesRadio.Checked then
        ActiveControl := YesRadio
      else
      if NoRadio.Visible and NoRadio.Checked then
        ActiveControl := NoRadio
      else
        ActiveControl := NextButton;
  end;
end;

function TWizardForm.SkipCurPage: Boolean;
begin
  Result :=
    ((CurPage = wpPassword) and not NeedPassword) or
    ((CurPage = wpLicense) and (SetupHeader.LicenseText = '')) or
    ((CurPage = wpInfoBefore) and (SetupHeader.InfoBeforeText = '')) or
    ((CurPage = wpSelectDir) and ((shDisableDirPage in SetupHeader.Options) or not(shCreateAppDir in SetupHeader.Options))) or
    ((CurPage = wpSelectProgramGroup) and ((shDisableProgramGroupPage in SetupHeader.Options) or not HasIcons));
end;

procedure TWizardForm.NextButtonClick(Sender: TObject);

  function CheckPassword: Boolean;
  var
    S: String;
    SaveCursor: HCURSOR;
  begin
    S := PasswordEdit.Text;
    Result := GetCRC32(S[1], Length(S)) = SetupHeader.Password;
    if Result then
      NeedPassword := False
    else begin
      { Delay for 750 ms when an incorrect password is entered to
        discourage brute-force attempts }
      SaveCursor := GetCursor;
      SetCursor (LoadCursor(0, IDC_WAIT));
      Sleep (750);
      SetCursor (SaveCursor);
      MsgBox (SetupMessages[msgIncorrectPassword], '', mbError, MB_OK);
      PasswordEdit.Text := '';
      PasswordEdit.SetFocus;
    end;
  end;

  function GetTotalFreeSpace (const Drive: Char; var TotalSpace: Longint): Boolean;
  var
    SectorsPerCluster, BytesPerSector, FreeClusters, TotalClusters, Temp: Cardinal;
  begin
    Result := GetDiskFreeSpace(PChar(Drive + ':\'), DWORD(SectorsPerCluster),
      DWORD(BytesPerSector), DWORD(FreeClusters), DWORD(TotalClusters));
    if Result then begin
      Temp := BytesPerSector * SectorsPerCluster;
      { Windows 95/98 cap the result of GetDiskFreeSpace at 2GB, but NT 4.0
        does not, so we must use a 64-bit multiply operation to avoid an
        overflow. Delphi versions < 4 don't natively support 64-bit multiplies,
        so use assembler code to do this. }
      asm
        mov eax, Temp
        mul FreeClusters     { Multiplies EAX by FreeClusters, places 64-bit result in EDX:EAX }
        test edx, edx        { EDX will be zero if result is < 4 GB }
        jz @@1
        mov eax, 0FFFFFFFFh  { If > 4 GB then cap at 4 GB }
        @@1:
        mov Temp, eax
      end;
      TotalSpace := Temp;
    end;
  end;

  function SpaceString (const S: String): String;
  var
    I: Integer;
  begin
    Result := '';
    for I := 1 to Length(S) do begin
      if S[I] = ' ' then Continue;
      if Result <> '' then Result := Result + ' ';
      Result := Result + S[I];
    end;
  end;

  function CheckSelectDirPage: Boolean;
  const
    BadDirChars3 = '/:*?"<>|,()[]';
    BadDirChars4 = '/:*?"<>|';
  var
    T: String;
    I: Integer;
    TotalFreeSpace: Longint;
    BadDirChars: String;
  begin
    Result := False;

    { Remove any leading or trailing spaces }
    DirEdit.Text := Trim(DirEdit.Text);

    T := DirEdit.Text;  { reduce calls to GetText }

    { Check for UNC pathname }
    if Pos('\\', T) <> 0 then begin
      MsgBox (SetupMessages[msgToUNCPathname], '', mbError, MB_OK);
      Exit;
    end;

    { Check if is at least 4 chars long and it has a colon and backslash }
    if not(shAllowRootDirectory in SetupHeader.Options) then
      I := 4
    else
      I := 3;
    if (Length(T) < I) or (T[2] <> ':') or (T[3] <> '\') then begin
      MsgBox (SetupMessages[msgInvalidPath], '', mbError, MB_OK);
      Exit;
    end;

    { Check for invalid characters after x:\ }
    if UsingWindows4 then
      BadDirChars := BadDirChars4
    else
      BadDirChars := BadDirChars3;
    for I := 4 to Length(T) do
      if Pos(T[I], BadDirChars) <> 0 then begin
        MsgBox (FmtSetupMessage1(msgBadDirName32, SpaceString(BadDirChars)), '',
          mbError, MB_OK);
        Exit;
      end;

    { Check if it's a valid drive }
    if not(UpCase(T[1]) in ['A'..'Z']) or not GetTotalFreeSpace(T[1], TotalFreeSpace) then begin
      MsgBox (SetupMessages[msgInvalidDrive], '', mbError, MB_OK);
      Exit;
    end;

    { Remove trailing backslash }
    T := RemoveBackslashUnlessRoot(T);
    DirEdit.Text := T;

    { Check if enough disk space (using an unsigned long compare) }
    if Cardinal(TotalFreeSpace) < Cardinal(MinimumSpace) then
      { If not, show warning }
      if MsgBox(FmtSetupMessage(msgDiskSpaceWarning,
           [IntToStr(MinimumSpace div 1024), IntToStr(TotalFreeSpace div 1024)]),
         SetupMessages[msgDiskSpaceWarningTitle],
         mbConfirmation, MB_YESNO or MB_DEFBUTTON2) <> ID_YES then
        Exit;

    { Check if directory already exists }
    if ((SetupHeader.DirExistsWarning = ddYes) or
        ((SetupHeader.DirExistsWarning = ddAuto) and (T <> PreviousAppDir))) and
       DirExists(T) then
      { If so, ask if user wants to install there anyway }
      if MsgBox(FmtSetupMessage1(msgDirExists, T), SetupMessages[msgDirExistsTitle],
        mbConfirmation, MB_YESNO) <> ID_YES then Exit;

    { Check if directory *doesn't* already exist }
    if (shEnableDirDoesntExistWarning in SetupHeader.Options) and
       not DirExists(T) then
      { If not, ask if user wants to install there anyway }
      if MsgBox(FmtSetupMessage1(msgDirDoesntExist, T), SetupMessages[msgDirDoesntExistTitle],
        mbConfirmation, MB_YESNO) <> ID_YES then Exit;

    Result := True;
  end;

  function CheckSelectProgramGroupPage: Boolean;
  const
    BadGroupCharsOld = '\/:*?"<>|,';
    BadGroupCharsNew = '/:*?"<>|,';
  var
    T, BadGroupChars: String;
    I: Integer;
  begin
    Result := False;

    if not NoIconsCheck.Checked then begin
      { Remove any leading or trailing spaces and out of place backslashes }
      T := Trim(GroupEdit.Text);
      I := Pos('\\', T);
      while I <> 0 do begin
        Delete (T, I, 1);
        I := Pos('\\', T);
      end;
      if (T <> '') and (T[1] = '\') then
        Delete (T, 1, 1);
      T := RemoveBackslash(T);
      GroupEdit.Text := T;

      if T = '' then begin
        MsgBox (SetupMessages[msgMustEnterGroupName], '', mbError, MB_OK);
        Exit;
      end;

      { Check for invalid characters. Quotes are not allowed because the
        PROGMAN DDE interface uses them to delimit command strings. And
        since Windows 95/NT 4.0 stores program groups as literal directory
        names, quotes and the rest of the characters in the BadGroupChars
        constant aren't allowed either. }
      if not UsingWindows4 then
        BadGroupChars := BadGroupCharsOld
      else
        BadGroupChars := BadGroupCharsNew;
      for I := 1 to Length(T) do
        if Pos(T[I], BadGroupChars) <> 0 then begin
          MsgBox (FmtSetupMessage1(msgBadGroupName, SpaceString(BadGroupChars)),
            '', mbError, MB_OK);
          Exit;
        end;
    end;

    Result := True;
  end;

begin
  case CurPage of
    wpPassword: if not CheckPassword then Exit;
    wpSelectDir: if not CheckSelectDirPage then Exit;
    wpSelectProgramGroup: if not CheckSelectProgramGroupPage then Exit;
  end;

  { Go to the next page, or close wizard if it was on the last page }
  if CurPage in [wpReady, wpFinished] then begin
    CantShow := True;
    DisableCloseQuery := True;
    Close;
    DisableCloseQuery := False;
    PostMessage (MainForm.Handle, WM_SETUPNEXTSTEP, 0, 0);
  end
  else begin
    Inc (CurPage);
    while SkipCurPage do
      Inc (CurPage);
    CurPageChanged;
  end;
end;

procedure TWizardForm.BackButtonClick(Sender: TObject);
begin
  if CurPage = Low(TWizardPage) then Exit;

  { Go to the previous page }
  Dec (CurPage);
  while SkipCurPage do
    Dec (CurPage);
  CurPageChanged;
end;

procedure TWizardForm.CancelButtonClick(Sender: TObject);
begin
  { Clicking Cancel will do the same thing as the Close button }
  Close;
end;

procedure TWizardForm.FormCloseQuery(Sender: TObject;
  var CanClose: Boolean);
begin
  if not DisableCloseQuery then begin
    DisableCloseQuery := True;
    try
      { Redirect an attempt to close this form to MainForm }
      MainForm.Close;
      { And only close if MainForm closed }
      CanClose := MainForm.Closing;
    finally
      DisableCloseQuery := False;
    end;
  end;
end;

procedure TWizardForm.ReadGroupList;
{ Fills in the list box of existing groups }
var
  PROGMAN, Item: TDDEString;
  Conv: TDDEConversation;
  DataHandle: TDDEData;
  DataPointer: PChar;
  DataSize: LongInt;
  H: THandle;
  FindData: TWin32FindData;
  S: String;
begin
  try
    if not UsingWindows4 then begin
      PROGMAN := DDE.CreateString('PROGMAN');
      Item := DDE.CreateString('GROUPS');
      Conv := DDE.BeginConnection(PROGMAN, PROGMAN);
      try
        DDE.RequestBegin (Conv, Item, CF_TEXT, DataHandle, Pointer(DataPointer), DataSize);
        try
          GroupList.Items.SetText (DataPointer);
        finally
          DDE.RequestEnd (DataHandle);
        end;
      finally
        DDE.EndConnection (Conv);
        DDE.FreeString (Item);
        DDE.FreeString (PROGMAN);
      end;
    end
    else begin
      H := FindFirstFile(PChar(ChangeDirConst('{group}\*')), FindData);
      if H <> INVALID_HANDLE_VALUE then begin
        repeat
          if FindData.dwFileAttributes and FILE_ATTRIBUTE_DIRECTORY <> 0 then begin
            S := FindData.cFileName;
            if (S <> '.') and (S <> '..') then
              GroupList.Items.Add (S);
          end;
        until not FindNextFile(H, FindData);
        WinProcs.FindClose (H);
      end;
    end;
  except
    { ignore any exceptions }
  end;
end;

procedure TWizardForm.DirListChange(Sender: TObject);
begin
  if not(shDisableAppendDir in SetupHeader.Options) then
    DirEdit.Text := AddBackslash(DirList.Directory) +
      ExtractFileName(ChangeDirConst(SetupHeader.DefaultDirName))
  else
    DirEdit.Text := DirList.Directory;
end;

procedure TWizardForm.GroupEditChange(Sender: TObject);
var
  T: String;
  I: Integer;
  SaveOnClick: TNotifyEvent;
label 1;
begin
  T := GroupEdit.Text;
  with GroupList do begin
    for I := 0 to Items.Count-1 do
      if AnsiCompareText(T, Items[I]) = 0 then
        goto 1;
    I := -1;
  1:if ItemIndex <> I then begin
      SaveOnClick := OnClick;
      { Temporarily disabled the OnChange event }
      OnClick := nil;
      try
        ItemIndex := I;
      finally
        OnClick := SaveOnClick;
      end;
    end;
  end;
end;

procedure TWizardForm.GroupListClick(Sender: TObject);
var
  SaveOnChange: TNotifyEvent;
begin
  with GroupList do
    if ItemIndex <> -1 then begin
      SaveOnChange := GroupEdit.OnChange;
      { Temporarily disabled the OnChange event }
      GroupEdit.OnChange := nil;
      try
        GroupEdit.Text := Items[ItemIndex];
      finally
        GroupEdit.OnChange := SaveOnChange;
      end;
    end;
end;

procedure TWizardForm.NoIconsCheckClick(Sender: TObject);
const
  ColorChange: array[Boolean] of TColor = (clBtnFace, clWindow);
begin
  with GroupEdit do begin
    Enabled := not NoIconsCheck.Checked;
    Color := ColorChange[Enabled];
  end;
  with GroupList do begin
    Enabled := not NoIconsCheck.Checked;
    Color := ColorChange[Enabled];
    if not Enabled then ItemIndex := -1;
  end;
end;

procedure TWizardForm.PicturePaintBoxPaint(Sender: TObject);
var
  R: TRect;
  X, Y: Integer;
begin
  with PicturePaintBox do begin
    R := ClientRect;
    { Draw the "sunken" edge, and exclude the edge from the clipping region }
    NewDrawEdge (Canvas, R, deThinSunken);
    InflateRect (R, -1, -1);
    IntersectClipRect (Canvas.Handle, R.Left, R.Top, R.Right, R.Bottom);
    { Draw the teal background }
    Canvas.Brush.Color := SetupHeader.WizardImageBackColor;
    Canvas.FillRect (R);
    { And finally draw the image }
    if Assigned(WizardImage) then begin
      X := R.Left + ((R.Right - R.Left) - WizardImage.Width) div 2;
      if X < 0 then X := 0;
      Y := R.Top + ((R.Bottom - R.Top) - WizardImage.Height) div 2;
      if Y < 0 then Y := 0;
      Canvas.Draw (X, Y, WizardImage);
    end;
  end;
end;

procedure TWizardForm.SetupIconPaint(Sender: TObject);
begin
  (Sender as TPaintBox).Canvas.Draw (0, 0, Application.Icon);
end;

procedure TWizardForm.WMSysCommand (var Message: TWMSysCommand);
begin
  if Message.CmdType and $FFF0 = SC_MINIMIZE then
    { A minimize button is shown on the wizard form when (shWindowVisible in
      SetupHeader.Options). When it is clicked we want to minimize the whole
      application. }
    Application.Minimize
  else
  if Message.CmdType = 9999 then
    MainForm.ShowAboutBox
  else
    inherited;
end;

end.
