program Uninst;

{
  Inno Setup
  Copyright (C) 1998-2000 Jordan Russell
  For conditions of distribution and use, see LICENSE.TXT.

  Uninstaller

  NOTE: This project should only be saved in Delphi 3 or later, otherwise the
  version info may be stripped. I still compile this project in Delphi 2,
  though.

  $Id: Uninst.dpr,v 1.3 2000/12/09 00:54:19 jr Exp $
}

uses
  WinProcs,
  WinTypes,
  SysUtils,
  Messages,
  CmnFunc2 in 'CmnFunc2.pas',
  Undo in 'Undo.pas',
  Msgs in 'Msgs.pas',
  MsgIDs in 'MsgIDs.pas',
  InstFunc in 'InstFunc.pas',
  Struct in 'Struct.pas',
  UninstSharedFileDlg in 'UninstSharedFileDlg.pas',
  UninstProgressDlg in 'UninstProgressDlg.pas',
  WinDlgs in 'WinDlgs.pas';

{$R *.RES}

type
  TExtUninstallLog = class(TUninstallLog)
  private
    FNoSharedFileDlgs: Boolean;
    FRemoveSharedFiles: Boolean;
  protected
    function ShouldRemoveSharedFile (const Filename: String): Boolean; override;
    procedure ShowException (E: Exception); override;
    procedure StatusUpdate (StartingCount, CurCount: Integer); override;
  end;

const
  SecondPhaseWndName = '_InnoSetupUninstallerWindow_';  { don't localize }
  SecondPhaseParamName = '/EUFT32A';
  WM_KillFirstPhase = WM_USER + 333;
  SelfReadMode = fmOpenRead or fmShareDenyWrite;
  RemovedMsgs: array[Boolean] of TSetupMessageID =
    (msgUninstalledMost, msgUninstalledAll);

var
  TempFile, UninstExeFile, UninstDataFile, UninstMsgFile: String;
  UninstLog: TExtUninstallLog = nil;
  Title: String;
  SecondPhase, RemovedAll: Boolean;
  Silent: Boolean;
  FirstPhaseWnd, SecondPhaseWnd: HWND;
  ProcessHandle: THandle;
  F: File;
  UninstallerMsgTail: TUninstallerMsgTail;
  OldWindowProc: Pointer;

procedure ShowExceptionMsg (const Msg: String);
var
  M: String;
begin
  M := Msg;
  if (M <> '') and (M[Length(M)] > '.') then M := M + '.';
  MessageBox (ProgressDlg, PChar(M), Pointer(SetupMessages[msgErrorTitle]),
    MB_OK or MB_ICONSTOP);
    { ^ use a Pointer cast instead of a PChar cast so that it will use "nil"
      if SetupMessages[msgErrorTitle] is empty due to the messages not being
      loaded yet. MessageBox displays 'Error' as the caption if the lpCaption
      parameter is nil. }
end;

function ProcessMsgs: Boolean; forward;

function TExtUninstallLog.ShouldRemoveSharedFile (const Filename: String): Boolean;
begin
  if Silent then
    Result := True
  else begin
    if not FNoSharedFileDlgs then
      { FNoSharedFileDlgs will be set to True if a "...to All" button is clicked }
      FRemoveSharedFiles := ExecuteRemoveSharedFileDlg(ProgressDlg, Filename,
        FNoSharedFileDlgs);
    Result := FRemoveSharedFiles;
  end;
end;

procedure TExtUninstallLog.ShowException (E: Exception);
begin
  ShowExceptionMsg (E.Message);
end;

procedure TExtUninstallLog.StatusUpdate (StartingCount, CurCount: Integer);
begin
  if ProgressDlg = 0 then begin
    CreateUninstProgressDlg (Title, UninstLog.AppName);
    SetForegroundWindow (ProgressDlg);
  end;
  { Only update the progress bar if it's at the beginning or end, or if
    125 msec has passed since the last update (so that updating the progress
    bar doesn't slow down the actual uninstallation process). }
  if ProgressDlgUpdateTimerExpired or (StartingCount = CurCount) or
     (CurCount = 0) then begin
    ProgressDlgUpdateTimerExpired := False;
    UpdateProgressDlg (StartingCount - CurCount, StartingCount);
  end;
  ProcessMsgs;
end;

function MessageBoxFmt1 (const ID: TSetupMessageID; const Arg1: String;
  const Title: String; const Flags: UINT): Integer;
begin
  Result := MessageBox(0, PChar(FmtSetupMessage1(ID, Arg1)), PChar(Title), Flags);
end;

procedure RaiseLastError (const Msg: TSetupMessageID);
var
  ErrorCode: DWORD;
begin
  ErrorCode := GetLastError;
  raise Exception.Create(FmtSetupMessage(msgLastErrorMessage,
    [SetupMessages[Msg], IntToStr(ErrorCode), SysErrorMessage(ErrorCode)]));
end;

function EnumThreadWindowsProc (Wnd: HWND; LParam: Longint): BOOL; stdcall;
begin
  Result := False;
  SetForegroundWindow (Wnd);
end;

function Exec (const Filename: String; const Parms: String): THandle;
var
  CmdLine: String;
  StartupInfo: TStartupInfo;
  ProcessInfo: TProcessInformation;
begin
  CmdLine := AddQuotes(Filename) + ' ' + Parms;

  FillChar (StartupInfo, SizeOf(StartupInfo), 0);
  StartupInfo.cb := SizeOf(StartupInfo);
  if not CreateProcess(nil, PChar(CmdLine), nil, nil, False, 0, nil, nil,
     StartupInfo, ProcessInfo) then
    RaiseLastError (msgLdrCannotExecTemp);
  CloseHandle (ProcessInfo.hThread);
  Result := ProcessInfo.hProcess;
end;

function GenerateUniqueExeName (Path: String; var Filename: String): Boolean;
{ Returns True if it overwrote an existing file. }
  function IntToBase32 (Number: Longint): String;
  const
    Table: array[0..31] of Char = '0123456789ABCDEFGHIJKLMNOPQRSTUV';
  var
    I: Integer;
  begin
    Result := '';
    for I := 0 to 4 do begin
      Insert (Table[Number and 31], Result, 1);
      Number := Number shr 5;
    end;
  end;
var
  Rand, RandOrig: Longint;
  F: THandle;
  Success: Boolean;
begin
  Path := AddBackslash(Path);
  RandOrig := $123456;
  Rand := RandOrig;
  Success := False;
  Result := False;
  repeat
    Inc (Rand);
    if Rand > $1FFFFFF then Rand := 0;
    if Rand = RandOrig then
      { practically impossible to go through 33 million possibilities,
        but check "just in case"... }
      raise Exception.Create(FmtSetupMessage1(msgErrorTooManyFilesInDir,
        RemoveBackslashUnlessRoot(Path)));
    { Generate a random name }
    Filename := Path + '_iu' + IntToBase32(Rand) + '.tmp';
    if DirExists(Filename) then Continue;
    Success := True;
    Result := FileExists(Filename);
    if Result then begin
      F := CreateFile(PChar(Filename), GENERIC_READ or GENERIC_WRITE, 0,
        nil, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, 0);
      Success := F <> INVALID_HANDLE_VALUE;
      if Success then CloseHandle (F);
    end;
  until Success;
end;

function ProcessMsgs: Boolean;
var
  Msg: TMsg;
begin
  Result := False;
  while PeekMessage(Msg, 0, 0, 0, PM_REMOVE) do begin
    if Msg.Message = WM_QUIT then begin
      Result := True;
      Break;
    end;
    TranslateMessage (Msg);
    DispatchMessage (Msg);
  end;
end;

function FirstPhaseWindowProc (Wnd: HWND; Msg: UINT; wParam: WPARAM;
  lParam: LPARAM): LRESULT; stdcall;
begin
  Result := 0;
  case Msg of
    WM_QUERYENDSESSION: ;  { Return zero to deny any shutdown requests }
    WM_KillFirstPhase: PostQuitMessage (0);
  else
    Result := CallWindowProc(OldWindowProc, Wnd, Msg, wParam, lParam);
  end;
end;

function SecondPhaseWindowProc (Wnd: HWND; Msg: UINT; wParam: WPARAM;
  lParam: LPARAM): LRESULT; stdcall;
begin
  if Msg = WM_QUERYENDSESSION then
    { Return zero to deny any shutdown requests }
    Result := 0
  else
    Result := CallWindowProc(OldWindowProc, Wnd, Msg, wParam, lParam);
end;

procedure DeleteUninstallDataFiles; far;
var
  ProcessID: DWORD;
  Process: THandle;
begin
  DeleteFile (UninstDataFile);
  if UninstMsgFile <> '' then
    DeleteFile (UninstMsgFile);
  GetWindowThreadProcessId (FirstPhaseWnd, @ProcessID);
  Process := OpenProcess(STANDARD_RIGHTS_REQUIRED or SYNCHRONIZE, False,
    ProcessID);
  SendMessage (FirstPhaseWnd, WM_KillFirstPhase, 0, 0);
  WaitForSingleObject (Process, INFINITE);
  CloseHandle (Process);
  Sleep (0);  { give first phase a chance to terminate completely }
  DelayDeleteFile (UninstExeFile, 12);
  { Control Panel will try to reclaim the focus when the first phase
    terminates. Take it back. }
  if ProgressDlg <> 0 then
    SetForegroundWindow (ProgressDlg);
end;

procedure ProcessCommandLine;
var
  C, I: Integer;
  S: String;
begin
  C := ParamCount;
  I := 1;
  while I <= C do begin
    S := ParamStr(I);
    if S = SecondPhaseParamName then begin
      SecondPhase := True;
      UninstExeFile := ParamStr(I+1);
      FirstPhaseWnd := StrToInt(ParamStr(I+2));
      Inc (I, 2);
    end
    else if CompareText(S, '/SILENT') = 0 then
      Silent := True;
    Inc (I);
  end;
end;

begin
  try
    SetCurrentDirectory (PChar(GetWinDir));

    UninstExeFile := ParamStr(0);
    ProcessCommandLine;

    { Open own EXE and keep it open so that other programs can't write to it.
      This was added for the 16-bit version of Inno Setup, and may not be
      necessary on Win32, though, since it always locks EXEs during execution. }
    AssignFile (F, ParamStr(0));
    FileMode := SelfReadMode;
    Reset (F, 1);
    try
      Seek (F, FileSize(F) - SizeOf(UninstallerMsgTail));
      BlockRead (F, UninstallerMsgTail, SizeOf(UninstallerMsgTail));
      if UninstallerMsgTail.ID <> UninstallerMsgTailID then begin
        { No valid UninstallerMsgTail record found at the end of the EXE;
          load messages from an external .msg file. }
        UninstMsgFile := ChangeFileExt(UninstExeFile, '.msg');
        LoadSetupMessages (UninstMsgFile, 0, fmOpenRead or fmShareDenyWrite);
      end
      else
        LoadSetupMessages (UninstExeFile, UninstallerMsgTail.Offset, SelfReadMode);
      UninstDataFile := ChangeFileExt(UninstExeFile, '.dat');

      { Search for previous uninstaller instances, and if one is found,
        activate its window. }
      SecondPhaseWnd := FindWindow(nil, SecondPhaseWndName);
      if SecondPhaseWnd <> 0 then begin
        EnumThreadWindows (GetWindowThreadProcessId(SecondPhaseWnd, nil),
          @EnumThreadWindowsProc, 0);
        Exit;
      end;

      if not SecondPhase then begin
        { Verify that uninstall data file exists }
        if not FileExists(UninstDataFile) then begin
          MessageBoxFmt1 (msgUninstallNotFound, UninstDataFile,
            SetupMessages[msgUninstallAppTitle], MB_ICONSTOP or MB_OK);
          Exit;
        end;

        { Copy self to TEMP directory with a name like _iu14D2N.tmp. The
          actual uninstallation process must be done from somewhere outside
          the application directory since EXE's can't delete themselves while
          they are running. }
        if not GenerateUniqueExeName(GetTempDir, TempFile) then
          try
            RestartReplace (TempFile, '');
          except
            { ignore exceptions }
          end;
        if not FileCopy(UninstExeFile, TempFile, False, SelfReadMode) then begin
          RaiseLastError (msgLdrCannotCreateTemp);
          Exit;
        end;

        { Create FirstPhaseWnd. FirstPhaseWnd waits for a WM_KillFirstPhase
          message from the second phase process, and terminates itself in
          response. The reason the first phase doesn't just terminate
          immediately is because the Control Panel Add/Remove applet refreshes
          its list as soon as the program terminates. So it waits until the
          uninstallation is complete before terminating. }
        FirstPhaseWnd := CreateWindowEx(0, 'STATIC', '', 0, 0, 0, 0, 0,
          HWND_DESKTOP, 0, HInstance, nil);  { do not localize }
        Longint(OldWindowProc) := SetWindowLong(FirstPhaseWnd, GWL_WNDPROC,
          Longint(@FirstPhaseWindowProc));
        try
          { Execute the copy of itself ("second phase") }
          ProcessHandle := Exec(TempFile, Format(SecondPhaseParamName + ' %s $%x %s',
            [AddQuotes(ParamStr(0)), FirstPhaseWnd, GetCmdTail]));

          { Wait till the second phase process unexpectedly dies or is ready
            for the first phase to terminate. }
          repeat until ProcessMsgs or (MsgWaitForMultipleObjects(1,
            ProcessHandle, False, INFINITE, QS_ALLINPUT) <> WAIT_OBJECT_0+1);
          CloseHandle (ProcessHandle);
        finally
          DestroyWindow (FirstPhaseWnd);
        end;
      end
      else begin
        { Create a window with the name SecondPhaseWndName. In order to prevent
          multiple instances, the uninstaller searches for this window at
          startup and exits if it finds it already exists. }
        SecondPhaseWnd := CreateWindowEx(0, 'STATIC', SecondPhaseWndName,
          0, 0, 0, 0, 0, HWND_DESKTOP, 0, HInstance, nil);  { do not localize }
        Longint(OldWindowProc) := SetWindowLong(SecondPhaseWnd, GWL_WNDPROC,
          Longint(@SecondPhaseWindowProc));
        try
          UninstLog := TExtUninstallLog.Create;
          UninstLog.Load (UninstDataFile);
          Title := FmtSetupMessage(msgUninstallAppFullTitle, UninstLog.AppName);

          if (ufAdminInstalled in UninstLog.Flags) and not IsAdminLoggedOn then begin
            MessageBox (0, PChar(SetupMessages[msgOnlyAdminCanUninstall]), PChar(Title),
              MB_OK or MB_ICONEXCLAMATION);
            Exit;
          end;

          if not Silent then begin
            if MessageBoxFmt1(msgConfirmUninstall, UninstLog.AppName, PChar(Title),
               MB_ICONQUESTION or MB_YESNO or MB_DEFBUTTON2) <> ID_YES then
              Exit;
          end;

          { Is the app running? }
          while UninstLog.CheckMutexes do
            { Yes, tell user to close it }
            if MessageBoxFmt1(msgUninstallAppRunningError, UninstLog.AppName, PChar(Title),
               MB_OKCANCEL or MB_ICONEXCLAMATION) <> IDOK then
              Exit;

          { Start the actual uninstall process }
          RemovedAll := UninstLog.PerformUninstall(True, DeleteUninstallDataFiles);

          { Wait until the timer on ProgressDlg destroys the window }
          if ProgressDlg <> 0 then begin
            SetForegroundWindow (ProgressDlg);
            while not ProgressDlgCloseTimerExpired do begin
              WaitMessage;
              ProcessMsgs;
            end;
            DestroyWindow (ProgressDlg);
            ProgressDlg := 0;
          end;

          if not Silent then
            MessageBoxFmt1 (RemovedMsgs[RemovedAll], UninstLog.AppName,
              Title, MB_ICONINFORMATION or MB_OK or MB_SETFOREGROUND);
        finally
          UninstLog.Free;
          DestroyWindow (SecondPhaseWnd);
        end;
      end;
    finally
      CloseFile (F);
    end;
  except
    on E: Exception do
      ShowExceptionMsg (E.Message);
  end;
end.
